//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.Drawing;

using ircam.jmax.fts;
using ircam.jmax.editors.patcher;
using ircam.jmax.editors.patcher.objects;

namespace ircam.jmax.editors.patcher.interactions
{
    /// <summary>Make a connection from an outlet to an inlet </summary>

    internal class DragConnectInteraction : Interaction
    {
        internal bool dragged = false;
        internal Point dragStart = new Point();
        internal GraphicObject src;
        internal int outlet;
        internal GraphicObject dst;
        internal int inlet;

        internal override void configureInputFilter(InteractionEngine filter)
        {
            filter.isFollowingMoves = true; // need the drag
            filter.isFollowingInOutletLocations = true;
            filter.isFollowingInletLocations = true;
            filter.isFollowingAlsoConnectionLocations = false;
            filter.isAutoScrolling = true;
        }

        // Utility method

        private void doConnection(ErmesSketchPad editor, GraphicObject src, int outlet, GraphicObject dst, int inlet)
        {
            if (src == dst)
            {
                editor.showMessage("Cannot connect an object to itself !");
                return;
            }

            editor.resetUndoRedo();

            editor.addingConnection();
            editor.FtsPatcher.RequestAddConnection(src.FtsObject, outlet, dst.FtsObject, inlet);
        }

        internal bool destinationChoosen = false;

        internal override void gotSqueack(ErmesSketchPad editor, int squeack, SensibilityArea area, Point mouse, Point oldMouse)
        {
            if (Squeack.isDown(squeack) && Squeack.onOutlet(squeack))
            {
                ErmesSelection.patcherSelection.redraw();
                ErmesSelection.patcherSelection.DeselectAll();

                src = (GraphicObject)area.Target;
                outlet = area.Number;

                dragStart.X = src.getOutletAnchorX(outlet);
                dragStart.Y = src.getOutletAnchorY(outlet);
                //editor.resetHighlightedOutlet();
                editor.ConnectingObject = src;
                dragged = false;
            }
            else if (Squeack.isUp(squeack))
            {
                if (dragged)
                {
                    // Dragged: 
                    //  do the connection if we have a destination
                    if (destinationChoosen)
                    {
                        editor.resetHighlightedInlet();
                        doConnection(editor, src, outlet, dst, inlet);
                    }

                    // clean up
                    if ((!Squeack.isShift(squeack)) || (!destinationChoosen))
                    {
                        editor.resetHighlightedOutlet(); //??

                        editor.DisplayList.NoDrag();
                        editor.DisplayList.RedrawDragLine();
                        editor.ConnectingObject = null;
                        destinationChoosen = false;

                        editor.endInteraction();
                    }
                    else if (Squeack.isShift(squeack))
                    {
                        //shift pressed: moveReverseInteraction setting to do multiconnect 
                        editor.Engine.AInteraction = Interactions.moveConnectInteraction;
                        ((MoveConnectInteraction)editor.Engine.AInteraction).setSrc(src, outlet);
                        editor.Engine.AInteraction.gotSqueack(editor, squeack, area, mouse, oldMouse);
                    }
                }
                else
                {
                    // Not dragged, start a move connection
                    editor.Engine.AInteraction = Interactions.moveConnectInteraction;
                    editor.Engine.AInteraction.gotSqueack(editor, squeack, area, mouse, oldMouse);
                }
            }
            else if (Squeack.isDrag(squeack) && Squeack.onInlet(squeack))
            {
                dragged = true;

                if ((!destinationChoosen) || dst != (GraphicObject)area.Target || inlet != area.Number)
                {
                    if (!editor.DisplayList.ThisConnectionExist(src, outlet, (GraphicObject)area.Target, area.Number))
                    {
                        dst = (GraphicObject)area.Target;
                        inlet = area.Number;
                        editor.setHighlightedInlet(dst, inlet);
                        destinationChoosen = true;
                        editor.DisplayList.DragLine();
                        editor.DisplayList.RedrawDragLine();

                        editor.DisplayList.SetDragLine(dragStart.X, dragStart.Y, dst.getInletAnchorX(inlet), dst.getInletAnchorY(inlet));
                        editor.DisplayList.RedrawDragLine();
                    }
                    else
                    {
                        destinationChoosen = false;
                        editor.resetHighlightedInlet();
                        dst = null;

                        editor.DisplayList.DragLine();
                        editor.DisplayList.RedrawDragLine();
                        editor.DisplayList.SetDragLine(dragStart.X, dragStart.Y, mouse.X, mouse.Y);
                        editor.DisplayList.RedrawDragLine();
                    }
                }
            }
            else if (Squeack.isDrag(squeack))
            {
                dragged = true;
                if (destinationChoosen)
                {
                    editor.resetHighlightedInlet();
                    destinationChoosen = false;
                }

                editor.DisplayList.DragLine();
                editor.DisplayList.RedrawDragLine();
                editor.DisplayList.SetDragLine(dragStart.X, dragStart.Y, mouse.X, mouse.Y);
                editor.DisplayList.RedrawDragLine();
            }
        }
    }
}