//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.Drawing;

using ircam.jmax;
using ircam.jmax.fts;
using ircam.fts.client;
using ircam.jmax.dialogs;
using ircam.jmax.editors.patcher;
using ircam.jmax.editors.patcher.interactions;
using ircam.jmax.toolkit;

namespace ircam.jmax.editors.patcher.objects
{
    // The base class of all the graphic objects on the sketch.
    // This class has a knowledge of its corrisponding fos object. It contains
    // methods for:
    // - painting, selecting, moving, dragging (services to the sketch)
    // - create fos objects, change the value when the FTS value
    // has changed (services to FTS), redefine itself.
    // - sending values when the user interact with the object
    // - handle the object with data and open the associated editor
    // (example: subpatchers, table, etc.)

    // A survivor...

    [Serializable]
    abstract public class GraphicObject : IDisplayObject
    {
        public class ObjectGeometry
        {
            // All this parameters can be changed without changing the geometry
            // of the patch, i.e. the connection positions, unless stated otherwise

            public const int INOUTLET_WIDTH = 5;
            public const int INOUTLET_HEIGHT = 3;

            public const int HIGHLIGHTED_INOUTLET_HEIGHT = 6;
            public const int HIGHLIGHTED_INOUTLET_WIDTH = 7;

            // The part of the highlight that go inside the 
            // Rectangle 

            public const int HIGHLIGHTED_INLET_OVERLAP = 5;

            // The part of the highlight that go inside the 
            // Rectangle 

            public const int HIGHLIGHTED_OUTLET_OVERLAP = 5;

            // PAD is the distance between the object border and the 
            // center of the inlet/outlet; CHANGE the Connection geometry

            public const int INOUTLET_PAD = 4;

            // INLET_OVERLAP is the part of the inlet rectangle that
            // go inside the object

            public const int INLET_OVERLAP = 2;

            // INLET_OFFSET is the vertical distance between the anchor point
            // and the object; CHANGE the Connection geometry

            public const int INLET_OFFSET = 1;

            // OUTLET_OVERLAP is the part of the inlet rectangle that
            // go inside the object

            public const int OUTLET_OVERLAP = 2;

            // OUTLET_OFFSET is the vertical distance between the anchor point
            // and the object; CHANGE the Connection geometry

            public const int OUTLET_OFFSET = 0;

            // Sensibility areas 

            // Resize area 

            public const int H_RESIZE_SENSIBLE_WIDTH = 4;
            public const int V_RESIZE_SENSIBLE_HEIGHT = 5;

            // Inlet and outlets Max

            public const int INOUTLET_MAX_SENSIBLE_AREA = 10;

            //dimension of in/outlet sensible area during connection

            public const int INOUTLET_CONNECTED_SENSIBLE_AREA = 20;
        }

        // This two flags say if the object parts are sensible, and if the in/outlets are sensibles
        // global for all the objects.

        private static bool followingLocations = false;
        private static bool followingInOutletLocations = false;
        private static bool followingInletLocations = false;
        private static bool followingOutletLocations = false;

        internal const int ERROR_MESSAGE_DISPLAY_PAD = ObjectGeometry.INOUTLET_PAD + ObjectGeometry.INOUTLET_WIDTH + 1;
        [NonSerialized()]
        private static Font errorFont = new Font((string)null, 4, System.Drawing.FontStyle.Bold);

        public static bool isFollowingInOutletLocations
        {
            set
            {
                followingInOutletLocations = value;
            }
        }

        public static bool isFollowingInletLocations
        {
            set
            {
                followingInletLocations = value;
            }
        }

        public static bool isFollowingOutletLocations
        {
            set
            {
                followingOutletLocations = value;
            }
        }

        public static bool isFollowingLocations
        {
            set
            {
                followingLocations = value;
            }
        }

        [NonSerialized()]
        protected internal ErmesSketchPad itsSketchPad;

        [NonSerialized()]
        protected internal FtsGraphicObject ftsObject = null;

        [NonSerialized()]
        private bool selected = false;

        [NonSerialized()]
        private int inletDistance; // the distance between two inlets anchor point
        [NonSerialized()]
        private int outletDistance; // the distance between two outlets anchor point

        public int InletDistance
        {
            get
            {
                return inletDistance;
            }
            set
            {
                inletDistance = value;
            }
        }

        public int OutletDistance
        {
            get
            {
                return outletDistance;
            }
            set
            {
                outletDistance = value;
            }
        }

        protected internal Font itsFont = null;
        protected internal FontFamily itsFontMetrics = null;

        public const int ON_INLET = 0;
        public const int ON_OUTLET = 1;
        public const int ON_OBJECT = 2;

        protected internal GraphicObject(FtsGraphicObject theFtsObject)
        {
            string fontName = null;
            float fontSize;
            int fontStyle;

            ftsObject = theFtsObject;
            itsSketchPad = ((ErmesSketchWindow)((FtsPatcherObject)ftsObject.Parent).EditorForm).SketchPad;

            ftsObject.ObjectListener = this;

            selected = false;

            fontName = ftsObject.Font;
            fontSize = ftsObject.FontSize;
            fontStyle = ftsObject.FontStyle;

            if ((object)fontName == null)
                fontName = itsSketchPad.DefaultFontName;

            if (fontSize < 0)
                fontSize = itsSketchPad.DefaultFontSize;
            if (fontStyle < 0)
                fontStyle = itsSketchPad.DefaultFontStyle;

            //itsFont = FontCache.lookupFont(fontName, fontSize, fontStyle);
            //itsFontMetrics = FontCache.lookupFontMetrics(fontName, fontSize, fontStyle);

            updateInOutlets();
        }

        // Destructor 

        public void Delete()
        {
            itsSketchPad.DisplayList.ReleaseConnectionsForObject(this);
            itsSketchPad.DisplayList.Remove(this);
            itsSketchPad.FtsPatcher.RemoveObject(ftsObject);
            if (inspector != null)
            {
                inspector.Visible = false;
                //UNDONE: 'ircam.jmax.editors.patcher.objects.ObjectInspector' does not contain a definition for 'Dispose'
                //                inspector.Dispose();
                inspector = null;
            }
            Dispose();
        }

        public int X
        {
            get
            {
                return ScaleTransform.Instance.ScaleX(ftsObject.X);
            }
            set
            {
                ftsObject.X = ScaleTransform.Instance.InvScaleX(value);
                itsSketchPad.DisplayList.UpdateConnectionsFor(this);
            }
        }

        public int Y
        {
            get
            {
                return ScaleTransform.Instance.ScaleY(ftsObject.Y);
            }
            set
            {
                ftsObject.Y = ScaleTransform.Instance.InvScaleY(value);
                itsSketchPad.DisplayList.UpdateConnectionsFor(this);
            }

        }

        public int Width
        {
            get
            {
                return ScaleTransform.Instance.ScaleX(ftsObject.Width + VariableWidth);
            }
            set
            {
                if (value > 0)
                {
                    ftsObject.Width = ScaleTransform.Instance.InvScaleX(value - VariableWidth);
                    updateInOutlets();
                    itsSketchPad.DisplayList.UpdateConnectionsFor(this);
                }
            }
        }

        public int WidthShift
        {
            set
            {
                Width = value;
            }
        }

        /* redefined only in Standard object */
        public virtual int VariableWidth
        {
            get
            {
                return 0;
            }
        }

        public int Height
        {
            get
            {
                if (isSquare)
                    return ScaleTransform.Instance.ScaleX(ftsObject.Height);
                else
                    return ScaleTransform.Instance.ScaleY(ftsObject.Height);
            }
            set
            {
                if (value > 0)
                {
                    if (isSquare)
                        ftsObject.Height = ScaleTransform.Instance.InvScaleX(value);
                    else
                        ftsObject.Height = ScaleTransform.Instance.InvScaleY(value);
                    itsSketchPad.DisplayList.UpdateConnectionsFor(this);
                }
            }
        }

        public void setCurrentBounds(int x, int y, int w, int h)
        {
            if ((w <= 0) || (h <= 0))
            {
                ftsObject.CurrentX = ScaleTransform.Instance.InvScaleX(x);
                ftsObject.CurrentY = ScaleTransform.Instance.InvScaleY(y);
                setDefaults();
            }
            else
                ftsObject.SetCurrentBounds(ScaleTransform.Instance.InvScaleX(x), ScaleTransform.Instance.InvScaleY(y), ScaleTransform.Instance.InvScaleX(w - VariableWidth), isSquare ? ScaleTransform.Instance.InvScaleX(h) : ScaleTransform.Instance.InvScaleY(h));
        }

        public virtual string CurrentName
        {
            set { }
        }

        public virtual void setDefaults() { }

        public bool isSquare
        {
            get
            {
                return false;
            }
        }

        public void scale(float scaleX, float scaleY)
        {
            X = (int)System.Math.Round((double)(X * scaleX));
            Y = (int)System.Math.Round((double)(Y * scaleY));
            Width = (int)System.Math.Round((double)(Width * scaleX));
            Height = (int)System.Math.Round((double)(Height * scaleY));
        }

        // Special version that do not update the connections.
        protected internal int HeightNoConnections
        {
            set
            {
                if (value > 0)
                {
                    Height = value;
                }
            }
        }

        public virtual Font AFont
        {
            get
            {
                return itsFont;
            }
            set
            {
                itsFont = value;
                itsFontMetrics = new FontFamily(itsFont.Name);

                ftsObject.Font = itsFont.Name;
                ftsObject.FontSize = itsFont.Size;
                ftsObject.FontStyle = (int)itsFont.Style;
            }
        }

        public string FontName
        {
            get
            {
                return itsFont.Name;
            }

            set
            {
                AFont = FontCache.lookupFont(value, (int)itsFont.Size, (int)itsFont.Style);
            }
        }

        public int FontSize
        {
            get
            {
                return (int)itsFont.Size;
            }

            set
            {
                AFont = FontCache.lookupFont(itsFont.Name, value, (int)itsFont.Style);
            }
        }

        public int FontStyle
        {
            get
            {
                return (int)itsFont.Style;
            }
            set
            {
                AFont = FontCache.lookupFont(itsFont.Name, (int)itsFont.Size, value);
            }
        }

        public FontFamily FontMetrics
        {
            get
            {
                return itsFontMetrics;
            }
        }

        public void changeFontStyle(string style, bool selected)
        {
            int fstyle = -1;

            if (style.Equals("Bold"))
            {
                if (itsFont.Bold && !selected)
                    if (itsFont.Italic)
                        fstyle = (int)System.Drawing.FontStyle.Italic;
                    else
                    {
                        fstyle = (int)System.Drawing.FontStyle.Regular;
                    }
                else if (!itsFont.Bold && selected)
                    if (!itsFont.Italic)
                        fstyle = (int)System.Drawing.FontStyle.Bold;
                    else
                        fstyle = (int)System.Drawing.FontStyle.Bold + (int)System.Drawing.FontStyle.Italic;
            }
            else if (style.Equals("Italic"))
            {
                if (itsFont.Italic && !selected)
                    if (itsFont.Bold)
                        fstyle = (int)System.Drawing.FontStyle.Bold;
                    else
                    {
                        fstyle = (int)System.Drawing.FontStyle.Regular;
                    }
                else if (!itsFont.Italic && selected)
                    if (!itsFont.Bold)
                        fstyle = (int)System.Drawing.FontStyle.Italic;
                    else
                        fstyle = (int)System.Drawing.FontStyle.Bold + (int)System.Drawing.FontStyle.Italic;
            }

            if (fstyle != -1)
                AFont = FontCache.lookupFont(itsFont.Name, (int)itsFont.Size, fstyle);
        }

        public void fontSmaller()
        {
            int size;

            size = (int)itsFont.Size;

            if (size > 8)
                AFont = FontCache.lookupFont(itsFont.Name, size - 2, (int)itsFont.Style);
        }

        public void fontBigger()
        {
            AFont = FontCache.lookupFont(itsFont.Name, (int)itsFont.Size + 2, (int)itsFont.Style);
        }

        public virtual void SetCurrentFont(string fontName, float fontSize, int fontStyle)
        {
            if ((object)fontName == null)
                fontName = itsSketchPad.DefaultFontName;
            if (fontSize <= 0)
                fontSize = itsSketchPad.DefaultFontSize;
            if ((fontStyle != (int)System.Drawing.FontStyle.Regular) && (fontStyle != (int)System.Drawing.FontStyle.Italic) && (fontStyle != (int)System.Drawing.FontStyle.Bold))
                fontStyle = itsSketchPad.DefaultFontStyle;

            setCurrentFont(FontCache.lookupFont(fontName, fontSize, fontStyle));
        }

        public virtual void setCurrentFont(Font font)
        {
            itsFont = font;
            itsFontMetrics = new FontFamily(itsFont.Name);

            ftsObject.CurrentFontName = itsFont.Name;
            ftsObject.CurrentFontSize = itsFont.Size;
            ftsObject.CurrentFontStyle = (int)itsFont.Style;
        }

        public virtual void fitToText() { }

        public int getOutletAnchorX(int outlet)
        {
            return X + ObjectGeometry.INOUTLET_PAD + outlet * outletDistance;
        }

        public int getOutletAnchorY(int outlet)
        {
            return Y + Height;
        }

        public int getInletAnchorX(int inlet)
        {
            return X + ObjectGeometry.INOUTLET_PAD + inlet * inletDistance;
        }

        public int getInletAnchorY(int inlet)
        {
            return Y - 1;
        }

        public void paintInlets(Graphics g)
        {
            int n = ftsObject.NumberOfInlets;

            for (int i = 0; i < n; i++)
            {
                int x, y;

                x = getInletAnchorX(i);
                y = getInletAnchorY(i);

                if (itsSketchPad.isHighlightedInlet(this, i))
                {
                    int w = ObjectGeometry.HIGHLIGHTED_INOUTLET_WIDTH;
                    int h = ObjectGeometry.HIGHLIGHTED_INOUTLET_HEIGHT;
                    int xHL = x - w / 2;
                    int yHL = y + ObjectGeometry.HIGHLIGHTED_INLET_OVERLAP + ObjectGeometry.INLET_OFFSET;

                    g.FillRectangle(new SolidBrush(Color.Blue), xHL, yHL - h, w, h - 3);
                    g.DrawLine(new Pen(Color.Black), x - 2, yHL - 3, x + 2, yHL - 3);
                    g.DrawLine(new Pen(Color.Black), x - 1, yHL - 2, x + 1, yHL - 2);
                    g.DrawLine(new Pen(Color.Black), x - 0, yHL - 1, x + 0, yHL - 1);
                }
                else
                {
                    int w = ObjectGeometry.INOUTLET_WIDTH;
                    int h = ObjectGeometry.INOUTLET_HEIGHT;

                    g.FillRectangle(new SolidBrush(Color.Blue), x - w / 2, y - h + ObjectGeometry.INLET_OVERLAP + ObjectGeometry.INLET_OFFSET, w, h);
                }
            }
        }

        public void paintOutlets(Graphics g)
        {
            int n = ftsObject.NumberOfOutlets;

            for (int i = 0; i < n; i++)
            {
                int x, y;

                x = getOutletAnchorX(i);
                y = getOutletAnchorY(i);

                if (itsSketchPad.isHighlightedOutlet(this, i))
                {
                    int w = ObjectGeometry.HIGHLIGHTED_INOUTLET_WIDTH;
                    int h = ObjectGeometry.HIGHLIGHTED_INOUTLET_HEIGHT;
                    int xHL = x - w / 2;
                    int yHL = y - ObjectGeometry.HIGHLIGHTED_OUTLET_OVERLAP - ObjectGeometry.OUTLET_OFFSET;

                    g.DrawLine(new Pen(Color.Black), x - 0, yHL + 0, x + 0, yHL + 0);
                    g.DrawLine(new Pen(Color.Black), x - 1, yHL + 1, x + 1, yHL + 1);
                    g.DrawLine(new Pen(Color.Black), x - 2, yHL + 2, x + 2, yHL + 2);
                    g.FillRectangle(new SolidBrush(Color.Blue), xHL, yHL + 3, w, h - 3);
                }
                else
                {
                    int w = ObjectGeometry.INOUTLET_WIDTH;
                    int h = ObjectGeometry.INOUTLET_HEIGHT;

                    g.FillRectangle(new SolidBrush(Color.Blue), x - w / 2, y - ObjectGeometry.OUTLET_OVERLAP - ObjectGeometry.OUTLET_OFFSET, w, h);
                }
            }
        }

        public virtual void Paint(Graphics g)
        {
            Color color;

            if (ftsObject.IsError)
                color = Color.Gray;
            else
                color = Color.Black;

            g.DrawRectangle(new Pen(color), X, Y, Width - 1, Height - 1);

            paintInlets(g);
            paintOutlets(g);
        }

        public virtual void UpdatePaint(Graphics g) { }


        // Add one to cope with the inlet/outlets

        public void Redraw()
        {
            //HACK: 'javax.swing.JComponent.repaint()' was not converted.
            //            itsSketchPad.repaint(X,
            //                Y - ObjectGeometry.HIGHLIGHTED_INOUTLET_HEIGHT +
            //                ObjectGeometry.INLET_OFFSET + ObjectGeometry.INLET_OVERLAP - 1,
            //                getWidth(),
            //                getHeight() + 2 * ObjectGeometry.HIGHLIGHTED_INOUTLET_HEIGHT  -
            //                ObjectGeometry.INLET_OFFSET - ObjectGeometry.INLET_OVERLAP -
            //                ObjectGeometry.OUTLET_OFFSET - ObjectGeometry.OUTLET_OVERLAP + 2);

        }

        public virtual void updateRedraw()
        {
            itsSketchPad.paintAtUpdateEnd(this, X, Y - ObjectGeometry.HIGHLIGHTED_INOUTLET_HEIGHT + ObjectGeometry.INLET_OFFSET + ObjectGeometry.INLET_OVERLAP, Width, Height + 2 * ObjectGeometry.HIGHLIGHTED_INOUTLET_HEIGHT - ObjectGeometry.INLET_OFFSET - ObjectGeometry.INLET_OVERLAP - ObjectGeometry.OUTLET_OFFSET - ObjectGeometry.OUTLET_OVERLAP);
        }

        public void redrawConnections()
        {
            itsSketchPad.DisplayList.RedrawConnectionsFor(this);
        }

        public void updateInOutlets()
        {
            if (ftsObject.NumberOfInlets > 1)
                inletDistance = (Width - VariableWidth - 2 * ObjectGeometry.INOUTLET_PAD) / (ftsObject.NumberOfInlets - 1);

            if (ftsObject.NumberOfOutlets > 1)
                outletDistance = (Width - VariableWidth - 2 * ObjectGeometry.INOUTLET_PAD) / (ftsObject.NumberOfOutlets - 1);
        }

        // redefine provide a default empty implementation
        // for the object that do not redefine themselves

        public virtual void redefine(string text)
        {
            updateInOutlets();
        }

        public virtual void redefined() { }

        public bool isSelected
        {
            get
            {
                return selected;
            }
            set
            {
                selected = value;
            }
        }

        public FtsGraphicObject FtsObject
        {
            get
            {
                return ftsObject;
            }
        }

        public ErmesSketchPad SketchPad
        {
            get
            {
                return itsSketchPad;
            }
        }

        // Utility functions for GetSensibilityAreaAt

        private SensibilityArea makeInletArea(int mouseX, int mouseY, int n, int xcost)
        {
            int inletsAnchorY = Y - 1;
            SensibilityArea area = SensibilityArea.getSensibilityArea(this, Squeack.INLET);

            if ((mouseY < inletsAnchorY) || (mouseX < X) || (mouseX > X + Width) || (mouseY > inletsAnchorY + Height))
            {
                area.isTransparent = true;
                area.Cost = xcost + Math.Abs(mouseY - inletsAnchorY);
            }

            area.Number = n;

            return area;
        }

        private SensibilityArea makeOutletArea(int mouseX, int mouseY, int n, int xcost)
        {
            int outletsAnchorY = Y + Height;
            SensibilityArea area = SensibilityArea.getSensibilityArea(this, Squeack.OUTLET);

            if ((mouseY > outletsAnchorY) || (mouseX < X) || (mouseY < Y - 1) || (mouseX > X + Width))
            {
                area.isTransparent = true;
                area.Cost = xcost + Math.Abs(mouseY - outletsAnchorY);
            }

            area.Number = n;
            return area;
        }

        public SensibilityArea GetSensibilityAreaAt(int mouseX, int mouseY)
        {
            if (itsSketchPad.DisplayList.IsDragLine)
                return getDrawingLineSensibilityAreaAt(mouseX, mouseY);
            else
                return getMouseMovingSensibilityAreaAt(mouseX, mouseY);
        }


        internal SensibilityArea getMouseMovingSensibilityAreaAt(int mouseX, int mouseY)
        {
            SensibilityArea area = null;
            int x = X;
            int y = Y;
            int w = Width;
            int h = Height;
            int verticalInOutletSensibility;
            int horizontalInletSensibility;
            int horizontalOutletSensibility;
            int inletsAnchorY = Y - 1;
            int outletsAnchorY = Y + Height;
            int nInlets = ftsObject.NumberOfInlets;
            int nOutlets = ftsObject.NumberOfOutlets;

            if (nInlets == 1)
                horizontalInletSensibility = System.Math.Min(ObjectGeometry.INOUTLET_MAX_SENSIBLE_AREA, w / 2);
            else
                horizontalInletSensibility = System.Math.Min(ObjectGeometry.INOUTLET_MAX_SENSIBLE_AREA, inletDistance / 2);

            if (nOutlets == 1)
                horizontalOutletSensibility = System.Math.Min(ObjectGeometry.INOUTLET_MAX_SENSIBLE_AREA, w / 2);
            else
                horizontalOutletSensibility = System.Math.Min(ObjectGeometry.INOUTLET_MAX_SENSIBLE_AREA, outletDistance / 2);

            verticalInOutletSensibility = System.Math.Min(ObjectGeometry.INOUTLET_MAX_SENSIBLE_AREA, h / 3);

            if (followingInOutletLocations)
            {
                // if the point is the vertical inlet zone,
                // check if the point in an inlet sensibility area

                if ((nInlets > 0) && (mouseY >= inletsAnchorY - verticalInOutletSensibility) && (mouseY <= inletsAnchorY + verticalInOutletSensibility))
                {
                    int start = getInletAnchorX(0);
                    int d;

                    if ((nInlets == 1) || (mouseX < start))
                    {
                        d = System.Math.Abs(mouseX - start);
                        if (d < horizontalInletSensibility)
                            return makeInletArea(mouseX, mouseY, 0, d);
                    }
                    else if (nInlets > 1)
                    {
                        int n = (mouseX - start) / inletDistance;
                        d = (mouseX - start) % inletDistance;
                        if ((d < horizontalInletSensibility) && (n >= 0) && (n < nInlets))
                            return makeInletArea(mouseX, mouseY, n, d);
                        else if ((d > inletDistance - horizontalInletSensibility) && (n >= 0) && (n < (nInlets - 1)))
                            return makeInletArea(mouseX, mouseY, n + 1, inletDistance - d);
                    }
                }

                // if we have outlets, and the point is the vertical outlet zone,
                // check if the point in an outlet sensibility area

                if ((nOutlets > 0) && (mouseY >= outletsAnchorY - verticalInOutletSensibility) && (mouseY <= outletsAnchorY + verticalInOutletSensibility))
                {
                    int start = getOutletAnchorX(0);
                    int d = 0, n = 0;

                    if (nOutlets == 1)
                    {
                        n = 0;
                        d = Math.Abs(mouseX - start);
                    }
                    else if (nOutlets > 1)
                    {
                        n = (mouseX - start) / outletDistance;
                        d = Math.Abs((mouseX - start) % outletDistance);
                    }

                    if (n == 0)
                    {
                        if (d < horizontalOutletSensibility)
                            return makeOutletArea(mouseX, mouseY, 0, d);
                    }
                    else if (n > 0)
                    {
                        if ((d < horizontalOutletSensibility) && (n >= 0) && (n < nOutlets))
                            return makeOutletArea(mouseX, mouseY, n, d);
                        else if ((d > outletDistance - horizontalOutletSensibility) && (n >= 0) && (n < (nOutlets - 1)))
                            return makeOutletArea(mouseX, mouseY, n + 1, outletDistance - d);
                    }
                }
            }

            if (followingLocations)
            {
                // Every other sensibility area is internal, so we check
                // now if the point is inside the rectangle, and return null if outside

                if ((mouseX < x) || (mouseX > (x + w)) || (mouseY < y) || (mouseY > (y + h)))
                    return null;

                // Check for horizantal resize area (assuming a point inside the rectangle)


                if ((isResizable) && (mouseX >= x + w - ObjectGeometry.H_RESIZE_SENSIBLE_WIDTH && mouseY > y + ObjectGeometry.H_RESIZE_SENSIBLE_WIDTH && mouseY < y + h - ObjectGeometry.H_RESIZE_SENSIBLE_WIDTH))
                {
                    return SensibilityArea.getSensibilityArea(this, Squeack.HRESIZE_HANDLE);
                }

                // Try subclass specialized methods

                area = findSensibilityArea(mouseX, mouseY);

                if (area == null)
                    area = SensibilityArea.getSensibilityArea(this, Squeack.OBJECT);
            }
            return area;
        }

        internal SensibilityArea getDrawingLineSensibilityAreaAt(int mouseX, int mouseY)
        {
            SensibilityArea area = null;

            if (followingInletLocations)
                area = getDrawingLineInletSensibilityArea(mouseX, mouseY);
            else if (followingOutletLocations)
                area = getDrawingLineOutletSensibilityArea(mouseX, mouseY);

            return area;
        }

        internal SensibilityArea getDrawingLineInletSensibilityArea(int mouseX, int mouseY)
        {
            SensibilityArea area = null;
            int verticalSensibility;
            int horizontalSensibility;
            int anchorY = Y - 1;
            int nInlets = ftsObject.NumberOfInlets;

            if (itsSketchPad.ConnectingObject != this)
            {
                int d;
                int start = getInletAnchorX(0);

                if (pointInObject(mouseX, mouseY))
                //mouse in object
                {
                    if (nInlets == 1)
                    {
                        d = Math.Abs(mouseX - start);
                        return makeInletArea(mouseX, mouseY, 0, d);
                    }
                    else if (nInlets > 0)
                    {
                        horizontalSensibility = inletDistance / 2;

                        int n = (mouseX - start) / inletDistance;
                        d = (mouseX - start) % inletDistance;

                        if ((d <= horizontalSensibility) && (n >= 0) && (n < nInlets))
                            return makeInletArea(mouseX, mouseY, n, d);
                        else if ((d >= inletDistance - horizontalSensibility) && (n >= 0) && (n < (nInlets - 1)))
                            return makeInletArea(mouseX, mouseY, n + 1, inletDistance - d);
                    }
                }
                //mouse out of object 
                else
                {
                    verticalSensibility = ObjectGeometry.INOUTLET_CONNECTED_SENSIBLE_AREA;

                    if ((mouseY >= anchorY - verticalSensibility) && (mouseY <= anchorY + verticalSensibility))
                    {
                        if (nInlets == 1)
                        {
                            horizontalSensibility = ObjectGeometry.INOUTLET_CONNECTED_SENSIBLE_AREA;
                            d = Math.Abs(mouseX - start);

                            if (d < horizontalSensibility)
                                return makeInletArea(mouseX, mouseY, 0, d);
                        }
                        else if (nInlets > 0)
                        {
                            if (mouseX < start)
                            //left of first inlet
                            {
                                horizontalSensibility = ObjectGeometry.INOUTLET_CONNECTED_SENSIBLE_AREA;
                                d = Math.Abs(mouseX - start);

                                if (d < horizontalSensibility)
                                    return makeInletArea(mouseX, mouseY, 0, d);
                            }
                            else
                            {
                                horizontalSensibility = inletDistance / 2;
                                int n = (mouseX - start) / inletDistance;
                                d = (mouseX - start) % inletDistance;

                                if ((d <= horizontalSensibility) && (n >= 0) && (n < nInlets))
                                    return makeInletArea(mouseX, mouseY, n, d);
                                else if ((d >= inletDistance - horizontalSensibility) && (n >= 0) && (n < (nInlets - 1)))
                                    return makeInletArea(mouseX, mouseY, n + 1, inletDistance - d);
                            }
                        }
                    }
                }
            }
            return area;
        }


        internal SensibilityArea getDrawingLineOutletSensibilityArea(int mouseX, int mouseY)
        {
            SensibilityArea area = null;
            int verticalSensibility;
            int horizontalSensibility;
            int anchorY = Y + Height;
            int nOutlets = ftsObject.NumberOfOutlets;

            if (itsSketchPad.ConnectingObject != this)
            {

                int d;
                int start = getInletAnchorX(0);

                if (pointInObject(mouseX, mouseY))
                //mouse in object
                {
                    if (nOutlets == 1)
                    {
                        d = Math.Abs(mouseX - start);
                        return makeOutletArea(mouseX, mouseY, 0, d);
                    }
                    else if (nOutlets > 0)
                    {
                        horizontalSensibility = outletDistance / 2;

                        int n = (mouseX - start) / outletDistance;
                        d = (mouseX - start) % outletDistance;

                        if ((d <= horizontalSensibility) && (n >= 0) && (n < nOutlets))
                            return makeOutletArea(mouseX, mouseY, n, d);
                        else if ((d >= outletDistance - horizontalSensibility) && (n >= 0) && (n < (nOutlets - 1)))
                            return makeOutletArea(mouseX, mouseY, n + 1, outletDistance - d);
                    }
                }
                //mouse out of object 
                else
                {
                    verticalSensibility = ObjectGeometry.INOUTLET_CONNECTED_SENSIBLE_AREA;

                    if ((mouseY >= anchorY - verticalSensibility) && (mouseY <= anchorY + verticalSensibility))
                    {
                        if (nOutlets == 1)
                        {
                            horizontalSensibility = ObjectGeometry.INOUTLET_CONNECTED_SENSIBLE_AREA;
                            d = Math.Abs(mouseX - start);

                            if (d < horizontalSensibility)
                                return makeOutletArea(mouseX, mouseY, 0, d);
                        }
                        else if (nOutlets > 0)
                        {
                            if (mouseX < start)
                            //left of first inlet
                            {
                                horizontalSensibility = ObjectGeometry.INOUTLET_CONNECTED_SENSIBLE_AREA;
                                d = Math.Abs(mouseX - start);

                                if (d < horizontalSensibility)
                                    return makeOutletArea(mouseX, mouseY, 0, d);
                            }
                            else
                            {
                                horizontalSensibility = outletDistance / 2;
                                int n = (mouseX - start) / outletDistance;
                                d = (mouseX - start) % outletDistance;

                                if ((d <= horizontalSensibility) && (n >= 0) && (n < nOutlets))
                                    return makeOutletArea(mouseX, mouseY, n, d);
                                else if ((d >= outletDistance - horizontalSensibility) && (n >= 0) && (n < (nOutlets - 1)))
                                    return makeOutletArea(mouseX, mouseY, n + 1, outletDistance - d);
                            }
                        }
                    }
                }
            }
            return area;
        }

        protected internal virtual SensibilityArea findSensibilityArea(int mouseX, int mouseY)
        {
            return null;
        }

        internal int itsWherePopup = ON_OBJECT;
        public int wherePopup()
        {
            return itsWherePopup;
        }

        public virtual void popUpUpdate(bool onInlet, bool onOutlet, SensibilityArea area)
        {
            string text = "";
            if (onInlet)
            {
                itsWherePopup = ON_INLET;
                text = "Disconnect Inlet n." + area.Number;
            }
            else if (onOutlet)
            {
                itsWherePopup = ON_OUTLET;
                text = "Disconnect Outlet n." + area.Number;
            }
            else
            {
                itsWherePopup = ON_OBJECT;
                text = "";
            }
            ObjectPopUp.update(itsWherePopup, text, isInspectable);
        }

        public bool isPopUpVisible
        {
            get
            {
                //UNDONE: Class 'javax.swing.JMenuItem.isVisible()' was not converted.
                //                return ObjectPopUp.Instance.isVisible();
                return true;
            }
        }

        public void popUpEdit(Point p)
        {
            //UNDONE: Class 'java.awt.Component' was not converted.
            //                        popup(Component invoker, GraphicObject object, int x, int y)
            //            ObjectPopUp.Popup(itsSketchPad, this, p.X, p.Y);
        }

        public void runModePopUpEdit(Point p)
        {
            //UNDONE: Class 'javax.swing.JPopupMenu' was not converted.
            //            JPopupMenu popup = RunModePopUpMenu;
            //            if (popup != null)
            //            {
            //                popup.show(itsSketchPad, p.X - 2, p.Y - 2);
            //            }
        }

        //UNDONE: Class 'javax.swing.JPopupMenu' was not converted.
        //        public JPopupMenu RunModePopUpMenu
        //        {
        //            get
        //            {
        //                return null;
        //            }
        //        }

        public virtual void popUpReset() { }

        // This method is called when we want to edit the object 
        // within this editor (its text content or equivalent)
        // The point argument give the mouse click location, when
        // it is usefull

        public virtual void Edit() { }

        public virtual void Edit(Point p) { }

        // This method is called whenever we want to edit the content of
        // an object within a separate editor; do nothing by default

        public virtual void editContent() { }

        public virtual bool hasContent
        {
            get
            {
                return false;
            }
        }

        // Squeack handling; note that squeacks delivered to objects
        // have the locations fields masked to zero, to simplify code writing.

        public virtual void gotSqueack(int squeack, Point mouse, Point oldMouse)
        {
        }

        public void moveBy(int dx, int dy)
        {
            if (dx != 0)
                X = X + dx;
            if (dy != 0)
                Y = Y + dy;
        }

        public void resizing(bool isResizing)
        {
        }

        public bool isResizable
        {
            get
            {
                return true;
            }
        }

        public virtual bool isInspectable
        {
            get
            {
                return false;
            }
        }

        internal ObjectInspector inspector = null;
        public void inspect()
        {
            if (isInspectable)
            {
                if (inspector == null)
                    inspector = new ObjectInspector();
                //UNDONE: Method relocatetoobject Not Yet Implemented.
                //                else
                //                    inspector.relocatetoobject();
            }
        }

        public void inspectionDone()
        {
            inspector.Visible = false;
            //UNDONE: Method Dispose Not Yet Implemented.
            //            inspector.Dispose();
            inspector = null;
        }

        public virtual string Name
        {
            get
            {
                return null;
            }
        }

        public virtual IObjectControlPanel ControlPanel
        {
            get
            {
                return null;
            }
        }

        // Get the bounds in a rectangle

        public void getBounds(Rectangle bounds)
        {
            bounds.X = X;
            bounds.Y = (Y - ObjectGeometry.HIGHLIGHTED_INOUTLET_HEIGHT + ObjectGeometry.INLET_OFFSET + ObjectGeometry.INLET_OVERLAP);
            bounds.Width = Width;
            bounds.Height = (Height + 2 * ObjectGeometry.HIGHLIGHTED_INOUTLET_HEIGHT - ObjectGeometry.INLET_OFFSET - ObjectGeometry.INLET_OVERLAP - ObjectGeometry.OUTLET_OFFSET - ObjectGeometry.OUTLET_OVERLAP);
        }

        // There are two intersect function: one cover the paint needs, and include all
        // the possible space covered by temporary details like highlighted inlets,
        // while the other, coreIntersect, only consider the core objetc, and it is 
        // used for selection

        public bool Intersects(Rectangle r)
        {
            return !((r.X + r.Width <= X) || (r.Y + r.Height <= (Y - ObjectGeometry.HIGHLIGHTED_INOUTLET_HEIGHT + ObjectGeometry.INLET_OFFSET + ObjectGeometry.INLET_OVERLAP - 1)) || (r.X >= X + Width) || (r.Y >= (Y + Height + ObjectGeometry.HIGHLIGHTED_INOUTLET_HEIGHT - ObjectGeometry.OUTLET_OFFSET - ObjectGeometry.OUTLET_OVERLAP + 1)));
        }

        public bool coreIntersects(Rectangle r)
        {
            return !((r.X + r.Width <= X) || (r.Y + r.Height <= Y) || (r.X >= X + Width) || (r.Y >= (Y + Height)));
        }

        public bool pointInObject(int px, int py)
        {
            int x = X;
            int y = Y;
            return ((px >= x) && (px <= x + Width) && (py >= y) && (py <= y + Height));
        }

        public void rectangleUnion(Rectangle r)
        {
            //UNDONE: Method 'javax.swing.SwingUtilities.computeUnion' was not converted.
            //            SwingUtilities.computeUnion(X, Y, Width, Height, r);
        }

        public int Layer
        {
            get
            {
                return ftsObject.Layer;
            }
            set
            {
                ftsObject.Layer = value;
            }
        }

        // Called at GraphicObject disposal

        public void Dispose()
        {
            ftsObject.ObjectListener = null;
        }

        public void showErrorDescription()
        {
            if (ftsObject != null)
            {
                int ax, ay, ah, aw;
                string annotation;
                string val;
                Graphics g;
                Color color;

                annotation = ftsObject.ErrorDescription;

                if (ftsObject.IsError && (object)annotation != null)
                {
                    //UNDONE: Method 'javax.swing.JComponent.getGraphics()' was not converted.
                    //                    g = itsSketchPad.getGraphics();
                    //
                    //                    color = Color.Red;
                    //                
                    //                    paintInlets(g);
                    //                    paintOutlets(g);
                    //                    
                    //                    g.DrawRectangle(new Pen(color), X, Y, Width - 1, Height - 1);

                    //UNDONE: Method 'java.awt.FontMetrics.stringWidth' was not converted.                    
                    //                    aw = itsSketchPad.getFontMetrics(errorFont).stringWidth(annotation) + 1;
                    //                    ah = (int) itsSketchPad(errorFont).getFontMetrics.Height;
                    //                    ax = X + ERROR_MESSAGE_DISPLAY_PAD;
                    //                    ay = Y + ah / 4;
                    //                    
                    //                    
                    //                    color = Color.White;
                    //                    g.DrawString(annotation, errorFont, new SolidBrush(color), ax + 0, ay - 1);
                    //                    g.DrawString(annotation, errorFont, new SolidBrush(color), ax + 0, ay + 1);
                    //                    g.DrawString(annotation, errorFont, new SolidBrush(color), ax + - 1, ay + 0);
                    //                    g.DrawString(annotation, errorFont, new SolidBrush(color), ax + 1, ay + 0);
                    //                    
                    //                    color = Color.Red;
                    //                    g.DrawString(annotation, errorFont, ax + 0, ay + 0);
                    //                    
                    //                    g.Dispose();
                }
            }
        }

        // Assist code
        // Protected against repetitions of assist messages

        //private static FtsAtom assistArgs[] = null;

        [NonSerialized()]
        internal static FtsObject lastAssistedObject;
        [NonSerialized()]
        internal static int lastPosition;

        internal const int ASSIST_OBJECT = 1;
        internal const int ASSIST_INLET = 2;
        internal const int ASSIST_OUTLET = 3;

        internal static int lastAssistOperation = 0;

        public bool canDoAssist(int operation, int n)
        {
            bool ret = false;

            if (itsSketchPad.isMessageReset)
                ret = true;
            else if (lastAssistOperation != operation)
                ret = true;
            else if (lastAssistedObject != ftsObject)
                ret = true;
            else if ((lastAssistOperation == ASSIST_INLET) || (lastAssistOperation == ASSIST_OUTLET))
                ret = (n != lastPosition);

            lastAssistOperation = operation;
            lastPosition = n;
            lastAssistedObject = ftsObject;

            return ret;
        }

        public void assistOnObject()
        {
            if (canDoAssist(ASSIST_OBJECT, 0))
            {
                /*assistArgs[0].stringValue = "object";
                ftsObject.sendMessage(-1, "assist", 1, assistArgs);*/
            }
        }

        public void assistOnInlet(int n)
        {
            if (canDoAssist(ASSIST_INLET, n))
            {
                /*assistArgs[0].SetString("inlet");
                assistArgs[1].SetInt(n);
                ftsObject.sendMessage(-1, "assist", 2, assistArgs);*/
            }
        }

        public void assistOnOutlet(int n)
        {
            if (canDoAssist(ASSIST_OUTLET, n))
            {
                /*assistArgs[0].SetString("outlet");
                assistArgs[1].SetInt(n);
                ftsObject.sendMessage(-1, "assist", 2, assistArgs);*/
            }
        }

        public override string ToString()
        {
            return "GraphicObject<" + ftsObject.ToString() + ">";
        }

        /************** Undo/Redo *******************/
        internal int ux, uy, uw, uh;
        internal int rx, ry, rw, rh;
        internal Font uFont, rFont;

        public void setUndo()
        {
            ux = X; uy = Y; uw = Width; uh = Height;
            uFont = AFont;
        }

        public void setRedo()
        {
            rx = X; ry = Y; rw = Width; rh = Height;
            rFont = AFont;
        }

        public virtual void undo()
        {
            if (uFont != AFont)
                AFont = uFont;
            X = ux;
            Y = uy;
            Width = uw;
            Height = uh;
        }

        public virtual void redo()
        {
            if (rFont != AFont)
                AFont = rFont;
            X = rx;
            Y = ry;
            Width = rw;
            Height = rh;
        }

        public bool instantEdit()
        {
            return false;
        }
    }
}