//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.IO;
using System.Collections;

using ircam.fts.client;
using ircam.jmax;
using ircam.jmax.editors.project;

namespace ircam.jmax.fts
{
    public class FtsAudioConfig : FtsObject
    {
        protected internal FtsArgs args = new FtsArgs();

        static FtsAudioConfig()
        {
            FtsObject.RegisterMessageHandler(typeof(FtsAudioConfig), FtsSymbol.Get("sampling_rates"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsAudioConfig)obj).SetSamplingRates(args.Length, args.Atoms);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsAudioConfig), FtsSymbol.Get("buffer_sizes"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsAudioConfig)obj).SetBufferSizes(args.Length, args.Atoms);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsAudioConfig), FtsSymbol.Get("sampling_rate"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsAudioConfig)obj).SamplingRate = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsAudioConfig), FtsSymbol.Get("buffer_size"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsAudioConfig)obj).BufferSize = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsAudioConfig), FtsSymbol.Get("inputs"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsAudioConfig)obj).SetSources(args.Length, args.Atoms);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsAudioConfig), FtsSymbol.Get("outputs"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsAudioConfig)obj).SetDestinations(args.Length, args.Atoms);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsAudioConfig), FtsSymbol.Get("insert"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsAudioConfig)obj).InsertLabel(args.GetInt(0), args.GetInt(1), args.Atoms, 2, args.Length);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsAudioConfig), FtsSymbol.Get("remove"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsAudioConfig)obj).RemoveLabel(args.GetInt(0));
                });
            FtsObject.RegisterMessageHandler(typeof(FtsAudioConfig), FtsSymbol.Get("clear"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsAudioConfig)obj).Clear();
                });
        }

        public FtsAudioConfig(FtsServer server, FtsObject parent, int id)
            : base(server, parent, id)
        {
            labels = new ArrayList();
        }

        public FtsAudioConfig()
            : base(JMaxLibrary.FtsServer, JMaxLibrary.RootPatcher, FtsSymbol.Get("audio_config"))
        {
            labels = new ArrayList();
            sources = new string[0];
            sourceChannels = new int[0];
            destinations = new string[0];
            destinationChannels = new int[0];
            bufferSizes = new string[0];
            samplingRates = new string[0];
        }

        public ircam.jmax.editors.configuration.AudioConfigPanel Listener
        {
            set
            {
                this.listener = value;

                for (System.Collections.IEnumerator e = labels.GetEnumerator(); e.MoveNext(); )
                {
                    ((FtsAudioLabel)e.Current).Listener = value;
                }
            }
        }

        /*****************************************************/
        /***********     Client -----> Server    *************/
        /*****************************************************/

        public void RequestInsertLabel(int index, string label)
        {
            args.Clear();
            args.AddInt(index);
            args.AddSymbol(FtsSymbol.Get(label));

            try
            {
                Send(FtsSymbol.Get("insert"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsAudioConfig: I/O Error sending insert Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void RequestRemoveLabel(int index)
        {
            args.Clear();
            args.AddInt(index);

            try
            {
                Send(FtsSymbol.Get("remove"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsAudioConfig: I/O Error sending remove Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void RequestSetBufferSize(int bufferSize)
        {
            args.Clear();
            args.AddInt(bufferSize);

            try
            {
                Send(FtsSymbol.Get("buffer_size"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsAudioConfig: I/O Error sending buffer_size Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        public void RequestSetSamplingRate(int sRate)
        {
            args.Clear();
            args.AddInt(sRate);

            try
            {
                Send(FtsSymbol.Get("sampling_rate"), args);
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsAudioConfig: I/O Error sending sampling_rate Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }

        /*************************************************************/

        internal void SetSamplingRates(int nArgs, FtsAtom[] args)
        {
            samplingRates = new string[nArgs];
            for (int i = 0; i < nArgs; i++)
                samplingRates[i] = "" + args[i].intValue;

            if (listener != null)
                listener.SamplingRatesChanged();
        }

        public string[] GetSamplingRates()
        {
            return samplingRates;
        }

        internal int SamplingRate
        {
            get
            {
                return samplingRate;
            }
            set
            {
                samplingRate = value;

                if (listener != null)
                    listener.SamplingRateChanged();
            }
        }

        internal void SetBufferSizes(int nArgs, FtsAtom[] args)
        {
            bufferSizes = new string[nArgs];
            for (int i = 0; i < nArgs; i++)
                bufferSizes[i] = "" + args[i].intValue;

            if (listener != null)
                listener.BufferSizesChanged();
        }

        public string[] GetBufferSizes()
        {
            return bufferSizes;
        }

        internal int BufferSize
        {
            get
            {
                return bufferSize;
            }
            set
            {
                bufferSize = value;

                if (listener != null)
                    listener.BufferSizeChanged();
            }
        }

        internal void SetSources(int nArgs, FtsAtom[] args)
        {
            sources = new string[nArgs / 2];
            sourceChannels = new int[nArgs / 2];
            int j = 0;
            for (int i = 0; i < nArgs; i += 2)
            {
                sources[j] = args[i].symbolValue.ToString();
                sourceChannels[j] = args[i + 1].intValue;
                j++;
            }
            if (listener != null)
                listener.SourcesChanged();
        }

        public string[] GetSources()
        {
            return sources;
        }

        public int GetInDeviceChannels(string input)
        {
            int id = 0;
            for (int i = 0; i < sources.Length; i++)
                if (sources[i].Equals(input))
                {
                    id = i;
                    break;
                }
            return sourceChannels[id];
        }

        internal void SetDestinations(int nArgs, FtsAtom[] args)
        {
            destinations = new string[nArgs / 2];
            destinationChannels = new int[nArgs / 2];
            int j = 0;
            for (int i = 0; i < nArgs; i += 2)
            {
                destinations[j] = args[i].symbolValue.ToString();
                destinationChannels[j] = args[i + 1].intValue;
                j++;
            }
            if (listener != null)
                listener.DestinationsChanged();
        }

        public int GetOutDeviceChannels(string output)
        {
            int id = 0;
            for (int i = 0; i < destinations.Length; i++)
                if (destinations[i].Equals(output))
                {
                    id = i;
                    break;
                }
            return destinationChannels[id];
        }

        public string[] GetDestinations()
        {
            return destinations;
        }

        internal void InsertLabel(int index, int id, FtsAtom[] args, int offset, int nArgs)
        {
            FtsAudioLabel label = new FtsAudioLabel(JMaxLibrary.FtsServer, JMaxLibrary.RootPatcher,
                                                    id, args, offset, nArgs);
            labels.Insert(index, label);

            if (listener != null)
                label.Listener = listener;

            if (listener != null)
                listener.LabelInserted(index, label);
        }

        internal void RemoveLabel(int index)
        {
            if (index >= 0 && index < labels.Count)
                labels.RemoveAt(index);
        }

        public IEnumerator GetLabels()
        {
            return labels.GetEnumerator();
        }

        public FtsAudioLabel GetLabelAt(int index)
        {
            return (FtsAudioLabel)labels[index];
        }

        public FtsAudioLabel GetLabelByName(string lb)
        {
            FtsAudioLabel label;
            for (IEnumerator e = labels.GetEnumerator(); e.MoveNext(); )
            {
                label = (FtsAudioLabel)e.Current;
                if (lb.Equals(label.Label))
                    return label;
            }
            return null;
        }

        public int GetLabelIndex(FtsAudioLabel lb)
        {
            FtsAudioLabel label;
            int i = 0;
            for (IEnumerator e = labels.GetEnumerator(); e.MoveNext(); )
            {
                label = (FtsAudioLabel)e.Current;
                if (lb.Label.Equals(label.Label))
                    return i;
                i++;
            }
            return -1;
        }

        public void Clear()
        {
            labels.Clear();
        }

        private string[] sources;
        private int[] sourceChannels;
        private string[] destinations;
        private int[] destinationChannels;
        private string[] bufferSizes;
        private string[] samplingRates;
        private ArrayList labels;
        private int samplingRate;
        private int bufferSize;
        private ircam.jmax.editors.configuration.AudioConfigPanel listener;
    }
}