//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.IO;
using System.Collections;
using System.Windows.Forms;

using ircam.fts.client;
using ircam.jmax.toolkit;
using ircam.jmax.editors.patcher;
using ircam.jmax.editors.patcher.objects;

namespace ircam.jmax.fts
{
    public class FtsGraphicObject : FtsObject
    {
        static FtsGraphicObject()
        {
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("n_inlets"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsGraphicObject)obj).NumberOfInlets = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("n_outlets"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsGraphicObject)obj).NumberOfOutlets = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("error"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    if (args.Length > 0)
                    {
                        ((FtsGraphicObject)obj).IsError = true;
                        ((FtsGraphicObject)obj).ErrorDescription = args.GetSymbol(0).ToString();
                    }
                    else
                        ((FtsGraphicObject)obj).IsError = false;
                });
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("setErrorDescription"),
                delegate(FtsObject obj, FtsArgs args) { });
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("setX"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsGraphicObject)obj).CurrentX = (float)args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("setY"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsGraphicObject)obj).CurrentY = (float)args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("setWidth"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsGraphicObject)obj).CurrentWidth = (float)args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("setHeight"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsGraphicObject)obj).CurrentHeight = (float)args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("setFont"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsGraphicObject)obj).SetCurrentFont(args.GetSymbol(0).ToString(), args.GetInt(1), args.GetInt(2));
                });
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("setFontName"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsGraphicObject)obj).CurrentFontName = args.GetSymbol(0).ToString();
                });
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("setFontSize"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsGraphicObject)obj).CurrentFontSize = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("setFontStyle"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsGraphicObject)obj).CurrentFontStyle = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("setLayer"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsGraphicObject)obj).CurrentLayer = args.GetInt(0);
                });
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("name"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsGraphicObject)obj).VariableName = args.GetSymbol(0).ToString();
                });
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("persistence"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsGraphicObject)obj).Persistent = (args.GetInt(0));
                });
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("setComment"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsGraphicObject)obj).CurrentComment = args.GetSymbol(0).ToString();
                });
            FtsObject.RegisterMessageHandler(typeof(FtsGraphicObject), FtsSymbol.Get("openFileDialog"),
                delegate(FtsObject obj, FtsArgs args)
                {
                    ((FtsGraphicObject)obj).OpenFileDialog(args.Length, args.Atoms);
                });
        }

        [NonSerialized()]
        private FtsArgs args = new FtsArgs();

        public FtsGraphicObject(FtsServer server, FtsObject parent, int id, string className, FtsAtom[] args, int offset, int length)
            : this(server, parent, id, className, args[offset].stringValue) { }

        public FtsGraphicObject(FtsServer server, FtsObject parent, int id, string className, string description)
            : base(server, parent, id)
        {
            this.description = description;
            this.className = className;
        }

        public FtsGraphicObject(FtsServer server, FtsObject parent, int id) : base(server, parent, id) { }

        public FtsGraphicObject(FtsServer server, FtsObject parent, FtsSymbol className)
            : base(server, parent, className)
        {
            this.description = className.ToString();
        }

        /****************************************************************************/
        /*                          graphic properties                              */
        /****************************************************************************/

        private float x = (float)(-1);
        private float y = (float)(-1);
        private float width = (float)(-1);
        private float height = (float)(-1);
        private bool isError = false;
        private string errorDescription;
        private string font = null;
        private float fontSize = (float)(-1);
        private int fontStyle = -1;
        private int layer = -1;
        private string comment = "";
        private int noutlets = -1;
        private int ninlets = -1;
        private string varName = null;
        private int persistent = -1;

        /// <summary>Get the X property </summary>
        public float X
        {
            get
            {
                return x;
            }
            set
            {
                if (this.x != value)
                {
                    args.Clear();
                    args.AddSymbol(FtsSymbol.Get("x"));
                    args.AddInt((int)value);

                    try
                    {
                        SendProperty(args);
                    }
                    catch (IOException e)
                    {
                        Console.Error.WriteLine("FtsGraphicObject: I/O Error sending setX Message!");
                        Console.Error.WriteLine(e.StackTrace);
                    }

                    this.x = value;
                }
            }
        }

        public float CurrentX
        {
            set
            {
                this.x = value;
            }
        }

        /// <summary>Get the Y property </summary>
        /// <summary>Set the Y property. Tell it to the server, too. </summary>
        public float Y
        {
            get
            {
                return y;
            }
            set
            {
                if (this.y != value)
                {
                    args.Clear();
                    args.AddSymbol(FtsSymbol.Get("y"));
                    args.AddInt((int)value);

                    try
                    {
                        SendProperty(args);
                    }
                    catch (IOException e)
                    {
                        Console.Error.WriteLine("FtsGraphicObject: I/O Error sending setY Message!");
                        Console.Error.WriteLine(e.StackTrace);
                    }

                    this.y = value;
                }
            }
        }

        public float CurrentY
        {
            set
            {
                this.y = value;
            }
        }

        /// <summary>Get the Width property </summary>
        public float Width
        {
            get
            {
                return width;
            }
            set
            {
                if (this.width != value)
                {
                    args.Clear();
                    args.AddSymbol(FtsSymbol.Get("w"));
                    args.AddInt((int)value);

                    try
                    {
                        SendProperty(args);
                    }
                    catch (IOException e)
                    {
                        Console.Error.WriteLine("FtsGraphicObject: I/O Error sending setWidth Message!");
                        Console.Error.WriteLine(e.StackTrace);
                    }

                    this.width = value;
                }
            }
        }

        public float CurrentWidth
        {
            set
            {
                this.width = value;
            }
        }

        /// <summary>Get the Height property </summary>
        /// <summary>Set the Height property. Tell it to the server, too. </summary>
        public float Height
        {
            get
            {
                return height;
            }
            set
            {
                if (this.height != value)
                {
                    args.Clear();
                    args.AddSymbol(FtsSymbol.Get("h"));
                    args.AddInt((int)value);

                    try
                    {
                        SendProperty(args);
                    }
                    catch (IOException e)
                    {
                        Console.Error.WriteLine("FtsGraphicObject: I/O Error sending setHeight Message!");
                        Console.Error.WriteLine(e.StackTrace);
                    }

                    this.height = value;
                }
            }
        }

        public float CurrentHeight
        {
            set
            {
                this.height = value;
            }
        }

        public void SetCurrentBounds(float x, float y, float width, float height)
        {
            this.x = x;
            this.y = y;
            this.width = width;
            this.height = height;
        }

        /// <summary>Get the Error property. Error is a read only property. </summary>
        public bool IsError
        {
            get
            {
                return isError;
            }
            set
            {
                this.isError = value;

                if (listener is IFtsObjectErrorListener)
                    ((IFtsObjectErrorListener)listener).ErrorChanged(isError);
            }
        }

        /// <summary>Get the error description property. Error description is a read only property. </summary>
        public string ErrorDescription
        {
            get
            {
                return errorDescription;
            }
            set
            {
                errorDescription = value;
            }
        }

        /// <summary>Get the font property </summary>
        /// <summary>Set the font property. Tell it to the server, too. </summary>
        public string Font
        {
            get
            {
                return font;
            }
            set
            {
                if ((this.font == null) || (!this.font.Equals(value)))
                {
                    args.Clear();
                    args.AddSymbol(FtsSymbol.Get("font"));
                    args.AddSymbol(FtsSymbol.Get(value));

                    try
                    {
                        SendProperty(args);
                    }
                    catch (IOException e)
                    {
                        Console.Error.WriteLine("FtsGraphicObject: I/O Error sending setFont Message!");
                        Console.Error.WriteLine(e.StackTrace);
                    }

                    this.font = value;
                }
            }
        }

        public void SetCurrentFont(string font, int size, int style)
        {
            if (ObjectListener != null)
                ((GraphicObject)ObjectListener).SetCurrentFont(font, size, style);
        }

        public string CurrentFontName
        {
            set
            {
                this.font = value;

                if (PatcherFontManager.Instance.IsToRecoverFont(font))
                    font = PatcherFontManager.Instance.RecoveringFont;
            }
        }

        /// <summary>Set the font size property. Tell it to the server, too. </summary>
        /// <summary>Get the font size property </summary>
        public float FontSize
        {
            get
            {
                return fontSize;
            }
            set
            {
                if (this.fontSize != value)
                {
                    args.Clear();
                    args.AddSymbol(FtsSymbol.Get("fs"));
                    args.AddDouble(value);

                    try
                    {
                        SendProperty(args);
                    }
                    catch (IOException e)
                    {
                        Console.Error.WriteLine("FtsGraphicObject: I/O Error sending setFontSize Message!");
                        Console.Error.WriteLine(e.StackTrace);
                    }

                    this.fontSize = value;
                }
            }
        }

        public float CurrentFontSize
        {
            set
            {
                this.fontSize = value;
            }
        }

        /// <summary>Get the font style property </summary>
        /// <summary>Set the font size property. Tell it to the server, too. </summary>
        public int FontStyle
        {
            get
            {
                return fontStyle;
            }
            set
            {
                if (this.fontStyle != value)
                {
                    args.Clear();
                    args.AddSymbol(FtsSymbol.Get("fst"));
                    args.AddInt(value);

                    try
                    {
                        SendProperty(args);
                    }
                    catch (IOException e)
                    {
                        System.Console.Error.WriteLine("FtsGraphicObject: I/O Error sending setFontStyle Message!");
                        Console.Error.WriteLine(e.StackTrace);
                    }

                    this.fontStyle = value;
                }
            }
        }

        public int CurrentFontStyle
        {
            set
            {
                this.fontStyle = value;
            }
        }

        /// <summary>Get the layer  property </summary>
        /// <summary>Set the layer property. Tell it to the server, too. </summary>
        public int Layer
        {
            get
            {
                return layer;
            }
            set
            {
                if (this.layer != value)
                {
                    args.Clear();
                    args.AddSymbol(FtsSymbol.Get("layer"));
                    args.AddInt(value);

                    try
                    {
                        SendProperty(args);
                    }
                    catch (IOException e)
                    {
                        Console.Error.WriteLine("FtsGraphicObject: I/O Error sending setLayer Message!");
                        Console.Error.WriteLine(e.StackTrace);
                    }

                    this.layer = value;
                }
            }
        }

        public int CurrentLayer
        {
            set
            {
                this.layer = value;
            }
        }

        public virtual int NumberOfInlets
        {
            get
            {
                return ninlets;
            }
            set
            {
                ninlets = value;

                if ((Parent != null) && (Parent is FtsPatcherObject))
                    ((FtsPatcherObject)Parent).FirePatcherChangedNumberOfInlets(ninlets);

                if (listener is IFtsInletsListener)
                    ((IFtsInletsListener)listener).InletsChanged(ninlets);
            }
        }

        public virtual int NumberOfOutlets
        {
            get
            {
                return noutlets;
            }
            set
            {
                noutlets = value;

                if ((Parent != null) && (Parent is FtsPatcherObject))
                    ((FtsPatcherObject)Parent).FirePatcherChangedNumberOfOutlets(noutlets);

                if (listener is IFtsOutletsListener)
                    ((IFtsOutletsListener)listener).OutletsChanged(noutlets);
            }
        }

        public string Comment
        {
            get
            {
                return comment;
            }
            set
            {
                args.Clear();
                args.AddSymbol(FtsSymbol.Get("comment"));
                args.AddSymbol(FtsSymbol.Get(value));

                try
                {
                    SendProperty(args);
                }
                catch (IOException e)
                {
                    Console.Error.WriteLine("FtsGraphicObject: I/O Error sending setComment Message!");
                    Console.Error.WriteLine(e.StackTrace);
                }
                this.comment = value;
            }
        }

        public string CurrentComment
        {
            set
            {
                this.comment = value;
            }
        }

        public void requestSetName(string name)
        {
            if (this.varName != name)
            {
                args.Clear();
                args.AddSymbol(FtsSymbol.Get(name));

                try
                {
                    Send(FtsSymbol.Get("name"), args);
                }
                catch (IOException e)
                {
                    Console.Error.WriteLine("FtsGraphicObject: I/O Error sending set_name Message!");
                    Console.Error.WriteLine(e.StackTrace);
                }

                this.varName = name;
            }
        }

        public string VariableName
        {
            get
            {
                return varName;
            }
            set
            {
                this.varName = value;

                if (ObjectListener != null)
                    ((GraphicObject)ObjectListener).CurrentName = value;
            }
        }

        public int Persistent
        {
            get
            {
                return persistent;
            }
            set
            {
                this.persistent = value;

                if (ObjectListener != null)
                    ((GraphicObject)ObjectListener).Redraw();
            }
        }

        public void RequestSetPersistent(int persistent)
        {
            if (this.persistent != persistent)
            {
                args.Clear();
                args.AddInt(persistent);

                try
                {
                    Send(FtsSymbol.Get("persistence"), args);
                }
                catch (IOException e)
                {
                    Console.Error.WriteLine("FtsGraphicObject: I/O Error sending persistence Message!");
                    Console.Error.WriteLine(e.StackTrace);
                }
            }
        }

        public void SendDoubleClick()
        {
            try
            {
                Send(FtsSymbol.Get("double_click"));
            }
            catch (IOException e)
            {
                Console.Error.WriteLine("FtsGraphicObject: I/O Error sending double_click Message!");
                Console.Error.WriteLine(e.StackTrace);
            }
        }
        /****************************************************************************/

        [NonSerialized()]
        private object listener;

        /// <summary>Get the current object listener </summary>
        /// <summary>Set the unique object listener </summary>
        public object ObjectListener
        {
            get
            {
                return listener;
            }
            set
            {
                listener = value;
            }
        }

        [NonSerialized()]
        private IFtsGraphicListener graphicListener = null;

        public IFtsGraphicListener GraphicListener
        {
            get
            {
                return graphicListener;
            }
            set
            {
                graphicListener = value;
            }
        }

        public string ClassName
        {
            get
            {
                return className;
            }
        }

        public virtual string Description
        {
            get
            {
                return description;
            }
            set
            {
                description = value;
            }
        }

        public bool IsARootPatcher
        {
            get
            {
                return (Parent == JMaxLibrary.RootPatcher);
            }
        }

        public FtsPatcherObject GetRootPatcher()
        {
            FtsGraphicObject current = this;

            while (!current.IsARootPatcher)
                current = (FtsGraphicObject)current.Parent;

            return (FtsPatcherObject)current;
        }

        public IEnumerator getGenealogy()
        {
            ArrayList gen = new ArrayList();
            FtsGraphicObject current = this;
            gen.Add(current);

            if (!IsARootPatcher)
                while (!((FtsGraphicObject)current.Parent).IsARootPatcher)
                {
                    gen.Insert(0, current.Parent);
                    current = (FtsGraphicObject)current.Parent;
                }
            if (((FtsGraphicObject)current.Parent).IsARootPatcher)
                gen.Insert(0, current.Parent);

            return gen.GetEnumerator();
        }

        private string description;
        private string className;
        /*************************************************************/
        //final variables used by invokeLater method

        [NonSerialized()]
        private OpenFileDialog openFileDialog;
        [NonSerialized()]
        private Form parentForm;
        [NonSerialized()]
        private string dialogText;
        [NonSerialized()]
        private string callbackMethod;

        public void OpenFileDialog(int nArgs, FtsAtom[] a)
        {
            parentForm = null;
            string defaultPath = "";
            string defaultName = "";
            callbackMethod = a[0].symbolValue.ToString();
            dialogText = a[1].symbolValue.ToString();
            if (nArgs > 2)
            {
                defaultPath = a[2].symbolValue.ToString();
                defaultName = a[3].symbolValue.ToString();
            }

            openFileDialog = new OpenFileDialog();
            openFileDialog.InitialDirectory = defaultPath;
            openFileDialog.Title = dialogText;
            openFileDialog.FileName = defaultName;

            if (this is FtsObjectWithEditor)
                parentForm = ((FtsObjectWithEditor)this).EditorForm;

            FtsObject current = this;
            while (((parentForm == null) || (!parentForm.Visible)) && (!IsARootPatcher))
            {
                current = current.Parent;
                if (current is FtsObjectWithEditor)
                    parentForm = ((FtsObjectWithEditor)current).EditorForm;
            }

            if (parentForm != null)
            {
                if (openFileDialog.ShowDialog() == DialogResult.OK)
                {
                    string path = openFileDialog.FileName;
                    args.Clear();
                    args.AddSymbol(FtsSymbol.Get(path));

                    try
                    {
                        Send(FtsSymbol.Get(callbackMethod), args);
                    }
                    catch (IOException e)
                    {
                        Console.Error.WriteLine("FtsGraphicObject: I/O Error sending" + callbackMethod + " Message!");
                        Console.Error.WriteLine(e.StackTrace);
                    }
                }
            }
        }
    }
}