//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

using System;
using System.Collections;
using System.Windows.Forms;

namespace ircam.jmax.toolkit
{
    /// <summary> An abstract base class implementing the selection interface,
    /// taking care of the listener's mechanism; actually, only the 
    /// selectAll() method needs to be implemented.
    /// An abstract selection implements the Transferable interface, in
    /// order to provide a support for clipboard operations based on selections.
    /// Add the flavours this selection can support via the addFlavour method,
    /// And redefine the getTransferData method in order to return the
    /// Transferable suited for your kind of object
    /// </summary>
    /// <seealso cref="SelectionHandler">
    /// </seealso>
    public abstract class AbstractSelection : ISelectionHandler, IDataObject
    {
        /// <summary> constructor</summary>
        public AbstractSelection()
        {
            selected = new MaxVector();
            listeners = new MaxVector();
            dataFlavors = new MaxVector();
        }

        /// <summary> select the given object </summary>
        public void Select(object obj)
        {
            selected.AddElement(obj);
            NotifyListeners(OBJECT_SELECTED);
        }

        /// <summary> select the given enumeration of objects </summary>
        public void Select(IEnumerator e)
        {
            int count = 0;
            while (e.MoveNext())
            {
                selected.AddElement(e.Current);
                count++;
            }

            if (count == 1)
                NotifyListeners(OBJECT_SELECTED);
            else if (count > 1)
                NotifyListeners(GROUP_SELECTED);
        }

        /// <summary> remove the object from the selection (deselect the given object)</summary>
        public void DeSelect(object obj)
        {
            selected.RemoveElement(obj);
            NotifyListeners(OBJECT_DESELECTED);
        }

        /// <summary> remove the Enumeration of object from the selection (deselect the given
        /// enumeration of objects)
        /// </summary>
        public void DeSelect(IEnumerator e)
        {
            int count = 0;
            while (e.MoveNext())
            {
                selected.RemoveElement(e.Current);
                count++;
            }

            if (count == 1)
                NotifyListeners(OBJECT_DESELECTED);
            else if (count > 1)
                NotifyListeners(GROUP_DESELECTED);
        }


        /// <summary> returns true if the object is currently selected</summary>
        public bool IsInSelection(object obj)
        {
            return selected.Contains(obj);
        }

        /// <summary> returns an enumeration of all the selected objects</summary>
        public IEnumerator Selected
        {
            get
            {
                return selected.Elements();
            }
        }

        /// <summary> returns the lenght of the selection (how many selected elements) </summary>
        public int Size()
        {
            return selected.Size;
        }

        /// <summary> deselects all the objects currently selected</summary>
        public void DeselectAll()
        {
            selected.RemoveAllElements();
            NotifyListeners(GROUP_DESELECTED);
        }


        /// <summary> selects all the objects of the data model</summary>
        public abstract void SelectAll();


        /// <summary> require to be notified when data change</summary>
        public void AddSelectionListener(ISelectionListener theListener)
        {
            listeners.AddElement(theListener);
        }

        /// <summary> remove the listener</summary>
        public void RemoveSelectionListener(ISelectionListener theListener)
        {
            listeners.RemoveElement(theListener);
        }


        /// <summary> call back the listener, the selection content changed</summary>
        protected internal void NotifyListeners(int type)
        {
            ISelectionListener sl;

            for (IEnumerator e = listeners.Elements(); e.MoveNext(); )
            {
                sl = (ISelectionListener)e.Current;

                switch (type)
                {
                    case OBJECT_SELECTED:
                        sl.ObjectSelected();
                        break;

                    case GROUP_SELECTED:
                        sl.GroupSelected();
                        break;

                    case OBJECT_DESELECTED:
                        sl.ObjectDeselected();
                        break;

                    case GROUP_DESELECTED:
                        sl.GroupDeselected();
                        break;
                }
            }
        }

        /// <summary>utility function </summary>
        public void SetData(object flavor)
        {
            dataFlavors.AddElement(flavor);
        }

        public void SetData(string format, object flavor)
        {
            dataFlavors.AddElement(flavor);
        }

        public void SetData(Type format, object flavor)
        {
            dataFlavors.AddElement(flavor);
        }

        public void SetData(string format, bool autoConvert, object flavor)
        {
            dataFlavors.AddElement(flavor);
        }

        /// <summary>Transferable interface </summary>
        public bool GetDataPresent(string format)
        {
            return false;
        }

        public bool GetDataPresent(Type flavor)
        {
            for (int i = 0; i < dataFlavors.Size; i++)
            {
                if (flavor.Equals(dataFlavors.ObjectArray[i]))
                    return true;
            }
            return false;
        }

        public bool GetDataPresent(string format, bool autoConvert)
        {
            return false;
        }

        /// <summary>Transferable interface </summary>
        public string[] GetFormats()
        {
            return (string[])dataFlavors.ObjectArray;
        }

        public string[] GetFormats(bool autoConvert)
        {
            return (string[])dataFlavors.ObjectArray;
        }

        /// <summary>Transferable interface </summary>
        public object GetData(string flavor)
        {
            return null;
        }

        public object GetData(Type flavor)
        {
            return null;
        }

        public object GetData(string flavor, bool autoConvert)
        {
            return null;
        }

        //--- Fields

        public MaxVector selected;
        protected internal MaxVector listeners;
        protected internal MaxVector dataFlavors;

        public const int OBJECT_SELECTED = 1;
        public const int GROUP_SELECTED = 2;
        public const int OBJECT_DESELECTED = 4;
        public const int GROUP_DESELECTED = 8;
        public const int SELECTION_CHANGED = 16;

        public const int ALL_EVENTS = OBJECT_SELECTED | GROUP_SELECTED | OBJECT_DESELECTED | GROUP_DESELECTED | SELECTION_CHANGED;
    }
}