//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.Windows.Forms;

namespace ircam.jmax.toolkit
{
    /// <summary> The base classe of modules that handles the user interaction.
    /// Specialization of these modules are used by the tools, 
    /// in order to separate the low-level interaction (example rubber-banding 
    /// of a selection rect) 
    /// from the semantic (selection of the objects in the rect).
    /// This base class acts like a multiple adapter for the derived class.
    /// </summary>
    public class InteractionModule
    {
        /// <summary> constructor. </summary>
        public InteractionModule()
        {
        }

        /// <summary> called when this modules must take the interaction.
        /// don't call this function directly: call Tool.mountIModule instead
        /// </summary>
        protected internal void TakeInteraction(GraphicContext theGc)
        {
            if (currentActiveModule != null)
                currentActiveModule.UnBindFromProducer(); //of the old gc

            gc = theGc;
            currentActiveModule = this;

            BindToProducer(gc.GraphicSource); //in the new gc
        }

        /// <summary> make this UI Module listen the event coming from the given event producer.</summary>
        protected internal void BindToProducer(Control eventProducer)
        {
            if (eventProducer != null)
            {
                gc.GraphicSource.Click += new System.EventHandler(this.MouseClicked);
                gc.GraphicSource.MouseEnter += new System.EventHandler(this.MouseEntered);
                gc.GraphicSource.MouseLeave += new System.EventHandler(this.MouseExited);
                gc.GraphicSource.MouseDown += new MouseEventHandler(this.MousePressed);
                gc.GraphicSource.MouseUp += new MouseEventHandler(this.MouseReleased);
                gc.GraphicSource.MouseMove += new MouseEventHandler(this.MouseMoved);
                gc.GraphicSource.KeyDown += new KeyEventHandler(this.KeyPressed);
                gc.GraphicSource.KeyUp += new KeyEventHandler(this.KeyReleased);
                gc.GraphicSource.KeyPress += new KeyPressEventHandler(this.KeyTyped);
            }
        }

        /// <summary> stop listening events from the current producer</summary>
        protected internal virtual void UnBindFromProducer()
        {
            if (gc.GraphicSource != null)
            {
                gc.GraphicSource.Click -= new System.EventHandler(this.MouseClicked);
                gc.GraphicSource.MouseEnter -= new System.EventHandler(this.MouseEntered);
                gc.GraphicSource.MouseLeave -= new System.EventHandler(this.MouseExited);
                gc.GraphicSource.MouseDown -= new MouseEventHandler(this.MousePressed);
                gc.GraphicSource.MouseUp -= new MouseEventHandler(this.MouseReleased);
                gc.GraphicSource.MouseMove -= new MouseEventHandler(this.MouseMoved);
                gc.GraphicSource.KeyDown -= new KeyEventHandler(this.KeyPressed);
                gc.GraphicSource.KeyUp -= new KeyEventHandler(this.KeyReleased);
                gc.GraphicSource.KeyPress -= new KeyPressEventHandler(this.KeyTyped);
            }
        }

        /// <summary> set the starting point for this interaction module</summary>
        public virtual void InteractionBeginAt(int x, int y) { }

        /// <summary>temporarly suspend listening the events from the producer
        /// (does not change the current interaction module)
        /// </summary>
        public static void Suspend()
        {
            currentActiveModule.UnBindFromProducer();
            suspended = true;
        }

        /// <summary>is there a suspended module? </summary>
        public static bool IsSuspended
        {
            get
            {
                return suspended;
            }
        }

        /// <summary>resume the activitiy of the current Active module</summary>
        public static void Resume()
        {
            currentActiveModule.BindToProducer(currentActiveModule.gc.GraphicSource);
            suspended = false;
        }

        //----------- Mouse interface ------------
        public void MouseClicked(object sender, EventArgs e)
        {
        }

        public void MouseEntered(object sender, EventArgs e)
        {
        }

        public void MouseExited(object sender, EventArgs e)
        {
        }

        public virtual void MousePressed(object sender, MouseEventArgs e) { }

        public virtual void MouseReleased(object sender, MouseEventArgs e) { }

        //------------- Motion interface ----------
        public virtual void MouseDragged(object sender, MouseEventArgs e) { }

        public void MouseMoved(object sender, MouseEventArgs e)
        {
        }

        //------------- key interface ----------
        public void KeyTyped(object sender, KeyPressEventArgs e)
        {
        }

        public void KeyPressed(object sender, KeyEventArgs e)
        {
        }

        public void KeyReleased(object sender, KeyEventArgs e)
        {
        }

        //--------------- Fields
        internal static InteractionModule currentActiveModule = null;
        internal static bool suspended = false;

        protected internal GraphicContext gc;
    }
}