//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.Drawing;
using System.Windows.Forms;

namespace ircam.jmax.toolkit
{
    /// <summary> An abstract interaction module used to move a selection of objects.
    /// At the end of a move operation, it communicates the new position to the listener.
    /// Just redefine the XORDraw method to specify what to draw during drag.
    /// </summary>
    /// <seealso cref="DragListener">
    /// </seealso>
    abstract public class SelectionMover : InteractionModule, IXORPainter
    {
        /// <summary> constructor. 
        /// It uses a XORHandler to keep track of the drawing positions.
        /// </summary>
        public SelectionMover(IDragListener theListener, int theMovement)
            : base()
        {
            itsListener = theListener;
            itsXORHandler = new XORHandler(this, XORHandler.ABSOLUTE);
            itsStartingPoint = new Point(0, 0);

            itsMovements = theMovement;
        }

        /// <summary> set the direction of movement (constants HORIZONTAL_MOVEMENT  or 
        /// VERTICAL_MOVEMENT, or both)
        /// </summary>
        public int Direction
        {
            set
            {
                itsMovements = value;
            }
        }

        /// <summary> sets the point on which to start the movement</summary>
        public override void InteractionBeginAt(int x, int y)
        {
            itsStartingPoint = new Point(x, y);
            itsXORHandler.BeginAt(x, y);
            InteractionSemaphore.Lock();
        }

        /// <summary> overrides InteractionModule.mouseDragged()</summary>
        public override void MouseDragged(object sender, MouseEventArgs e)
        {
            itsXORHandler.MoveTo(e.X, e.Y);
        }

        public void MoveTo(int x, int y)
        {
            itsXORHandler.MoveTo(x, y);
        }

        /// <summary> overrides InteractionModule.mouseReleased()</summary>
        public override void MouseReleased(object sender, MouseEventArgs e)
        {
            itsXORHandler.End();

            int endX = itsStartingPoint.X;
            int endY = itsStartingPoint.Y;

            if ((itsMovements & VERTICAL_MOVEMENT) != 0)
                endY = e.Y;
            if ((itsMovements & HORIZONTAL_MOVEMENT) != 0)
                endX = e.X;

            itsListener.DragEnd(endX, endY, e);
            InteractionSemaphore.Unlock();
        }

        /// <summary> from the XORPainter interface</summary>
        public void XORErase()
        {
            XORDraw(previousX, previousY);
        }

        /// <summary> From the XORPainter interface. The actual drawing function.</summary>
        abstract public void XORDraw(int dx, int dy);

        //---- Fields
        protected internal IDragListener itsListener;
        protected internal XORHandler itsXORHandler;
        protected internal int previousX;
        protected internal int previousY;

        protected internal Point itsStartingPoint;
        protected internal int itsMovements;

        public int dragMode;
        public const int RECT_DRAG = 0;
        public const int GROUP_DRAG = 1;
        public const int HORIZONTAL_MOVEMENT = 1;
        public const int VERTICAL_MOVEMENT = 2;
    }
}