//
// jMax
// Copyright (C) 1994, 1995, 1998, 1999 by IRCAM-Centre Georges Pompidou, Paris, France.
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// See file LICENSE for further informations on licensing terms.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
// 

using System;
using System.Collections;

namespace ircam.jmax.toolkit
{
    /// <summary> This class centralizes the (graphic) context switching and the 
    /// resulting tool activation/desactivation. A generic editor calls the ToolManager
    /// to add the set of Tools and ContextSwitchers it needs; a Switcher is an object
    /// that implements a GraphicContext switch politic (ex., in response
    /// to a WindowActivateEvent, or to a MouseEnter in the editor's Component area). 
    /// When such a switch event occurs, the ToolManager takes care of redirecting the activity 
    /// of the current Tool on the new GraphicContext.<br>
    /// The activity of the ToolManager and of an EditorToolbar are coordinated, in the 
    /// sense that the EditorToolbar uses the ToolManager to change tool when the user
    /// clicks on the toolbar.
    /// Changing tool can also be done programmatically (ex.: by a "one shot tool", that
    /// perform an action and then mount the default tool).
    /// The ToolManager can't be static, every editor class (not necessarily every instance)
    /// have to have its ToolManager.
    /// </summary>
    /// <seealso cref="ircam.jmax.toolkit.ContextSwitcher">
    /// </seealso>
    /// <seealso cref="ircam.jmax.toolkit.Tool">
    /// 
    /// </seealso>

    //-------------------------------------------
    // Implementation notes: this class offer a set of functionalities formerly offered
    // by the EditorToolbar class, and uses the ContextSwitching logic to break the assumption
    // that an editor's GraphicContext is associated to a Window (this a prerequisite
    // for the implementation of "tracks" of the Sequence editor. )
    // Eventually, also the "ToolSwitching" politic should be isolated into specialized Switchers, 
    // making the EditorToolbar just one of such "ToolSwitcher".
    //-------------------------------------------

    public class ToolManager : ISwitchListener, ILockListener
    {
        /// <summary> Constructor without a provider. The tools must be communicated to
        /// this Manager via addTool() calls 
        /// </summary>
        public ToolManager() { }

        /// <summary> Constructor with a provider. It's an alternative to insert all the tools
        /// via direct addTool() calls 
        /// </summary>
        public ToolManager(IToolProvider theProvider)
        {
            Tool aTool;
            for (IEnumerator e = theProvider.Tools; e.MoveNext(); )
            {
                aTool = (Tool)e.Current;
                if (aTool == null)
                    System.Console.Error.WriteLine("warning: trying to add a null tool");
                else
                {
                    AddTool(aTool);
                }
            }
        }

        /// <summary> Adds a switcher to this manager </summary>
        public void AddContextSwitcher(IContextSwitcher switcher)
        {
            switcher.AddSwitchListener(this);
        }

        /// <summary> Remove a Switcher </summary>
        public void RemoveContextSwitcher(IContextSwitcher switcher)
        {
            switcher.RemoveSwitchListener(this);
        }

        /// <summary> Add a Tool to the manager </summary>
        public void AddTool(Tool tool)
        {
            tools.Add(tool);
        }

        /// <summary> Returns an enumeration of the tools currently handled by this manager </summary>
        public IEnumerator Tools
        {
            get
            {
                return tools.GetEnumerator();
            }
        }

        /// <summary> Returns the tool of the given name, if any </summary>
        // Implementation notes: an Hashtable?
        public Tool GetToolByName(string name)
        {
            Tool temp;
            for (IEnumerator e = Tools; e.MoveNext(); )
            {
                temp = (Tool)e.Current;
                if (temp.Name.Equals(name))
                    return temp;
            }

            return null;
        }

        /// <summary> Make the given tool the current one and activates it on the given graphic context </summary>
        public void Activate(Tool tool, GraphicContext gc)
        {
            currentContext = gc;

            ChangeTool(tool);
        }

        /// <summary> Programmatically make the given tool the current one, on the same
        /// graphic context then the old 
        /// </summary>
        public void ChangeTool(Tool newTool)
        {
            if (currentTool != null)
                currentTool.Deactivate();
            if (currentContext != null)
                newTool.ReActivate(currentContext);

            currentTool = newTool;

            if (currentTool != null)
            {
                ToolNotification(newTool);
            }
        }

        /// <summary> utility function. Communicate the tool-change event to the listeners</summary>
        private void ToolNotification(Tool theTool)
        {
            ToolChangeEvent aEvent = new ToolChangeEvent(theTool);

            IToolListener aListener;
            for (IEnumerator e = listeners.GetEnumerator(); e.MoveNext(); )
            {
                aListener = (IToolListener)e.Current;
                aListener.ToolChanged(aEvent);
            }
        }

        /// <summary> Called when a switcher triggered a graphic context. This should not
        /// be called directly by the user (use ContextSwitchers instead)
        /// </summary>
        public void ContextChanged(GraphicContext gc)
        {
            if (InteractionSemaphore.IsLocked)
            {
                InteractionSemaphore.Caller = this;
                lastContext = gc;
                return;
            }

            if (gc != null)
            {
                gc.Activate();
                currentContext = gc;
            }

            if (currentTool != null)
            {
                currentTool.ReActivate(gc);
            }
        }

        /// <summary> Called when a GraphicContext is destroyed (ex: the window associated to
        /// a WindowContextSwitcher is disposed). This should not be called directly by the user,
        /// but indirectly by a ContextSwitcher.
        /// </summary>
        public void ContextDestroyed(GraphicContext gc)
        {
            gc.Destroy();
        }

        /// <summary> Returns the current active tool </summary>
        public Tool CurrentTool
        {
            get
            {
                return currentTool;
            }
        }

        /// <summary> Returns the current active graphic context </summary>
        public GraphicContext CurrentGC
        {
            get
            {
                return currentContext;
            }
        }

        /// <summary> Inscribe a new Tool change listener to this Manager</summary>
        public void AddToolListener(IToolListener tl)
        {
            listeners.Add(tl);
        }


        /// <summary> remove a tool change listener</summary>
        public void RemoveToolListener(IToolListener tl)
        {
            listeners.Remove(tl);
        }

        public void Unlocked()
        {
            if (lastContext != null)
                ContextChanged(lastContext);

            lastContext = null;
        }

        //--- ToolManager fields ---
        internal ArrayList tools;
        internal GraphicContext currentContext;
        internal Tool currentTool;
        internal ArrayList listeners;
        internal bool contextLocked = false;
        internal GraphicContext lastContext;
    }
}