#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of Karesansui Core.
#
# Copyright (C) 2009-2010 HDE, Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#

"""
@authors: Junichi Shinohara <junichi@karesansui-project.info>
"""


import glob
import re
import os.path

from karesansui.lib.dict_op import DictOp
from karesansui.lib.utils import preprint_r, is_readable
from karesansui.lib.const import CRONTAB_PARSER_CONF_LIST
from karesansui.lib.file.configfile import ConfigFile

"""
Define Variables for This Parser
"""

class crontabParser:

    _delim               = "[ \t]+"
    _new_delim           = " "
    _env_delim           = "="
    _new_env_delim       = "="
    _comment             = "#"

    _module = "crontab"

    def __init__(self, paths=None):
        self.dop = DictOp()
        self.dop.addconf(self._module,{})
        if paths is None:
            self.set_source_file(CRONTAB_PARSER_CONF_LIST)
        else:
            self.set_source_file(paths)
        self.base_parser_name = self.__class__.__name__

    def set_delim(self, delim="="):
        self._delim = delim

    def set_new_delim(self, delim="="):
        self._new_delim = delim

    def _search_dir(self, path):
        retval = []
        if os.path.exists(path) is True and os.path.isdir(path) is True:
            for path_name in glob.glob(path + "/*"):
                retval.append(path_name)
        return retval

    def set_crontab_path(self, path):
        self.crontab_path = path

    def set_crond_paths(self, path):
        self.crond_paths = self._search_dir(path)

    def set_cronuser_paths(self, path):
        self.cronuser_paths = self._search_dir(path)

    def set_source_file(self, paths=[]):
        self.set_crontab_path(paths[0])
        self.set_crond_paths(paths[1])
        self.set_cronuser_paths(paths[2])
        self.paths = [self.crontab_path] + self.crond_paths + self.cronuser_paths
        return True

    def get_source_file(self):
        return self.paths

    def source_file(self):
        return self.get_source_file()

    def _read_conf(self, path_name, owner_name = False):
        retval = {}

        if os.path.exists(path_name) is True and \
            os.path.isfile(path_name) is True and \
            is_readable(path_name) is True:

            res = ConfigFile(path_name).read()

            dop = DictOp()
            dop.addconf('parser', {})
            dop.set('parser', ["path_name"], path_name)

            #print path_name
            record_num = 0
            for _aline in res:
                str_record_num = str(record_num)
                _aline = _aline.rstrip('\r\n')
                line_match_flag = False

                # envelop variable parse
                if line_match_flag is False:
                    regex_str_line = "^(?P<comment>%(comment)s)?[ \t]*(?P<key>[^ \t]+)%(delim)s(?P<value>.*)$" % {'comment': self._comment, 'delim': self._env_delim}
                    m_line = re.compile(r"%s" % regex_str_line).match(_aline)
                    if m_line:
                        comment = m_line.group('comment')
                        key     = m_line.group('key')
                        value   = m_line.group('value')
                        line_match_flag = True

                        if not value.rfind(self._comment) == -1:
                            value = value[:value.rfind(self._comment)]

                        dop.set('parser', ['variable', key], value)

                        if comment is not None:
                            dop.comment('parser', ['variable', key])
                        #print "%s=%s" % (key, dop.get('parser', ['variable', key]))

                # schedule parse
                if line_match_flag is False:
                    regex_str_line = "^(?P<comment>%(comment)s)?[ \t]*(?P<minute>[^ \t]+)%(delim)s(?P<hour>[^ \t]+)%(delim)s(?P<day>[^ \t]+)%(delim)s(?P<month>[^ \t]+)%(delim)s(?P<week>[^ \t]+)%(delim)s(?P<value>.*)$" % {'comment': self._comment, 'delim': self._delim}
                    m_line = re.compile(r"%s" % regex_str_line).match(_aline)
                    if m_line:
                        comment = m_line.group('comment')
                        value   = m_line.group('value')
                        line_match_flag = True

                        if not value.rfind(self._comment) == -1:
                            value = value[:value.rfind(self._comment)]

                        for filed_name in ['minute', 'hour', 'day', 'month', 'week']:
                            range = '*'
                            interval = '*'
                            filed_name_match_flag = False

                            if filed_name_match_flag is False:
                                m_div = re.compile(r"^(?P<range>[^/]+)/(?P<interval>[^/]+)$").match(m_line.group(filed_name))
                                if m_div:
                                    range = m_div.group('range')
                                    interval = m_div.group('interval')
                                    filed_name_match_flag = True
                            if filed_name_match_flag is False:
                                if m_line.group(filed_name) == '*':
                                    range = '*'
                                    filed_name_match_flag = True
                            if filed_name_match_flag is False:
                                m_multi = re.compile(r"^[0-9\,\-]+$").match(m_line.group(filed_name))
                                if m_multi:
                                    range = m_line.group(filed_name)
                                    filed_name_match_flag = True

                            dop.set('parser', ['schedule', str_record_num, filed_name, 'range'], range)
                            dop.set('parser', ['schedule', str_record_num, filed_name, 'interval'], interval)

                        if owner_name is False:
                            regex_str_value = "^(?P<owner_name>[^ \t]+)%(delim)s(?P<command>.*)$" % {'delim': self._delim}
                            regex_value = re.compile(r"%s" % regex_str_value)
                            m_value = regex_value.match(value)
                            if m_value:
                                dop.set('parser', ['schedule', str_record_num, 'owner_name'], m_value.group('owner_name'))
                                dop.set('parser', ['schedule', str_record_num, 'command'], m_value.group('command'))
                        else:
                            dop.set('parser', ['schedule', str_record_num, 'owner_name'], owner_name)
                            dop.set('parser', ['schedule', str_record_num, 'command'], value)

                        if comment is not None:
                            dop.comment('parser', ['schedule', str_record_num])
                        #print dop.get('parser', ['schedule', str_record_num])

                if line_match_flag is False:
                    continue
                #print record_num
                record_num += 1
            return dop.getconf('parser')

    def _write_conf(self, conf, system_crontab, dryrun=False):
        retval = True

        dop = DictOp()
        dop.addconf('parser', conf)
        if dop.isset('parser', ['path_name']) is True:
            path_name = dop.get('parser', ['path_name'])

        lines = []

        # restore envelop variable
        env_var_list = dop.query('parser', ['variable'])
        if type(env_var_list) != list:
            env_var_list = []
        for var_name in env_var_list:
            var_value = dop.get('parser', ['variable', var_name])
            if dop.iscomment('parser', ['variable', var_name]) is True:
                lines.append("%s%s%s%s" % (self._comment, var_name, self._new_env_delim, var_value))
            else:
                lines.append("%s%s%s" % (var_name, self._new_env_delim, var_value))

        # restore schedule
        schedule_num_list = dop.query('parser', ['schedule'])
        if type(schedule_num_list) != list:
            schedule_num_list = []
        for schedule_num in schedule_num_list:
            if dop.isset('parser', ['schedule', schedule_num]) is True \
                and dop.action('parser', ['schedule', schedule_num]) == 'del':

                continue

            schedule_line = {'new_delim':self._new_delim, 'comment':self._comment}
            for filed_name in ['minute', 'hour', 'day', 'month', 'week']:
                range = dop.get('parser', ['schedule', schedule_num, filed_name, 'range'])
                interval = dop.get('parser', ['schedule', schedule_num, filed_name, 'interval'])

                if interval == '*':
                    interval = ''

                if interval != '':
                    if range == '':
                        schedule_line.update({filed_name: "%s" % interval})
                    else:
                        schedule_line.update({filed_name: "%s/%s" (range, interval)})
                else:
                    schedule_line.update({filed_name: "%s" % range})
            owner_name = dop.get( 'parser', ['schedule', schedule_num, 'owner_name']);
            command = dop.get( 'parser', ['schedule', schedule_num, 'command']);


            if system_crontab is True:
                schedule_line.update({'value': "%s%s%s" % (owner_name, self._new_delim, command)})
            else:
                schedule_line.update({'value': command})

            if dop.iscomment('parser', ['schedule', schedule_num]) is True:
                lines.append("%(comment)s%(minute)s%(new_delim)s%(hour)s%(new_delim)s%(day)s%(new_delim)s%(month)s%(new_delim)s%(week)s%(new_delim)s%(value)s" % schedule_line)
            else:
                lines.append("%(minute)s%(new_delim)s%(hour)s%(new_delim)s%(day)s%(new_delim)s%(month)s%(new_delim)s%(week)s%(new_delim)s%(value)s" % schedule_line)

        if dryrun is False:
            if len(lines) > 0:
                ConfigFile(path_name).write("\n".join(lines) + "\n")
            if len(lines) == 0:
                ConfigFile(path_name).write("")
        else:
            print "---- %s ----" % path_name
            if len(lines) > 0:
                print "\n".join(lines)
            if len(lines) == 0:
                print ""

        return retval

    def crontab_read_conf(self,extra_args=None):
        if os.path.exists(self.crontab_path) is True:
            self.dop.set(self._module, ['system', 'crontab', 'crontab'], self._read_conf(self.crontab_path))

    def crond_read_conf(self,extra_args=None):
        for path_name in self.crond_paths:
            filename = os.path.basename(path_name)
            self.dop.set(self._module, ['system', 'cron.d', filename], self._read_conf(path_name))

    def cronuser_read_conf(self,extra_args=None):
        for path_name in self.cronuser_paths:
            owner_name = os.path.basename(path_name)
            self.dop.set(self._module, ['user', owner_name], self._read_conf(path_name, owner_name))

    def read_conf(self,extra_args=None):
        self.crontab_read_conf(extra_args)
        self.crond_read_conf(extra_args)
        self.cronuser_read_conf(extra_args)

        self.dop.set(self._module,['@BASE_PARSER'],self.base_parser_name)
        #self.dop.preprint_r(self._module)
        return self.dop.getconf(self._module)

    def crontab_write_conf(self,conf_arr={},extra_args=None,dryrun=False):
        retval = True

        if self.dop.isset(self._module, ['system', 'crontab', 'crontab']) == True:
            crontab_conf = self.dop.get(self._module, ['system', 'crontab', 'crontab'])
            self._write_conf(crontab_conf, True, dryrun)
        return retval

    def crond_write_conf(self,conf_arr={},extra_args=None,dryrun=False):
        retval = True

        crond_list = self.dop.query(self._module, ['system', 'cron.d'])
        if type(crond_list) != list:
            crond_list = []
        for crond in crond_list:
            if self.dop.action(self._module, ['system', 'cron.d', crond]) != 'del':
                crontab_conf = self.dop.get(self._module, ['system', 'cron.d', crond])
                self._write_conf(crontab_conf, True, dryrun)
            else:
                path_name = self.dop.get(self._module, ['system', 'cron.d', crond, 'path_name'])
                if os.path.exists(path_name) is True:
                    pass
        return retval

    def cronuser_write_conf(self,conf_arr={},extra_args=None,dryrun=False):
        retval = True

        user_list = self.dop.query(self._module, ['user'])
        if type(user_list) != list:
            user_list = []
        for user_name in user_list:
            if self.dop.action(self._module, ['user', user_name]) != 'del':
                crontab_conf = self.dop.get(self._module, ['user', user_name])
                self._write_conf(crontab_conf, False, dryrun)
            else:
                path_name = self.dop.get(self._module, ['user', user_name, 'path_name'])
                if os.path.exists(path_name) is True:
                    pass
        return retval

    def write_conf(self,conf_arr={},extra_args=None,dryrun=False):
        retval = True

        self.dop.addconf(self._module, conf_arr)
        crontab_retval = self.crontab_write_conf(conf_arr,extra_args,dryrun)
        crond_retval = self.crond_write_conf(conf_arr,extra_args,dryrun)
        cronuser_retval = self.cronuser_write_conf(conf_arr,extra_args,dryrun)

        retval = crontab_retval and crond_retval and cronuser_retval

        return retval

"""
"""
if __name__ == '__main__':
    """Testing
    """
    parser = crontabParser()

    dop = DictOp()
    dop.addconf("crontab",parser.read_conf())
    conf = dop.getconf("crontab")

    dop.comment('crontab', ['system', 'cron.d', 'crontab', 'variable', 'SHELL'])

    edit_id = "1"
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "path_name"],"/etc/cron.d/as_mail_report");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "minute", "range"],"0");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "minute", "interval"],"*");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "hour", "range"],"12");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "hour", "interval"],"*");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "day", "range"],"13");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "day", "interval"],"*");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "month", "range"],"*");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "month", "interval"],"*");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "week", "range"],"*");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "week", "interval"],"*");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "owner_name"],"root");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "command"],"/tmp/send_mail_report hoge");

    dop.set('crontab', ["system", "cron.d", "as_mail_report", "path_name"],"/etc/cron.d/as_mail_report");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "minute", "range"],"0");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "minute", "interval"],"*");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "hour", "range"],"12");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "hour", "interval"],"*");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "day", "range"],"13");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "day", "interval"],"*");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "month", "range"],"*");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "month", "interval"],"*");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "week", "range"],"*");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "week", "interval"],"*");
    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "owner_name"],"root");

    dop.set('crontab', ["system", "cron.d", "as_mail_report", "schedule", edit_id, "command"],"/tmp/send_mail_report hoge");

    #preprint_r(conf)
    parser.write_conf(conf, dryrun=True)
