#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of Karesansui.
#
# Copyright (C) 2009-2010 HDE, Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#

import snack
import traceback
import sys
import os
import socket
import re
import time

import installer.install
from installer.trans import _, N_
from installer.const import *

class SnackUI(object):

    def __init__(self, opts):

        self.opts = opts
        self.screen = snack.SnackScreen()

        self.size   = snack._snack.size()
        self.width  = min(self.size[0]-10,100)
        self.height = self.size[1]-15

        if opts.action == FLAG_DO_ALL:
            self.wins = [
                         WelcomeWin,
                         AdminSettingWin,
                         TLSCertificatesSettingWin,
                         DatabaseSettingWin,
                         ConfirmWin,
                         FinishWin,
                        ]
        else:
            self.wins = [WelcomeWin]
            if opts.action & FLAG_CREATE_ADM:
                self.wins.append(AdminSettingWin)
            if opts.action & FLAG_CREATE_CRT:
                self.wins.append(TLSCertificatesSettingWin)
            if opts.action & FLAG_CONFIG_DB:
                self.wins.append(DatabaseSettingWin)
            if opts.action & FLAG_CONFIG_PROXY:
                self.wins.append(ProxySettingWin)
            self.wins.append(ConfirmWin)
            self.wins.append(FinishWin)


    def run(self):
        self.screen.drawRootText(0, 0, "Welcome to %s %s" % (TITLE,VERSION,))
        self.screen.drawRootText(self.size[0]-len(COPYRIGHT), 0, "%s" % (COPYRIGHT))

        page = 0
        while page < len(self.wins):
            self.page = page
            func = self.wins[page](self.screen, self, self.opts)
            ret = func()

            if ret in [OK_VALUE, YES_VALUE, F12_VALUE]:
                page = page + 1
            elif ret == BACK_VALUE:
                page = page - 1
            elif ret in [CANCEL_VALUE, NO_VALUE, EXIT_VALUE]:
                page = page + 1
                break

        self.cleanup()

        if page >= len(self.wins):
            if ret in [OK_VALUE, YES_VALUE, F12_VALUE]:
                print _("Done.")

                if self.opts.is_installed is not True:
                    from installer.utils import get_kss_url
                    print _("You can now login through following url. Your login credential is '%s'.\n%s") % (self.opts.mailaddr, " " + "\n ".join(get_kss_url()))
                if self.opts.network_modified is True:
                    print _("Notice") + ":",
                    print _("Network device setting was modified. Please restart network. \n# /etc/init.d/network restart")
        else:
            installer.install.write_log(_("Aborted page [%s].") % self.wins[page-1].__name__)
            print _("Aborted.")

        installer.install.write_log(_("Finish time: %s") % time.ctime())
        print _("See '%s' for detail.") % self.opts.logfile

        return ret

    def cleanup(self):
        if self.screen:
            self.screen.finish()

    def __del__(self):
        self.cleanup()

class ProgressWin:
    def __init__(self, screen, width):
        self.screen = screen
        self.width = width
        self.scale = snack.Scale(int(width), 100)

    def pop(self):
        self.screen.popWindow()
        self.screen.refresh()
        del self.scale
        self.scale = None

    def set(self, amount):
        self.scale.set(amount)
        self.screen.refresh()

    def update(self, amount):
        self.set(amount)

    def end(self, amount):
        self.update(amount)

    def start(self, size=None, now=None, text=None):

        if size is None:
            size = 1

        width = 65
        if (len(text) > width):
            width = min(len(text), self.width)

        self.text = snack.TextboxReflowed(width, text)

        self.screen.refresh()
        g = snack.GridForm(self.screen, _("Progress"), 1, 2)
        g.add(self.text, 0, 0, (0, 0, 0, 1), anchorLeft=1)

        self.scale = snack.Scale(width, size)
        g.add(self.scale, 0, 1)

        g.draw()
        self.screen.refresh()

def progressWin(screen, width):
    return ProgressWin(screen, width)

class BaseWin:
    def __init__(self, screen, ui, opts):
        self.ui = ui
        self.opts = opts
        self.screen = screen
        size = snack._snack.size()

class WelcomeWin(BaseWin):
    def __init__(self, *kwargs):
        BaseWin.__init__(self, *kwargs)
        self.opts.rpmsdir = getattr(self.opts, 'rpmsdir', "")

    def __call__(self):
        current_page = "[%d/%d]" % (self.ui.page+1, len(self.ui.wins))

        if self.opts.action == FLAG_CREATE_ADM:
            self.opts.dbinit = True
            welcome_msg = _("Welcome to Karesansui Database Initialization Wizard.")
        elif self.opts.action == FLAG_CREATE_CRT:
            welcome_msg = _("Welcome to Karesansui Certificate Initialization Wizard.")
        else:
            welcome_msg = _("Welcome to Karesansui Installation Program.")
        text = snack.TextboxReflowed(self.ui.width, welcome_msg)

        buttons = snack.ButtonBar(self.screen, [OK_BUTTON, CANCEL_BUTTON], compact=BUTTON_COMPACT)

        g = snack.GridFormHelp(self.screen, _("Welcome") + current_page, None, 1, 2)
        g.add(text, 0, 0, padding = (0, 0, 0, 1))
        g.add(buttons, 0, 1, growx = 1)

        button = buttons.buttonPressed(g.runOnce())

        if button is None:
            button = OK_VALUE

        return button

class AdminSettingWin(BaseWin):
    def __init__(self, *kwargs):
        BaseWin.__init__(self, *kwargs)
        self.opts.fqdn = getattr(self.opts, 'fqdn', socket.gethostname())
        self.opts.lang = getattr(self.opts, 'lang', os.environ["LANG"][0:5])
        self.opts.mailaddr = getattr(self.opts, 'mailaddr', "")
        self.opts.password1 = getattr(self.opts, 'password1', "")
        self.opts.password2 = getattr(self.opts, 'password2', "")
        if self.opts.lang == "C":
            self.opts.lang = "en_US"

    def __call__(self):
        current_page = "[%d/%d]" % (self.ui.page+1, len(self.ui.wins))

        label_hostname = snack.Label(_("Hostname"))
        hostname = snack.Entry(40, text=self.opts.fqdn, hidden=False)

        label_pass1 = snack.Label(_("Password"))
        pass1 = snack.Entry(40, text=self.opts.password1, hidden=False, password=True)
        label_pass2 = snack.Label(_("Password (again)"))
        pass2 = snack.Entry(40, text=self.opts.password2, hidden=False, password=True)
        label_addr = snack.Label(_("Mail address"))
        addr = snack.Entry(40, text=self.opts.mailaddr, hidden=False)

        sub_g = snack.Grid(2, 4)
        sub_g.setField(label_hostname, 0, 0, anchorLeft=1)
        sub_g.setField(hostname,       1, 0, anchorLeft=1, padding=(1, 0, 0, 0))

        sub_g.setField(label_pass1,    0, 1, anchorLeft=1, padding=(0, 1, 0, 0))
        sub_g.setField(pass1,          1, 1, anchorLeft=1, padding=(1, 1, 0, 0))
        sub_g.setField(label_pass2,    0, 2, anchorLeft=1, padding=(0, 0, 0, 0))
        sub_g.setField(pass2,          1, 2, anchorLeft=1, padding=(1, 0, 0, 0))

        sub_g.setField(label_addr,     0, 3, anchorLeft=1, padding=(0, 1, 0, 0))
        sub_g.setField(addr,           1, 3, anchorLeft=1, padding=(1, 1, 0, 0))

        label_lang = snack.Label(_("Language"))
        items = []
        langs = {"ja_JP":_("Japanese"), "en_US":_("English")}
        selected = None
        for name,desc in langs.iteritems():
            label = "%s - %s" % (name,desc)

            if name == self.opts.lang:
                selected = name

            if len(label) >= self.ui.width:
                label = "%s..." % label[:self.ui.width]

            items.append((label, name))

        lb = snack.Listbox(min(self.ui.height-2, len(items)), scroll=1, returnExit=1)
        for (label, key) in items:
            lb.append(label, key)
        if selected != None:
            lb.setCurrent(selected)

        lang_g = snack.Grid(2, 1)
        lang_g.setField(label_lang, 0, 0, anchorLeft=1)
        lang_g.setField(lb,         1, 0, anchorLeft=1, padding=(0, 0, 0, 0))

        buttons = snack.ButtonBar(self.screen, [OK_BUTTON, BACK_BUTTON, CANCEL_BUTTON], compact=BUTTON_COMPACT)

        g = snack.GridFormHelp(self.screen, _("Administrator setting") + current_page, None, 1, 3)
        g.add(sub_g,      0, 0, growx=1)
        g.add(lang_g,     0, 1, growx=1, padding = (0, 1, 0, 0))
        g.add(buttons,    0, 2, growx=1, padding = (0, 1, 0, 0))

        button = buttons.buttonPressed(g.runOnce())

        if button is None:
            button = OK_VALUE

        if button == OK_VALUE:
            fqdn = hostname.value()
            self.opts.fqdn = fqdn
            mailaddr = addr.value()
            self.opts.mailaddr = mailaddr
            self.opts.lang = lb.current()
            password1 = pass1.value()
            password2 = pass2.value()
            self.opts.password1 = password1
            self.opts.password2 = password2

            if fqdn == '':
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is empty") % _('Hostname'), [OK_BUTTON], self.ui.width)
                return -1

            fqdn_error_msg = ""
            if fqdn != "localhost" and not re.compile(r'.*\.').match(fqdn):
                fqdn_error_msg = _('%s must include at least one dot . character.') % (_('Hostname'),)

            if re.compile(r'(^\.|.*\.$|.*\.\.|.*-\.|.*\.-|^-|.*-$)', re.VERBOSE).match(fqdn):
                fqdn_error_msg = _('%s must be specified like %s.') % (_('Hostname'), "host.example.com",)

            invalid_fqdn_regex = "[^-a-zA-Z0-9\.]"
            m = re.compile(invalid_fqdn_regex).search(fqdn)
            if m:
                fqdn_error_msg = _('%s\nAvailable characters are %s') % (_('Hostname'),_('a-z A-Z 0-9 . -'))
            if fqdn_error_msg != "":
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), fqdn_error_msg, [OK_BUTTON], self.ui.width)
                return -1

            if mailaddr == '':
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is empty") % _('Mail address'), [OK_BUTTON], self.ui.width)
                return -1

            regex = "^(?P<localpart>[^@]+)@(?P<domainpart>.+)?$"
            m = re.compile(regex).search(mailaddr)
            if not m:
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is in an invalid format.") % _('Mail address'), [OK_BUTTON], self.ui.width)
                return -1

            if password1 == '':
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is empty") % _('Password'), [OK_BUTTON], self.ui.width)
                return -1
            if password1 != password2:
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("Passwords mismatch"), [OK_BUTTON], self.ui.width)
                return -1

        return button

class TLSCertificatesSettingWin(BaseWin):
    def __init__(self, *kwargs):
        BaseWin.__init__(self, *kwargs)
        self.opts.fqdn = getattr(self.opts, 'fqdn', socket.gethostname())
        self.opts.lang = getattr(self.opts, 'lang', os.environ["LANG"][0:5])

        #import pdb; pdb.set_trace()
        default_certinit = 0
        infos = {}
        if os.path.exists(PKI_CLIENT_INFO):
            from installer.utils import sh_config_read
            infos = sh_config_read(PKI_CLIENT_INFO," = ")
        else:
            default_certinit = 1

        default_infos = {}
        for opt_name in ["country","state","locality","organization","cn"]:
            try:
                default_infos[opt_name] = infos[opt_name]
            except:
                default_infos[opt_name] = ""
                if opt_name == "country":
                    default_infos[opt_name] = "JP"
                if opt_name == "organization":
                    default_infos[opt_name] = "Karesansui Project"

        for opt_name in ["country","state","locality","organization","cn"]:
            exec("self.opts.%s = getattr(self.opts, '%s', '%s')" % (opt_name, opt_name, default_infos[opt_name]))

        self.opts.certinit = getattr(self.opts, 'certinit', default_certinit)

    def __call__(self):
        current_page = "[%d/%d]" % (self.ui.page+1, len(self.ui.wins))
        width = 25

        label_country = snack.Label(_("Country Name (ISO 2 letter code)"))
        country = snack.Entry(3, text=self.opts.country, hidden=False)

        label_state = snack.Label(_("State or Province Name (full name)"))
        state = snack.Entry(width, text=self.opts.state, hidden=False)

        label_locality = snack.Label(_("Locality Name (e.g. city)"))
        locality = snack.Entry(width, text=self.opts.locality, hidden=False)

        label_organization = snack.Label(_("Organization Name (eg, company)"))
        organization = snack.Entry(width, text=self.opts.organization, hidden=False)

        label_cn = snack.Label(_("Common Name (fully qualified domain name)"))
        cn = snack.Entry(width, text=self.opts.fqdn, hidden=False)

        cb = snack.Checkbox(_("Initialize SSL certificates settings?"), self.opts.certinit)

        sub_g = snack.Grid(2, 5)
        sub_g.setField(label_country,  0, 0, anchorLeft=1)
        sub_g.setField(country,        1, 0, anchorLeft=1, padding=(1, 0, 0, 0))
        sub_g.setField(label_state,    0, 1, anchorLeft=1, padding=(0, 0, 0, 0))
        sub_g.setField(state,          1, 1, anchorLeft=1, padding=(1, 0, 0, 0))
        sub_g.setField(label_locality, 0, 2, anchorLeft=1, padding=(0, 0, 0, 0))
        sub_g.setField(locality,       1, 2, anchorLeft=1, padding=(1, 0, 0, 0))
        sub_g.setField(label_organization, 0, 3, anchorLeft=1, padding=(0, 0, 0, 0))
        sub_g.setField(organization,   1, 3, anchorLeft=1, padding=(1, 0, 0, 0))
        sub_g.setField(label_cn,       0, 4, anchorLeft=1, padding=(0, 1, 0, 0))
        sub_g.setField(cn,             1, 4, anchorLeft=1, padding=(1, 1, 0, 0))

        buttons = snack.ButtonBar(self.screen, [OK_BUTTON, BACK_BUTTON, CANCEL_BUTTON], compact=BUTTON_COMPACT)

        g = snack.GridFormHelp(self.screen, _("TLS certificates setting") + current_page, None, 1, 3)
        g.add(sub_g,      0, 0, growx=1)
        g.add(cb,         0, 1, growx=1, padding = (0, 1, 0, 0))
        g.add(buttons,    0, 2, growx=1, padding = (0, 1, 0, 0))

        button = buttons.buttonPressed(g.runOnce())

        if button is None:
            button = OK_VALUE

        if button is OK_VALUE:

            """ Check country code """
            input_country = country.value()
            self.opts.country = input_country

            if input_country == '':
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is empty") % _('Country Name (ISO 2 letter code)'), [OK_BUTTON], self.ui.width)
                return -1

            regex = "^[A-Z]{2}$"
            m = re.compile(regex).search(input_country)
            if not m:
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is in an invalid format.") % _('Country Name (ISO 2 letter code)'), [OK_BUTTON], self.ui.width)
                return -1

            """ Check state name """
            input_state = state.value()
            self.opts.state = input_state

            if input_state == '':
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is empty") % _('State or Province Name (full name)'), [OK_BUTTON], self.ui.width)
                return -1

            regex = "^[-a-zA-Z0-9 ]{1,128}$"
            m = re.compile(regex).search(input_state)
            if not m:
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is in an invalid format.") % _('State or Province Name (full name)'), [OK_BUTTON], self.ui.width)
                return -1

            """ Check locality name """
            input_locality = locality.value()
            self.opts.locality = input_locality

            if input_locality == '':
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is empty") % _('Locality Name (e.g. city)'), [OK_BUTTON], self.ui.width)
                return -1

            regex = "^[-a-zA-Z0-9 ]{1,128}$"
            m = re.compile(regex).search(input_locality)
            if not m:
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is in an invalid format.") % _('Locality Name (e.g. city)'), [OK_BUTTON], self.ui.width)
                return -1

            """ Check organization name """
            input_organization = organization.value()
            self.opts.organization = input_organization

            if input_organization == '':
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is empty") % _('Organization Name (eg, company)'), [OK_BUTTON], self.ui.width)
                return -1

            regex = "^[-a-zA-Z0-9 .,\(\)]{1,64}$"
            m = re.compile(regex).search(input_locality)
            if not m:
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is in an invalid format.") % _('Organization Name (eg, company)'), [OK_BUTTON], self.ui.width)
                return -1

            """ Check common name """
            input_cn = cn.value()
            self.opts.cn = input_cn

            if input_cn == '':
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is empty") % _('Common Name (fully qualified domain name)'), [OK_BUTTON], self.ui.width)
                return -1

            input_cn_error_msg = ""
            if input_cn != "localhost" and not re.compile(r'.*\.').match(input_cn):
                input_cn_error_msg = _('%s must include at least one dot . character.') % (_('Common Name (fully qualified domain name)'),)

            if re.compile(r'(^\.|.*\.$|.*\.\.|.*-\.|.*\.-|^-|.*-$)', re.VERBOSE).match(input_cn):
                input_cn_error_msg = _('%s must be specified like %s.') % (_('Common Name (fully qualified domain name)'), "host.example.com",)

            invalid_input_cn_regex = "[^-a-zA-Z0-9\.]"
            m = re.compile(invalid_input_cn_regex).search(input_cn)
            if m:
                input_cn_error_msg = _('%s\nAvailable characters are %s') % (_('Common Name (fully qualified domain name)'),_('a-z A-Z 0-9 . -'))
            if input_cn_error_msg != "":
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), input_cn_error_msg, [OK_BUTTON], self.ui.width)
                return -1

            self.opts.certinit = cb.selected()

        return button

class DatabaseSettingWin(BaseWin):
    def __init__(self, *kwargs):
        BaseWin.__init__(self, *kwargs)
        self.opts.dbbind = getattr(self.opts, 'dbbind', DEFAULT_KARESANSUI_DB_URI)
        self.opts.pysilhouette_dbbind = getattr(self.opts, 'pysilhouette_dbbind', DEFAULT_PYSILHOUETTE_DB_URI)
        default_dbinit = 0
        if self.opts.is_installed is not True:
            default_dbinit = 1
        self.opts.dbinit = getattr(self.opts, 'dbinit', default_dbinit)

    def __call__(self):
        current_page = "[%d/%d]" % (self.ui.page+1, len(self.ui.wins))

        label_dbbind = snack.Label(_("Database path"))
        dbbind = snack.Entry(50, text=self.opts.dbbind, hidden=False)

        label_pysilhouette_dbbind = snack.Label(_("Database pysilhouette path"))
        pysilhouette_dbbind = snack.Entry(50, text=self.opts.pysilhouette_dbbind, hidden=False)

        label_example = snack.Label(_("Example"))
        text = snack.TextboxReflowed(self.ui.width - 20,
                 "%s\nmysql://username:password@hostname/karesansui?charset=utf8\npostgres://username:password@hostname:port/database" % DEFAULT_KARESANSUI_DB_URI)

        sub_g = snack.Grid(2, 3)
        sub_g.setField(label_dbbind,              0, 0, anchorLeft=1, padding=(0, 1, 0, 0))
        sub_g.setField(dbbind,                    1, 0, anchorLeft=1, padding=(1, 1, 0, 0))
        sub_g.setField(label_pysilhouette_dbbind, 0, 1, anchorLeft=1, padding=(0, 1, 0, 0))
        sub_g.setField(pysilhouette_dbbind,       1, 1, anchorLeft=1, padding=(1, 1, 0, 0))
        sub_g.setField(label_example,             0, 2, anchorLeft=1, padding=(0, 1, 0, 0))
        sub_g.setField(text,                      1, 2, anchorLeft=1, padding=(1, 1, 0, 0))

        #cb = snack.Checkbox(_("Initialize database?"))
        cb = snack.Checkbox(_("Initialize database?"), self.opts.dbinit)

        buttons = snack.ButtonBar(self.screen, [OK_BUTTON, BACK_BUTTON, CANCEL_BUTTON], compact=BUTTON_COMPACT)

        g = snack.GridFormHelp(self.screen, _("Database setting") + current_page, None, 1, 3)
        g.add(sub_g,   0, 0)
        cnt = 1
        if self.opts.action != FLAG_CREATE_ADM:
            g.add(cb,      0, cnt, padding=(0, 1, 0, 0))
            cnt = cnt + 1
        g.add(buttons, 0, cnt, padding=(0, 1, 0, 0), growx=1)

        button = buttons.buttonPressed(g.runOnce())

        if button is None:
            button = OK_VALUE

        if button == OK_VALUE:
            self.opts.dbbind = dbbind.value()
            self.opts.pysilhouette_dbbind = pysilhouette_dbbind.value()
            if self.opts.action == FLAG_CREATE_ADM:
                self.opts.dbinit = True
            else:
                self.opts.dbinit = cb.selected()

            if self.opts.dbbind == '':
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is empty") % _("Database path"), [OK_BUTTON], self.ui.width)
                return -1

            if self.opts.pysilhouette_dbbind == '':
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is empty") % _("Database pysilhouette path"), [OK_BUTTON], self.ui.width)
                return -1

            check = False
            for k,v in DATABASE_MODULES.iteritems():
                regex = "^%s://" % k
                if re.compile(regex).match(self.opts.dbbind):
                    check = True

            pysilhouette_check = False
            for k,v in DATABASE_MODULES.iteritems():
                regex = "^%s://" % k
                if re.compile(regex).match(self.opts.pysilhouette_dbbind):
                    pysilhouette_check = True

            #import pdb; pdb.set_trace()
            try:
                for y in [x.strip() for x in DEFAULT_PYTHONPATH.split(':') if x]:
                    if (y in sys.path) is False: sys.path.insert(0, y)
                from sqlalchemy import create_engine
                from sqlalchemy.exc import OperationalError

                if self.opts.dbinit != 0:
                    from installer.database import is_connect, DatabaseScriptError
                    try:
                        is_connect(self.opts.dbbind)
                    except DatabaseScriptError, e:
                        ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is not connectable") % _("Database path"), [OK_BUTTON], self.ui.width)
                        return -1

                    try:
                        is_connect(self.opts.pysilhouette_dbbind)
                    except DatabaseScriptError, e:
                        ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is not connectable") % _("Database pysilhouette path"), [OK_BUTTON], self.ui.width)
                        return -1
            except:
                pass

            if check is False:
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is in an invalid format.") % _("Database path"), [OK_BUTTON], self.ui.width)
                return -1

            if pysilhouette_check is False:
                ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is in an invalid format.") % _("Database pysilhouette path"), [OK_BUTTON], self.ui.width)
                return -1

        return button


class ProxySettingWin(BaseWin):
    def __init__(self, *kwargs):
        BaseWin.__init__(self, *kwargs)

        use_proxy = False
        proxy_host = ""
        proxy_port = ""
        yum_config = "/etc/yum.conf"
        if os.path.exists(yum_config):
            from installer.utils import sh_config_read
            try:
                proxy = sh_config_read(yum_config,"=")['proxy']
                if proxy != "":
                    proxy = re.sub(r"""^(http|ftp)s?://([^/]+).*""","\\2",proxy)
                    proxy_host = proxy.split(":")[0]
                    try:
                        proxy_port = proxy.split(":")[1]
                    except:
                        proxy_port = "80"
                    use_proxy = True
            except:
                pass

            try:
                proxy_user = sh_config_read(yum_config,"=")['proxy_username']
            except:
                proxy_user = ""

            try:
                proxy_pass = sh_config_read(yum_config,"=")['proxy_password']
            except:
                proxy_pass = ""

        self.opts.use_proxy = getattr(self.opts, 'use_proxy', use_proxy)
        self.opts.proxy_host = getattr(self.opts, 'proxy_host', proxy_host)
        self.opts.proxy_port = getattr(self.opts, 'proxy_port', proxy_port)
        self.opts.proxy_user = getattr(self.opts, 'proxy_user', proxy_user)
        self.opts.proxy_pass1 = getattr(self.opts, 'proxy_pass1', proxy_pass)
        self.opts.proxy_pass2 = getattr(self.opts, 'proxy_pass2', proxy_pass)

    def __call__(self):
        current_page = "[%d/%d]" % (self.ui.page+1, len(self.ui.wins))

        label_use_proxy = snack.Label(_("Use Proxy"))
        cb = snack.Checkbox(_("Yes"), self.opts.use_proxy)

        label_hostname = snack.Label(_("Hostname"))
        hostname = snack.Entry(20, text=self.opts.proxy_host, hidden=False)

        label_port = snack.Label(_("Port Number"))
        port = snack.Entry(6, text=self.opts.proxy_port, hidden=False)

        label_username = snack.Label(_("Username"))
        username = snack.Entry(20, text=self.opts.proxy_user, hidden=False, password=True)
        label_pass1 = snack.Label(_("Password"))
        pass1 = snack.Entry(20, text=self.opts.proxy_pass1, hidden=False, password=True)
        label_pass2 = snack.Label(_("Password (again)"))
        pass2 = snack.Entry(20, text=self.opts.proxy_pass2, hidden=False, password=True)

        sub_g = snack.Grid(2, 6)
        sub_g.setField(label_use_proxy,0, 0, anchorLeft=1, padding=(0, 0, 0, 0))
        sub_g.setField(cb,             1, 0, anchorLeft=1, padding=(1, 0, 0, 0))
        sub_g.setField(label_hostname, 0, 1, anchorLeft=1, padding=(0, 1, 0, 0))
        sub_g.setField(hostname,       1, 1, anchorLeft=1, padding=(1, 1, 0, 0))
        sub_g.setField(label_port,     0, 2, anchorLeft=1, padding=(0, 0, 0, 0))
        sub_g.setField(port,           1, 2, anchorLeft=1, padding=(1, 0, 0, 0))

        sub_g.setField(label_username, 0, 3, anchorLeft=1, padding=(0, 1, 0, 0))
        sub_g.setField(username,       1, 3, anchorLeft=1, padding=(1, 1, 0, 0))
        sub_g.setField(label_pass1,    0, 4, anchorLeft=1, padding=(0, 0, 0, 0))
        sub_g.setField(pass1,          1, 4, anchorLeft=1, padding=(1, 0, 0, 0))
        sub_g.setField(label_pass2,    0, 5, anchorLeft=1, padding=(0, 0, 0, 0))
        sub_g.setField(pass2,          1, 5, anchorLeft=1, padding=(1, 0, 0, 0))

        buttons = snack.ButtonBar(self.screen, [OK_BUTTON, BACK_BUTTON, CANCEL_BUTTON], compact=BUTTON_COMPACT)

        g = snack.GridFormHelp(self.screen, _("Proxy setting") + current_page, None, 1, 2)
        g.add(sub_g,      0, 0, growx=1, padding = (0, 1, 0, 0))
        g.add(buttons,    0, 1, growx=1, padding = (0, 1, 0, 0))

        button = buttons.buttonPressed(g.runOnce())

        if button is None:
            button = OK_VALUE

        if button == OK_VALUE:
            self.opts.use_proxy = cb.selected()
            if self.opts.use_proxy:
                self.opts.proxy_host = hostname.value()
                self.opts.proxy_port = port.value()

                if self.opts.proxy_host == '':
                    ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is empty") % _('Hostname'), [OK_BUTTON], self.ui.width)
                    return -1

                fqdn_error_msg = ""
                if self.opts.proxy_host != "localhost" and not re.compile(r'.*\.').match(self.opts.proxy_host):
                    fqdn_error_msg = _('%s must include at least one dot . character.') % (_('Hostname'),)

                if re.compile(r'(^\.|.*\.$|.*\.\.|.*-\.|.*\.-|^-|.*-$)', re.VERBOSE).match(self.opts.proxy_host):
                    fqdn_error_msg = _('%s must be specified like %s.') % (_('Hostname'), "host.example.com",)

                invalid_fqdn_regex = "[^-a-zA-Z0-9\.]"
                m = re.compile(invalid_fqdn_regex).search(self.opts.proxy_host)
                if m:
                    fqdn_error_msg = _('%s\nAvailable characters are %s') % (_('Hostname'),_('a-z A-Z 0-9 . -'))
                if fqdn_error_msg != "":
                    ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), fqdn_error_msg, [OK_BUTTON], self.ui.width)
                    return -1

                if self.opts.proxy_port == '':
                    ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), _("%s is empty") % _('Port Number'), [OK_BUTTON], self.ui.width)
                    return -1

                port_error_msg = ""
                invalid_port_regex = "[^0-9]"
                m = re.compile(invalid_port_regex).search(self.opts.proxy_port)
                if m:
                    port_error_msg = _('%s\nAvailable characters are %s') % (_('Port Number'),_('0-9'))
                else:
                    if int(self.opts.proxy_port) < 1 or int(self.opts.proxy_port) > 65535:
                        port_error_msg = _('%s\nOut of range %d-%d') % (_('Port Number'),1,65535)

                if port_error_msg != "":
                    ret = snack.ButtonChoiceWindow(self.screen, _("ERROR"), port_error_msg, [OK_BUTTON], self.ui.width)
                    return -1

            self.opts.proxy_user = username.value()
            self.opts.proxy_pass1 = pass1.value()
            self.opts.proxy_pass2 = pass2.value()

        return button








class ConfirmWin(BaseWin):
    def __init__(self, *kwargs):
        BaseWin.__init__(self, *kwargs)

    def __call__(self):
        current_page = "[%d/%d]" % (self.ui.page+1, len(self.ui.wins))

        if self.opts.action == FLAG_CREATE_ADM:
            confirm_msg = _("Are you sure to proceed to the initialization of database?")
        elif self.opts.action == FLAG_CREATE_CRT:
            confirm_msg = _("Are you sure to proceed to the initialization of certificates?")
        else:
            if self.opts.action == FLAG_DO_ALL:
                confirm_msg = _("Are you sure to proceed to install?")
            else:
                confirm_msg = _("Are you sure to proceed?")
                if self.opts.mode is not None:
                    confirm_msg += "\n[expected processes are"
                    for mode in self.opts.mode.split(":"):
                        confirm_msg += " " + mode[5:]
                    confirm_msg += "]"

        ret = snack.ButtonChoiceWindow(self.screen, _("Confirm") + current_page, confirm_msg,
                [OK_BUTTON, BACK_BUTTON, CANCEL_BUTTON], self.ui.width)

        if ret is None:
            ret = OK_VALUE

        if ret in [OK_VALUE, F12_VALUE]:

            dargs = {}
            dargs["progress_callback"] = progressWin(self.screen, self.ui.width)

            #import pdb; pdb.set_trace()
            try:
                installer.install.run(self.opts, **dargs)
                ret = OK_VALUE
            except installer.install.InstallError, e:
                error_msg = _("An error occurred while setting up the installer.\n%s" % e.value)
                installer.install.write_log(error_msg)
                r = snack.ButtonChoiceWindow(self.screen,_("ERROR"), error_msg,
                        [EXIT_BUTTON], self.ui.width)
                ret = EXIT_VALUE

        return ret


class FinishWin(BaseWin):
    def __init__(self, *kwargs):
        BaseWin.__init__(self, *kwargs)
        self.opts.config = getattr(self.opts, 'config', DEFAULT_KARESANSUI_CONF)
        self.opts.network_modified = getattr(self.opts, 'network_modified', False)

    def __call__(self):
        current_page = "[%d/%d]" % (self.ui.page+1, len(self.ui.wins))

        if self.opts.action == FLAG_CREATE_ADM:
            confirm_msg = _("Congratulations, the initialization of database was finished.")
        elif self.opts.action == FLAG_CREATE_CRT:
            confirm_msg = _("Congratulations, the initialization of certificates was finished.")
        else:
            if self.opts.action == FLAG_DO_ALL:
                confirm_msg = _("Congratulations, installation was finished.")
            else:
                confirm_msg = _("Congratulations, all the process have been done.")
        confirm_msg += "\n" + _("Please modify config file %s as needed.") % self.opts.config
        confirm_msg += "\n" + _("You can check the environment for Karesansui by executing the following script:\n# %s") % os.path.dirname(sys.argv[0]) + "/karesansui-checkenv"

        if self.opts.network_modified is True:
            confirm_msg += "\n\n" + _("Network device setting was modified. Please restart network. \n# /etc/init.d/network restart")

        return snack.ButtonChoiceWindow(self.screen,_("Finish") + current_page, confirm_msg,
                [OK_BUTTON], self.ui.width - 10)

