#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# This file is part of Karesansui.
#
# Copyright (C) 2010 HDE, Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#

import os
import sys
import logging
from optparse import OptionParser

from ksscommand import KssCommand, KssCommandException, KssCommandOptException

import __cmd__

try:
    import karesansui
    from karesansui import __version__
    from karesansui.lib.virt.virt import KaresansuiVirtConnection
    from karesansui.lib.utils import load_locale
    from karesansui.lib.const import STORAGE_POOL_TYPE, STORAGE_VOLUME_FORMAT, STORAGE_VOLUME_SIZE_MIN_LENGTH
    from karesansui.lib.storage import get_storage_volume_max_size_by_unit
except ImportError:
    print >>sys.stderr, "[Error] karesansui package was not found."
    sys.exit(1)

_ = load_locale()

usage = '%prog [options]'

def getopts():
    optp = OptionParser(usage=usage, version=__version__)
    optp.add_option('-n', '--name', dest='name', help=_('Volume Name'))
    optp.add_option('-p', '--pool_name', dest='pool_name', help=_('Pool Name'))
    optp.add_option('-f', '--format', dest='format', help=_('Format'))
    optp.add_option('-a', '--allocation', dest='allocation',
                    type="int", help=_('Allocation'), default=0)
    optp.add_option('-c', '--capacity', dest='capacity', type="int",
                    help=_('Capacity'), default=0)
    optp.add_option('-u', '--unit', dest='unit', help=_('Volume Size Unit'))
    optp.add_option('-o', '--permission_owner', dest='permission_owner', help=_('Permission Owner'))
    optp.add_option('-g', '--permission_group', dest='permission_group', help=_('Permission Group'))
    optp.add_option('-m', '--permission_mode', dest='permission_mode', help=_('Permission Mode'))

    optp.add_option('-U', '--use', dest='use',
                    help=_('Use of disk usage."guestos" or "adddisk"'))
    return optp.parse_args()

def chkopts(opts):
    if not opts.name:
        raise KssCommandOptException('ERROR: %s option is required.' % '-n or --name')
    if not opts.pool_name:
        raise KssCommandOptException('ERROR: %s option is required.' % '-p or --pool_name')
    if not opts.format:
        raise KssCommandOptException('ERROR: %s option is required.' % '-f or format')
    if not opts.format in STORAGE_VOLUME_FORMAT.values():
        raise KssCommandOptException('ERROR: Format is not available. '
                                     'raw or qcow2... is available. format=%s' % opts.format)

    storage_volume_max_size = get_storage_volume_max_size_by_unit(opts.pool_name, opts.unit)
    if opts.allocation < STORAGE_VOLUME_SIZE_MIN_LENGTH or storage_volume_max_size < opts.allocation:
        raise KssCommandOptException('ERROR: Allocation "%s" is out of available range.' % opts.allocation)
    if opts.capacity < STORAGE_VOLUME_SIZE_MIN_LENGTH or storage_volume_max_size < opts.capacity:
        raise KssCommandOptException('ERROR: Capacity "%s" is out of available range.' % opts.capacity)

    # TODO permission('s) use
    pass

class CreateStorageVolume(KssCommand):

    def process(self):
        (opts, args) = getopts()
        chkopts(opts)

        self.up_progress(10)
        conn = KaresansuiVirtConnection(readonly=False)

        try:
            try:
                inactive_storage_pools = conn.list_inactive_storage_pool()
                active_storage_pools = conn.list_active_storage_pool()
                if not (opts.pool_name in active_storage_pools or \
                        opts.pool_name in inactive_storage_pools):
                    raise KssCommandException('Specified storage pool does not exist. - pool=%s'
                                              % opts.pool_name)

                if conn.get_storage_volume(opts.pool_name, opts.name) is not None:
                    raise KssCommandException(
                        'We already have a storage volume. - pool=%s, vol=%s'
                        % (opts.pool_name, opts.name))

                if conn.create_storage_volume(opts.name, opts.pool_name,
                                              opts.format,
                                              use=opts.use,
                                              capacity=opts.capacity, allocation=opts.allocation,
                                              c_unit=opts.unit,
                                              t_p_owner=opts.permission_owner,
                                              t_p_group=opts.permission_group,
                                              t_p_mode=opts.permission_mode,
                                           ) is False:
                    raise KssCommandException('Failed to create storage volume. (libvirt) - pool=%s, vol=%s'
                                              % (opts.pool_name, opts.name))

                self.up_progress(40)

                # TODO symlink check.
                """
                vol_path = conn.get_storage_volume_path(opts.pool_name, opts.name)
                if vol_path is None:
                    raise KssCommandException(
                        'Could not get the normal storage pool or storage volume. - pool=%s, vol=%s' \
                        % (opts.pool, opts.name))

                if os.path.isfile(vol_path) is False:
                    raise KssCommandException(
                        'File does not exist in the path of a storage volume. - pool=%s, vol=%s' \
                        % (opts.pool, opts.name))

                """
                self.logger.info('Created storage volume. - vol=%s' % (opts.name))
                print >>sys.stdout, _('Created storage volume. - vol=%s') % (opts.name)
                return True
            except KssCommandException, e:
                raise e
        finally:
            conn.close()

if __name__ == "__main__":
    target = CreateStorageVolume()
    sys.exit(target.run())
