#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
# This file is part of Karesansui Core.
#
# Copyright (C) 2010 HDE, Inc.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.
#

import os
import re
import sys
import types

from karesansui.lib.conf import read_conf, write_conf
from karesansui.lib.utils import uniq_sort
from karesansui.lib.utils import preprint_r

MODULE = "collectd"

DictOp = None

FilterName    = "filter"
ThresholdName = "threshold"
PythonName    = "python"

def _get_collectd_config(webobj=None, host=None):
    modules = ["collectd","collectdplugin"]

    dop = read_conf(modules, webobj, host)
    if dop is False:
        return False

    return dop

DictOp = _get_collectd_config()

def plugin_list(webobj=None, host=None, dop=None):
    global DictOp
    retval = []

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    try:
        load_plugins = dop.query("collectd",["LoadPlugin"])
        if load_plugins is False:
            load_plugins = []
    except:
        load_plugins = []

    try:
        plugins = dop.getconf("collectdplugin")
        for _k,_v in plugins.iteritems():
            _load_plugins = dop.query("collectdplugin",[_k,"LoadPlugin"])
            if type(_load_plugins) is list or (_load_plugins is not False and len(_load_plugins) > 0):
                load_plugins = load_plugins + _load_plugins
        del plugins
    except:
        pass

    retval = uniq_sort(load_plugins)
    return retval

def active_plugin_list(webobj=None, host=None, dop=None):
    global DictOp
    retval = []

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    list = plugin_list(dop, webobj, host)
    #preprint_r(list)
    for plugin_name in list:
        iscomment = True
        if dop.isset("collectd",["LoadPlugin",plugin_name]) is True:
            iscomment = dop.iscomment("collectd",["LoadPlugin",plugin_name])

        if iscomment is True:
            plugins = dop.getconf("collectdplugin")
            for _k,_v in plugins.iteritems():
                if dop.isset("collectdplugin",[_k,"LoadPlugin",plugin_name]) is True:
                    iscomment = dop.iscomment("collectdplugin",[_k,"LoadPlugin",plugin_name])
                    break
            del plugins
        if iscomment is False:
            retval.append(plugin_name)

    return retval

def inactive_plugin_list(dop=None, webobj=None, host=None):
    global DictOp
    retval = []

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    list = plugin_list(dop, webobj, host)
    active = active_plugin_list(dop, webobj, host)
    for plugin_name in list:
        if not plugin_name in active:
            retval.append(plugin_name)

    return retval

def is_enabled_plugin(plugin_name, dop=None, webobj=None, host=None):
    global DictOp
    retval = False

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    active = active_plugin_list(dop, webobj, host)
    if plugin_name in active:
        retval = True

    return retval

def enabled_plugin(plugin_name, dop=None, webobj=None, host=None):
    global DictOp
    retval = False

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    active = active_plugin_list(dop, webobj, host)
    if not plugin_name in active:
        if dop.cdp_isset("collectd",["LoadPlugin",plugin_name]) is True:
            retval = dop.cdp_uncomment("collectd",["LoadPlugin",plugin_name])
        else:
            plugins = dop.getconf("collectdplugin")
            for _k,_v in plugins.iteritems():
                if dop.cdp_isset("collectdplugin",[_k,"LoadPlugin",plugin_name]) is True:
                    retval = dop.cdp_uncomment("collectdplugin",[_k,"LoadPlugin",plugin_name])
                    break
            del plugins

    return retval

def disabled_plugin(plugin_name, dop=None, webobj=None, host=None):
    global DictOp
    retval = False

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    active = active_plugin_list(dop, webobj, host)
    if plugin_name in active:
        if dop.cdp_isset("collectd",["LoadPlugin",plugin_name]) is True:
            retval = dop.cdp_comment("collectd",["LoadPlugin",plugin_name])
        else:
            plugins = dop.getconf("collectdplugin")
            for _k,_v in plugins.iteritems():
                if dop.cdp_isset("collectdplugin",[_k,"LoadPlugin",plugin_name]) is True:
                    retval = dop.cdp_comment("collectdplugin",[_k,"LoadPlugin",plugin_name])
                    break
            del plugins

    return retval


def get_global_parameter(name, dop=None, webobj=None, host=None):
    global DictOp

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    retval = dop.cdp_get("collectd",[name])

    if retval is False:
        retval = dop.get("collectd",[name])

    return retval

def set_global_parameter(name, value, dop=None, webobj=None, host=None, is_cdp=True):
    global DictOp

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    if is_cdp is True:
        return dop.cdp_set("collectd",[name],value)
    else:
        return dop.set("collectd",[name],value)

def where_is_plugin(plugin_name, dop=None, webobj=None, host=None):
    global DictOp
    retval = False

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    keyword = "LoadPlugin"
    keyword = "Plugin"

    if dop.cdp_isset("collectd",[keyword,plugin_name]) is True:
        retval = "@global"

    plugins = dop.getconf("collectdplugin")
    for _k,_v in plugins.iteritems():
        if dop.cdp_isset("collectdplugin",[_k,keyword,plugin_name]) is True:
            retval = _k
            break
    del plugins

    return retval

def switch_python_plugin(flag=True, dop=None, webobj=None, host=None):
    global DictOp
    retval = False

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)


    dop.cdp_set("collectdplugin",[PythonName,"LoadPlugin","python","Globals"],"true",multiple_file=True)

    _keys = [PythonName,"Plugin","python"]

    keys = _keys + ["ModulePath"]
    value = "\"/opt/karesansui/lib/python/karesansui/lib/collectd\""
    dop.cdp_set("collectdplugin",keys,value,multiple_file=True)

    keys = _keys + ["Encoding"]
    value = "utf-8"
    dop.cdp_set("collectdplugin",keys,value,multiple_file=True)

    keys = _keys + ["LogTraces"]
    value = "true"
    dop.cdp_set("collectdplugin",keys,value,multiple_file=True)

    keys = _keys + ["Import"]
    value = "\"notification\""
    dop.cdp_set("collectdplugin",keys,value,multiple_file=True)

    keys = _keys + ["Module","notification","CountupDBPath"]
    value = "\"/var/opt/karesansui/notify_count.db\""
    dop.cdp_set("collectdplugin",keys,value,multiple_file=True)

    keys = _keys + ["Module","notification","Logfile"]
    value = "\"/var/log/hde-collectd/notification.log\""
    dop.cdp_set("collectdplugin",keys,value,multiple_file=True)

    if flag is True:
        dop.cdp_uncomment("collectdplugin",[PythonName,"LoadPlugin","python"],recursive=True,multiple_file=True)
        dop.cdp_uncomment("collectdplugin",_keys,recursive=True,multiple_file=True)
    else:
        dop.cdp_comment("collectdplugin",[PythonName,"LoadPlugin","python"],recursive=True,multiple_file=True)
        dop.cdp_comment("collectdplugin",_keys,recursive=True,multiple_file=True)

def enable_python_plugin(dop=None, webobj=None, host=None):
    global DictOp
    retval = False

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    switch_python_plugin(flag=True, dop=dop, webobj=webobj, host=host)


def disable_python_plugin(dop=None, webobj=None, host=None):
    global DictOp
    retval = False

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    switch_python_plugin(flag=False, dop=dop, webobj=webobj, host=host)


def init_filter(dop=None, webobj=None, host=None):
    global DictOp
    retval = False

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    load_plugins = ["match_regex","match_value","target_notification"]

    if dop.cdp_isset("collectdplugin",[FilterName,"@ORDERS"],multiple_file=True) is True:
        orders = dop.cdp_get("collectdplugin",[FilterName,"@ORDERS"],multiple_file=True)
    else:
        orders = []

    for plugin_name in load_plugins:
        if dop.cdp_isset("collectdplugin",[FilterName,"LoadPlugin",plugin_name],multiple_file=True) is False:
            dop.cdp_set("collectdplugin",[FilterName,"LoadPlugin",plugin_name],plugin_name,multiple_file=True)
            orders.append(["LoadPlugin",plugin_name])

    dop.cdp_set("collectdplugin",[FilterName,"@ORDERS"],orders,multiple_file=True)

def set_chain_rule(type,chain,rule,params,dop=None, webobj=None, host=None):
    global DictOp
    retval = False

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    init_filter(dop, webobj, host)

    _keys = [FilterName,type,"\"%s\"" % chain]

    if dop.cdp_isset("collectdplugin", _keys, multiple_file=True) is False:
        dop.cdp_set("collectdplugin", _keys, chain, multiple_file=True, is_opt_multi=True)
        dop.cdp_set_pre_comment("collectdplugin", _keys, [''], multiple_file=True)

    _keys = [FilterName,"Chain",chain]

    try:
        plugin = "\"^%s$\"" % params["Plugin"]
        keys = _keys + ["Rule",rule,"Match","regex","Plugin"]
        dop.cdp_set("collectdplugin", keys, plugin, multiple_file=True)
    except:
        pass

    try:
        type_instance = "\"^%s$\"" % params["TypeInstance"]
        keys = _keys + ["Rule",rule,"Match","regex","TypeInstance"]
        dop.cdp_set("collectdplugin", keys, type_instance, multiple_file=True)
    except:
        pass

    try:
        min = params["Min"]
        keys = _keys + ["Rule",rule,"Match","value","Min"]
        dop.cdp_set("collectdplugin", keys, min, multiple_file=True)
    except:
        pass

    try:
        max = params["Max"]
        keys = _keys + ["Rule",rule,"Match","value","Max"]
        dop.cdp_set("collectdplugin", keys, max, multiple_file=True)
    except:
        pass

    try:
        invert = params["Invert"]
        keys = _keys + ["Rule",rule,"Match","value","Invert"]
        dop.cdp_set("collectdplugin", keys, invert, multiple_file=True)
    except:
        pass

    try:
        satisfy = "\"%s\"" % params["Satisfy"]
        keys = _keys + ["Rule",rule,"Match","value","Satisfy"]
        dop.cdp_set("collectdplugin", keys, satisfy, multiple_file=True)
    except:
        pass

    try:
        if params['Target'] == "notification":
            try:
                message = "\"%s\"" % params["Message"]
                keys = _keys + ["Rule",rule,"Target","notification","Message"]
                dop.cdp_set("collectdplugin", keys, message,multiple_file=True)
            except:
                pass

            try:
                severity = "\"%s\"" % params["Severity"]
                keys = _keys + ["Rule",rule,"Target","notification","Severity"]
                dop.cdp_set("collectdplugin", keys, severity,multiple_file=True)
            except:
                pass

        else:
            try:
                keys = _keys + ["Rule",rule,"Target",params['Target'],"Pass"]
                dop.cdp_set("collectdplugin", keys, "" ,multiple_file=True)
                dop.cdp_comment("collectdplugin", keys, multiple_file=True)
            except:
                pass
    except:
        pass

    #keys = _keys + ["Target"]
    #dop.cdp_set("collectdplugin", keys, "\"write\"", multiple_file=True)

def set_pre_cache_chain_rule(chain,rule,params,dop=None,webobj=None,host=None):
    global DictOp

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    set_chain_rule("PreCacheChain",chain,rule,params,dop,webobj,host)

def set_post_cache_chain_rule(chain,rule,params,dop=None,webobj=None,host=None):
    global DictOp

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    set_chain_rule("PostCacheChain",chain,rule,params,dop,webobj,host)


def set_threshold(plugin,type,params,dop=None, webobj=None, host=None):
    global DictOp
    retval = False

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    _keys = [ThresholdName,"Threshold","","Plugin",plugin,"Type",type]

    for _ts_param in ["DataSource","Instance","WarningMax","WarningMin","FailureMax","FailureMin","Percentage","Persist","Hits","Hysteresis"]:
        try:
            ts_param_val = params[_ts_param]
            keys = _keys + [_ts_param]
            dop.cdp_set("collectdplugin", keys, ts_param_val, multiple_file=True)
        except:
            pass

def disable_threshold(plugin,dop=None, webobj=None, host=None):
    global DictOp
    retval = False

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    keys = [ThresholdName,"Threshold","","Plugin",plugin]

    dop.cdp_comment("collectdplugin", keys, multiple_file=True)

def enable_threshold(plugin,dop=None, webobj=None, host=None):
    global DictOp
    retval = False

    if dop is None:
        if isinstance(DictOp, types.InstanceType) and DictOp.__class__.__name__ == "DictOp":
            dop = DictOp
        else:
            dop = _get_collectd_config(webobj, host)

    keys = [ThresholdName,"Threshold","","Plugin",plugin]

    dop.cdp_uncomment("collectdplugin", keys, recursive=True, multiple_file=True)



"""
"""
if __name__ == '__main__':
    """Testing
    """
    list = plugin_list()
    preprint_r(list)

    list = active_plugin_list(dop=DictOp)
    preprint_r(list)

    list = inactive_plugin_list(dop=DictOp)
    preprint_r(list)

    print is_enabled_plugin("syslog",dop=DictOp)
    print is_enabled_plugin("logfile",dop=DictOp)

    enabled_plugin("iptables",dop=DictOp)
    print is_enabled_plugin("iptables",dop=DictOp)

    disabled_plugin("logfile",dop=DictOp)
    print is_enabled_plugin("logfile",dop=DictOp)

    print get_global_parameter("Interval",dop=DictOp)
    print set_global_parameter("Interval","23",dop=DictOp)
    print get_global_parameter("Interval")

    #preprint_r(DictOp.getconf("collectdplugin"))

    print where_is_plugin("iptables")

    # rule: memory_cached_exceeded
    params = {}
    params['Plugin']   = "memory";
    params['TypeInstance'] = "cached";
    params['Min']      = "97000000";
    #params['Max']      = "1000000000";
    #params['Invert']  = "false";
    params['Satisfy']  = "Any";
    params['Target']   = "notification";
    params['Message']  = "Oops, the %{plugin} %{type_instance} memory_size is currently %{ds:value}!";
    params['Severity'] = "WARNING";
    set_post_cache_chain_rule("PostTestChain","memory_cached_exceeded",params)

    # rule: memory_used_exceeded
    params = {}
    params['Plugin']   = "memory";
    params['TypeInstance'] = "used";
    params['Min']      = "550000000";
    #params['Max']      = "10000000000";
    #params['Invert']  = "false";
    params['Satisfy']  = "Any";
    params['Target']   = "notification";
    params['Message']  = "Oops, the %{plugin} %{type_instance} memory_size is currently %{ds:value}!";
    params['Severity'] = "WARNING";
    set_post_cache_chain_rule("PostTestChain","memory_used_exceeded",params)

    # rule: cpu_exceeded
    params = {}
    params['Plugin']   = "cpu";
    #params['TypeInstance'] = "";
    params['Min']      = "0";
    params['Max']      = "80";
    params['Invert']  = "false";
    params['Satisfy']  = "All";
    params['Target']   = "notification";
    params['Message']  = "Oops, the %{plugin} %{type_instance} cpu is currently %{ds:value}!";
    params['Severity'] = "FAILURE";
    set_post_cache_chain_rule("PostTestChain","cpu_exceeded",params)

    # rule: cpu_exceeded
    params = {}
    params['Plugin']   = "cpua";
    #params['TypeInstance'] = "";
    params['Min']      = "0";
    params['Max']      = "80";
    params['Invert']  = "false";
    params['Satisfy']  = "All";
    params['Target']   = "stop";
    set_post_cache_chain_rule("PostTestChain","cpua_exceeded",params)

    #DictOp.unset("collectdplugin",["filter","Chain","PostTestChain","Rule","cpua_exceeded"],is_cdp=True,multiple_file=True)
    #DictOp.delete("collectdplugin",["filter","Chain","PostTestChain","Rule","cpua_exceeded","Match"],is_cdp=True,multiple_file=True)
    #DictOp.comment("collectdplugin",["filter","Chain","PostTestChain","Rule","cpua_exceeded"],is_cdp=True,multiple_file=True)
    #DictOp.uncomment("collectdplugin",["filter","Chain","PostTestChain","Rule","cpub_exceeded"],is_cdp=True,multiple_file=True)
    #print DictOp.action("collectdplugin",["filter","Chain","PostTestChain","Rule","cpua_exceeded","Match"],is_cdp=True,multiple_file=True)
    print DictOp.unset("collectdplugin",["filter","Chain","PostTestChain","Rule","cpua_exceeded","Match"],is_cdp=True,multiple_file=True)
    print DictOp.isset("collectdplugin",["filter","Chain","PostTestChain","Rule","cpua_exceeded","Match"],is_cdp=True,multiple_file=True)
    print DictOp.cdp_isset("collectdplugin",["filter","Chain","PostTestChain","Rule","cpua_exceeded","Match"],multiple_file=True)

    # threshold: df_exceeded
    params = {}
    params['Plugin']     = "df";
    params['Type']       = "df";
    params['WarningMax'] = "10";
    params['FailureMax'] = "25";
    params['Percentage'] = "true";
    params['Persist']    = "true";
    set_threshold("df","df",params)

    disable_threshold("df")
    enable_threshold("df")
    disable_threshold("df")

    disable_python_plugin()
    enable_python_plugin()

    conf = DictOp.getconf("collectdplugin")
    preprint_r(conf["filter"])
    preprint_r(conf["python"])
    preprint_r(conf["threshold"])

    from karesansui.lib.parser.collectdplugin import collectdpluginParser
    parser = collectdpluginParser()
    parser.write_conf(conf,dryrun=True)

    pass
