﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.IO;
using System.Reflection;
using System.Xml.Serialization;
using Kasuga.Ass;

namespace Kasuga.StandardPlugins.Formats
{
    [Serializable]
    [DisplayName("軽量ワイプ")]
    [Description("縁ワイプする字幕を提供します。")]
    public class LightBorderWipeFormat : FormatForVSF239, IFormat, IFormatForAss
    {
        public LightBorderWipeFormat() { }

        public LightBorderWipeFormat(
            string fontFamilyName,
            float fontSize,
            bool isBold,
            bool isItalic,
            bool hasUnderline,
            bool hasStrikeout,
            byte gdiCharSet,
            bool isGdiVerticalFont,
            float scaleX,
            float scaleY,
            float borderWidthX,
            float borderWidthY,
            float shadowDepthX,
            float shadowDepthY,
            float rotateX,
            float rotateY,
            float rotateZ,
            bool useOrigination,
            PointF origination,
            float shearX,
            float shearY,
            float blur,
            Color beforeForeColor,
            Color afterForeColor,
            Color beforeBorderColor,
            Color afterBorderColor,
            Color beforeShadowColor,
            Color afterShadowColor,
            bool isRightToLeft)
        {
            try
            {
                FontFamilyName = fontFamilyName;
                FontSize = fontSize;
                IsBold = isBold;
                IsItalic = isItalic;
                HasUnderline = hasUnderline;
                HasStrikeout = hasStrikeout;
                GdiCharSet = gdiCharSet;
                IsGdiVerticalFont = isGdiVerticalFont;
                ScaleX = scaleX;
                ScaleY = scaleY;
                BorderWidthX = borderWidthX;
                BorderWidthY = borderWidthY;
                ShadowDepthX = shadowDepthX;
                ShadowDepthY = shadowDepthY;
                RotateX = rotateX;
                RotateY = rotateY;
                RotateZ = rotateZ;
                UseOrigination = useOrigination;
                Origination = origination;
                ShearX = shearX;
                ShearY = shearY;
                Blur = blur;
                BeforeForeColor = beforeForeColor;
                AfterForeColor = afterForeColor;
                BeforeBorderColor = beforeBorderColor;
                AfterBorderColor = afterBorderColor;
                BeforeShadowColor = beforeShadowColor;
                AfterShadowColor = afterShadowColor;
                IsRightToLeft = isRightToLeft;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public static CatalogItem<IFormat> MainTextDefaultLightBorderWipeFormat =
            new CatalogItem<IFormat>(
                "軽量ワイプ 本文デフォルト",
                new LightBorderWipeFormat(
                    "ＭＳ Ｐゴシック",
                    48,
                    false,
                    false,
                    false,
                    false,
                    1,
                    false,
                    1,
                    1,
                    4,
                    4,
                    4,
                    4,
                    0,
                    0,
                    0,
                    false,
                    new PointF(0, 0),
                    0,
                    0,
                    0,
                    Color.White,
                    Color.Blue,
                    Color.Black,
                    Color.White,
                    Color.FromArgb(byte.MaxValue / 2, Color.Black),
                    Color.FromArgb(byte.MaxValue / 2, Color.Black),
                    false));

        public static CatalogItem<IFormat> RubyTextDefaultLightBorderWipeFormat =
            new CatalogItem<IFormat>(
                "軽量ワイプ ルビデフォルト",
                new LightBorderWipeFormat(
                    "ＭＳ Ｐゴシック",
                    22,
                    false,
                    false,
                    false,
                    false,
                    1,
                    false,
                    1,
                    1,
                    2.5F,
                    2.5F,
                    4,
                    4,
                    0,
                    0,
                    0,
                    false,
                    new PointF(0, 0),
                    0,
                    0,
                    0,
                    Color.White,
                    Color.Blue,
                    Color.Black,
                    Color.White,
                    Color.FromArgb(byte.MaxValue / 2, Color.Black),
                    Color.FromArgb(byte.MaxValue / 2, Color.Black),
                    false));

        [Category("書式")]
        [DisplayName("フォント")]
        [TypeConverter(typeof(FontConverter))]
        [XmlIgnore]
        public override Font Font
        {
            get
            {
                try
                {
                    return base.Font;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    base.Font = value;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Category("書式")]
        [DisplayName("X方向の縮尺")]
        public override float ScaleX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の縮尺")]
        public override float ScaleY { get; set; }
        [Category("書式")]
        [DisplayName("X方向の縁取りの太さ")]
        public override float BorderWidthX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の縁取りの太さ")]
        public override float BorderWidthY { get; set; }
        [Category("書式")]
        [DisplayName("X方向の影の深さ")]
        public override float ShadowDepthX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の影の深さ")]
        public override float ShadowDepthY { get; set; }
        [Category("書式")]
        [DisplayName("回転の中心点を指定する")]
        public override bool UseOrigination { get; set; }
        [Category("書式")]
        [DisplayName("回転の中心点")]
        [TypeConverter(typeof(PointFConverter))]
        public override PointF Origination { get; set; }
        [Category("書式")]
        [DisplayName("X方向の回転角度")]
        public override float RotateX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の回転角度")]
        public override float RotateY { get; set; }
        [Category("書式")]
        [DisplayName("Z方向の回転角度")]
        public override float RotateZ { get; set; }
        [Category("書式")]
        [DisplayName("X方向の剪断変形")]
        public override float ShearX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の剪断変形")]
        public override float ShearY { get; set; }
        [Category("書式")]
        [DisplayName("縁取りのぼかしの強さ")]
        public override float Blur { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ後の内側の色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public override Color AfterForeColor { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ前の内側の色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public override Color BeforeForeColor { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ後の縁取りの色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public override Color AfterBorderColor { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ前の縁取りの色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public override Color BeforeBorderColor { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ後の影の色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public override Color AfterShadowColor { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ前の影の色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public override Color BeforeShadowColor { get; set; }
        [Category("書式")]
        [DisplayName("右から左へワイプ")]
        [TypeConverter(typeof(BoolConverter))]
        public virtual bool IsRightToLeft { get; set; }

        public void DrawForPreview(
            SortedDictionary<int, Bitmap> bitmaps,
            KsgCharacter character,
            bool isWiped)
        {
            try
            {
                if (!char.IsWhiteSpace(character.Char))
                {
                    LightDrawingWay.Draw(this, bitmaps, character, isWiped);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public void ExportToAss(
            List<AssStyle> styles,
            List<AssEvent> events,
            KsgCharacter character)
        {
            try
            {
                if (char.IsWhiteSpace(character.Char))
                {
                    return;
                }

                AssStyle beforeStyle, afterStyle;
                beforeStyle = LightDrawingWay.CreateStyle(this, styles, false);
                afterStyle = LightDrawingWay.CreateStyle(this, styles, true);

                string commonTags = GetExtendTags() + LightDrawingWay.GetPositionTag(this, character);

                //from ViewStart to SingStart
                LightDrawingWay.AddDialogues(
                    events,
                    character.ViewStart,
                    character.SingStart,
                    beforeStyle,
                    commonTags + character.Char.ToString());
                //from SingStart to SingEnd
                if (RotateZ % 90 == 0 && RotateX % 360 == 0 && RotateY % 360 == 0 && ShearX == 0 && ShearY == 0 && Blur == 0)
                {
                    List<PositionAndPlayTime> wipeControls = new List<PositionAndPlayTime>();
                    {
                        List<PositionAndPlayTime> singTimes = character.SingTimes;
                        if (singTimes[0].Position > 0)
                        {
                            wipeControls.Add(new PositionAndPlayTime(0, character.SingStart));
                        }
                        wipeControls.AddRange(singTimes);
                        if (singTimes[singTimes.Count - 1].Position < 1)
                        {
                            wipeControls.Add(new PositionAndPlayTime(1, character.SingEnd));
                        }
                    }

                    for (int i = 0; i < wipeControls.Count - 1; i++)
                    {
                        string beforeClipTags, afterClipTags;
                        {
                            bool isNextCharacterSameFormat;
                            {
                                KsgCharacter nextCharacter = character.NextCharacter;
                                if (nextCharacter != null
                                    && !char.IsWhiteSpace(nextCharacter.Char)
                                    && nextCharacter.Format is LightBorderWipeFormat)
                                {
                                    LightBorderWipeFormat nextFormat = (LightBorderWipeFormat)nextCharacter.Format;
                                    isNextCharacterSameFormat =
                                        (nextFormat.RotateZ / 90) % 4 == (RotateZ / 90) % 4
                                        && nextFormat.RotateX % 360 == 0
                                        && nextFormat.RotateY % 360 == 0
                                        && nextFormat.ShearX == 1
                                        && nextFormat.ShearY == 1
                                        && nextFormat.Blur == 0
                                        && nextFormat.IsRightToLeft == IsRightToLeft;
                                }
                                else
                                {
                                    isNextCharacterSameFormat = false;
                                }
                            }
                            GetClipTags(
                                character,
                                isNextCharacterSameFormat,
                                wipeControls[i],
                                wipeControls[i + 1],
                                out beforeClipTags,
                                out afterClipTags);
                        }
                        LightDrawingWay.AddDialogues(
                            events,
                            wipeControls[i].PlayTime,
                            wipeControls[i + 1].PlayTime,
                            beforeStyle,
                            commonTags + beforeClipTags + character.Char.ToString());
                        LightDrawingWay.AddDialogues(
                            events,
                            wipeControls[i].PlayTime,
                            wipeControls[i + 1].PlayTime,
                            afterStyle,
                            commonTags + afterClipTags + character.Char.ToString());
                    }
                }
                else
                {
                    throw new NotImplementedException();
                }
                //from SingEnd to ViewEnd
                LightDrawingWay.AddDialogues(
                    events,
                    character.SingEnd,
                    character.ViewEnd,
                    afterStyle,
                    commonTags + character.Char.ToString());
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        protected void GetClipTags(
            KsgCharacter character,
            bool isNextCharacterSameFormat,
            PositionAndPlayTime start,
            PositionAndPlayTime end,
            out string before,
            out string after)
        {
            try
            {
                Size resolution = character.ParentSubtitle.Resolution;
                float clipStart, clipEnd;
                Corners corners = GetCharacterCorners(character);
                Corners borderCorners = this.GetBorderCorners(character);
                KsgCharacter nextCharacter = character.NextCharacter;
                Corners nextBorderCorners = null;
                if (nextCharacter != null)
                {
                    nextBorderCorners = this.GetBorderCorners(nextCharacter);
                }

                int rightToLeft = IsRightToLeft ? 2 : 0;
                switch (((int)RotateZ / 90 + rightToLeft) % 4)
                {
                    case 0:
                        if (start.Position <= 0)
                        {
                            clipStart = borderCorners.TopLeft.X;
                        }
                        else
                        {
                            clipStart = (float)(corners.TopLeft.X + (corners.BottomRight.X - corners.TopLeft.X) * start.Position);
                        }
                        if (isNextCharacterSameFormat
                            && character.SingEnd == nextCharacter.SingStart
                            && borderCorners.TopLeft.X < nextBorderCorners.TopLeft.X)
                        {
                            if (end.Position >= 1)
                            {
                                clipEnd = nextBorderCorners.TopLeft.X;
                            }
                            else
                            {
                                clipEnd = Math.Min(
                                    (float)(corners.TopLeft.X + (corners.BottomRight.X - corners.TopLeft.X) * end.Position),
                                    nextBorderCorners.TopLeft.X);
                            }
                        }
                        else
                        {
                            if (end.Position >= 1)
                            {
                                clipEnd = borderCorners.BottomRight.X + ShadowDepthX;
                            }
                            else
                            {
                                clipEnd = (float)(corners.TopLeft.X + (corners.BottomRight.X - corners.TopLeft.X) * end.Position);
                            }
                        }

                        before = BorderWipe.GenerateClipTags(this, SideKind.Left, resolution, clipStart, clipEnd, end.PlayTime - start.PlayTime, false);
                        after = BorderWipe.GenerateClipTags(this, SideKind.Right, resolution, clipStart, clipEnd, end.PlayTime - start.PlayTime, false);
                        break;
                    case 1:
                        if (start.Position <= 0)
                        {
                            clipStart = borderCorners.TopLeft.Y;
                        }
                        else
                        {
                            clipStart = (float)(corners.TopLeft.Y + (corners.BottomRight.Y - corners.TopLeft.Y) * start.Position);
                        }
                        if (isNextCharacterSameFormat
                            && character.SingEnd == nextCharacter.SingStart
                            && borderCorners.TopLeft.Y > nextBorderCorners.TopLeft.Y)
                        {
                            if (end.Position >= 1)
                            {
                                clipEnd = nextBorderCorners.TopLeft.Y;
                            }
                            else
                            {
                                clipEnd = Math.Min(
                                    (float)(corners.TopLeft.Y + (corners.BottomRight.Y - corners.TopLeft.Y) * end.Position),
                                    nextBorderCorners.TopLeft.Y);
                            }
                        }
                        else
                        {
                            if (end.Position >= 1)
                            {
                                clipEnd = borderCorners.BottomRight.Y - ShadowDepthY;
                            }
                            else
                            {
                                clipEnd = (float)(corners.TopLeft.Y + (corners.BottomRight.Y - corners.TopLeft.Y) * end.Position);
                            }
                        }

                        before = BorderWipe.GenerateClipTags(this, SideKind.Bottom, resolution, clipStart, clipEnd, end.PlayTime - start.PlayTime, false);
                        after = BorderWipe.GenerateClipTags(this, SideKind.Top, resolution, clipStart, clipEnd, end.PlayTime - start.PlayTime, false);
                        break;
                    case 2:
                        if (start.Position <= 0)
                        {
                            clipStart = borderCorners.TopLeft.X;
                        }
                        else
                        {
                            clipStart = (float)(corners.TopLeft.X + (corners.BottomRight.X - corners.TopLeft.X) * start.Position);
                        }
                        if (isNextCharacterSameFormat
                            && character.SingEnd == nextCharacter.SingStart
                            && borderCorners.TopLeft.X > nextBorderCorners.TopLeft.X)
                        {
                            if (end.Position >= 1)
                            {
                                clipEnd = nextBorderCorners.TopLeft.X;
                            }
                            else
                            {
                                clipEnd = Math.Min(
                                    (float)(corners.TopLeft.X + (corners.BottomRight.X - corners.TopLeft.X) * end.Position),
                                    nextBorderCorners.TopLeft.X);
                            }
                        }
                        else
                        {
                            if (end.Position >= 1)
                            {
                                clipEnd = borderCorners.BottomRight.X - ShadowDepthX;
                            }
                            else
                            {
                                clipEnd = (float)(corners.TopLeft.X + (corners.BottomRight.X - corners.TopLeft.X) * end.Position);
                            }
                        }

                        before = BorderWipe.GenerateClipTags(this, SideKind.Right, resolution, clipStart, clipEnd, end.PlayTime - start.PlayTime, false);
                        after = BorderWipe.GenerateClipTags(this, SideKind.Left, resolution, clipStart, clipEnd, end.PlayTime - start.PlayTime, false);
                        break;
                    case 3:
                        if (start.Position <= 0)
                        {
                            clipStart = borderCorners.TopLeft.Y;
                        }
                        else
                        {
                            clipStart = (float)(corners.TopLeft.Y + (corners.BottomRight.Y - corners.TopLeft.Y) * start.Position);
                        }
                        if (isNextCharacterSameFormat
                            && character.SingEnd == nextCharacter.SingStart
                            && borderCorners.TopLeft.Y < nextBorderCorners.TopLeft.Y)
                        {
                            if (end.Position >= 1)
                            {
                                clipEnd = nextBorderCorners.TopLeft.Y;
                            }
                            else
                            {
                                clipEnd = Math.Min(
                                    (float)(corners.TopLeft.Y + (corners.BottomRight.Y - corners.TopLeft.Y) * end.Position),
                                    nextBorderCorners.TopLeft.Y);
                            }
                        }
                        else
                        {
                            if (end.Position >= 1)
                            {
                                clipEnd = borderCorners.TopLeft.Y + ShadowDepthY;
                            }
                            else
                            {
                                clipEnd = (float)(corners.TopLeft.Y + (corners.BottomRight.Y - corners.TopLeft.Y) * end.Position);
                            }
                        }

                        before = BorderWipe.GenerateClipTags(this, SideKind.Top, resolution, clipStart, clipEnd, end.PlayTime - start.PlayTime, false);
                        after = BorderWipe.GenerateClipTags(this, SideKind.Bottom, resolution, clipStart, clipEnd, end.PlayTime - start.PlayTime, false);
                        break;
                    default:
                        before = string.Empty;
                        after = string.Empty;
                        break;
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                before = string.Empty;
                after = string.Empty;
            }
        }

        public object Clone()
        {
            try
            {
                XmlSerializer serializer = new XmlSerializer(typeof(LightBorderWipeFormat));
                StringWriter writer = new StringWriter();
                serializer.Serialize(writer, this);
                StringReader reader = new StringReader(writer.ToString());
                object obj = serializer.Deserialize(reader);
                reader.Close();
                writer.Close();
                return obj;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return null;
            }
        }
    }
}
