﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.IO;
using System.Reflection;
using System.Xml.Serialization;

namespace Kasuga.StandardPlugins.LineArrangement
{
    [Serializable]
    [DisplayName("線形")]
    [Description("行内要素を直線上に配置します。")]
    public class LinearLineArrangement : ILineArrangement
    {
        public LinearLineArrangement() { }

        public LinearLineArrangement(
            bool isRightToLeft)
        {
            try
            {
                IsRightToLeft = isRightToLeft;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public static CatalogItem<ILineArrangement> DefaultLinearLineArrangement =
            new CatalogItem<ILineArrangement>(
                "線形 デフォルト",
                new LinearLineArrangement(
                    false));

        [Category("行内要素の配置")]
        [DisplayName("右から並べる")]
        [TypeConverter(typeof(BoolConverter))]
        public virtual bool IsRightToLeft { get; set; }

        public float Measure(KsgLine line)
        {
            try
            {
                float width = 0;
                foreach (KsgWord word in line.Words)
                {
                    width += word.MeasureWidth();
                }
                return width;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return 0;
            }
        }

        public void Arrange(KsgLine line, PointF basePoint, float baseAngle)
        {
            try
            {
                float sinAngle, cosAngle;
                sinAngle = (float)Math.Sin(baseAngle * Math.PI / 180);
                cosAngle = (float)Math.Cos(baseAngle * Math.PI / 180);
                if (IsRightToLeft)
                {
                    for (int i = line.Words.Count - 1; i >= 0; i--)
                    {
                        KsgWord word = line.Words[i];
                        word.WordArrange(basePoint, baseAngle);
                        float width = word.MeasureWidth();
                        basePoint.X += width * cosAngle;
                        basePoint.Y -= width * sinAngle;
                    }
                }
                else
                {
                    for (int i = 0; i < line.Words.Count; i++)
                    {
                        KsgWord word = line.Words[i];
                        word.WordArrange(basePoint, baseAngle);
                        float width = word.MeasureWidth();
                        basePoint.X += width * cosAngle;
                        basePoint.Y -= width * sinAngle;
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        public object Clone()
        {
            try
            {
                XmlSerializer serializer = new XmlSerializer(typeof(LinearLineArrangement));
                StringWriter writer = new StringWriter();
                serializer.Serialize(writer, this);
                StringReader reader = new StringReader(writer.ToString());
                object obj = serializer.Deserialize(reader);
                reader.Close();
                writer.Close();
                return obj;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return null;
            }
        }
    }
}
