﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Reflection;
using System.Xml.Serialization;

namespace Kasuga.StandardPlugins.Formats
{
    public abstract class FormatBase
    {
        [Browsable(false)]
        public virtual string FontFamilyName { get; set; }
        [Browsable(false)]
        public virtual float FontSize { get; set; }
        [Browsable(false)]
        public virtual bool IsBold { get; set; }
        [Browsable(false)]
        public virtual bool IsItalic { get; set; }
        [Browsable(false)]
        public virtual bool HasUnderline { get; set; }
        [Browsable(false)]
        public virtual bool HasStrikeout { get; set; }
        [Browsable(false)]
        public virtual byte GdiCharSet { get; set; }
        [Browsable(false)]
        public virtual bool IsGdiVerticalFont { get; set; }

        [Browsable(false)]
        [XmlIgnore]
        public virtual string FontName
        {
            get
            {
                try
                {
                    if (IsGdiVerticalFont)
                    {
                        return "@" + FontFamilyName;
                    }
                    else
                    {
                        return FontFamilyName;
                    }
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return string.Empty;
                }
            }
        }

        [Browsable(false)]
        [XmlIgnore]
        public virtual FontStyle FontStyle
        {
            get
            {
                try
                {
                    FontStyle style = 0;
                    if (IsBold)
                    {
                        style |= FontStyle.Bold;
                    }
                    if (IsItalic)
                    {
                        style |= FontStyle.Italic;
                    }
                    if (HasUnderline)
                    {
                        style |= FontStyle.Underline;
                    }
                    if (HasStrikeout)
                    {
                        style |= FontStyle.Strikeout;
                    }
                    return style;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return FontStyle.Regular;
                }
            }
        }

        [Category("書式")]
        [DisplayName("フォント")]
        [TypeConverter(typeof(FontConverter))]
        [XmlIgnore]
        public virtual Font Font
        {
            get
            {
                try
                {
                    return new Font(
                        FontFamilyName,
                        FontSize,
                        FontStyle,
                        GraphicsUnit.Pixel,
                        GdiCharSet,
                        IsGdiVerticalFont);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
            set
            {
                try
                {
                    FontFamilyName = value.Name;
                    FontSize = value.Size;
                    IsBold = value.Bold;
                    IsItalic = value.Italic;
                    HasUnderline = value.Underline;
                    HasStrikeout = value.Strikeout;
                    GdiCharSet = value.GdiCharSet;
                    IsGdiVerticalFont = value.GdiVerticalFont;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Category("書式")]
        [DisplayName("X方向の縮尺")]
        public virtual float ScaleX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の縮尺")]
        public virtual float ScaleY { get; set; }
        [Category("書式")]
        [DisplayName("X方向の縁取りの太さ")]
        public virtual float BorderWidthX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の縁取りの太さ")]
        public virtual float BorderWidthY { get; set; }
        [Category("書式")]
        [DisplayName("X方向の影の深さ")]
        public virtual float ShadowDepthX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の影の深さ")]
        public virtual float ShadowDepthY { get; set; }
        [Category("書式")]
        [DisplayName("回転の中心点を指定する")]
        public virtual bool UseOrigination { get; set; }
        [Category("書式")]
        [DisplayName("回転の中心点")]
        [TypeConverter(typeof(PointFConverter))]
        public virtual PointF Origination { get; set; }
        [Category("書式")]
        [DisplayName("X方向の回転角度")]
        public virtual float RotateX { get; set; }
        [Category("書式")]
        [DisplayName("Y方向の回転角度")]
        public virtual float RotateY { get; set; }
        [Category("書式")]
        [DisplayName("Z方向の回転角度")]
        public virtual float RotateZ { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ後の内側の色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public virtual Color AfterForeColor { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ前の内側の色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public virtual Color BeforeForeColor { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ後の縁取りの色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public virtual Color AfterBorderColor { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ前の縁取りの色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public virtual Color BeforeBorderColor { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ後の影の色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public virtual Color AfterShadowColor { get; set; }
        [Category("書式")]
        [DisplayName("ワイプ前の影の色")]
        [TypeConverter(typeof(ColorConverter))]
        [XmlIgnore]
        public virtual Color BeforeShadowColor { get; set; }

        [Browsable(false)]
        public virtual string AfterForeColorString
        {
            get
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    return converter.ConvertToString(AfterForeColor);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return string.Empty;
                }
            }
            set
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    AfterForeColor = (Color)converter.ConvertFromString(value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        public virtual string BeforeForeColorString
        {
            get
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    return converter.ConvertToString(BeforeForeColor);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return string.Empty;
                }
            }
            set
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    BeforeForeColor = (Color)converter.ConvertFromString(value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        public virtual string AfterBorderColorString
        {
            get
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    return converter.ConvertToString(AfterBorderColor);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return string.Empty;
                }
            }
            set
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    AfterBorderColor = (Color)converter.ConvertFromString(value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        public virtual string BeforeBorderColorString
        {
            get
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    return converter.ConvertToString(BeforeBorderColor);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return string.Empty;
                }
            }
            set
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    BeforeBorderColor = (Color)converter.ConvertFromString(value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        public virtual string AfterShadowColorString
        {
            get
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    return converter.ConvertToString(AfterShadowColor);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return string.Empty;
                }
            }
            set
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    AfterShadowColor = (Color)converter.ConvertFromString(value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        [Browsable(false)]
        public virtual string BeforeShadowColorString
        {
            get
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    return converter.ConvertToString(BeforeShadowColor);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return string.Empty;
                }
            }
            set
            {
                try
                {
                    ColorConverter converter = new ColorConverter();
                    BeforeShadowColor = (Color)converter.ConvertFromString(value);
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                }
            }
        }

        public abstract PointF Transform(PointF point, RectangleF originalRectangle);
        public abstract void Rasterize(Bitmap bitmap, GraphicsPath blurPath);

        public Corners GetCharacterCorners(KsgCharacter character)
        {
            try
            {
                RectangleF rectangle = new RectangleF(character.Position, GetOriginalSize(character));
                return new Corners(
                    Transform(new PointF(rectangle.Left, rectangle.Top), rectangle),
                    Transform(new PointF(rectangle.Right, rectangle.Top), rectangle),
                    Transform(new PointF(rectangle.Right, rectangle.Bottom), rectangle),
                    Transform(new PointF(rectangle.Left, rectangle.Bottom), rectangle));
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return null;
            }
        }

        private SizeF GetOriginalSize(KsgCharacter character)
        {
            try
            {
                SizeF size;
                using (Graphics graphics = character.ParentSubtitle.Graphics)
                {
                    using (Font font = Font)
                    {
                        using (StringFormat stringFormat = StringFormat.GenericTypographic)
                        {
                            stringFormat.FormatFlags |= StringFormatFlags.MeasureTrailingSpaces;
                            size = graphics.MeasureString(character.Char.ToString(), font, int.MaxValue, stringFormat);
                            stringFormat.Dispose();
                        }
                        font.Dispose();
                    }
                    graphics.Dispose();
                }
                return size;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return SizeF.Empty;
            }
        }

        public Corners GetBorderCorners(KsgCharacter character)
        {
            try
            {
                Corners corners = GetCharacterCorners(character);
                return new Corners(
                    new PointF(corners.TopLeft.X - BorderWidthX, corners.TopLeft.Y - BorderWidthY),
                    new PointF(corners.TopRight.X + BorderWidthX, corners.TopRight.Y - BorderWidthY),
                    new PointF(corners.BottomRight.X + BorderWidthX, corners.BottomRight.Y + BorderWidthY),
                    new PointF(corners.BottomLeft.X - BorderWidthX, corners.BottomLeft.Y + BorderWidthY));
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return null;
            }
        }

        public GraphicsPath GetForePath(KsgCharacter character)
        {
            try
            {
                if (char.IsWhiteSpace(character.Char))
                {
                    return null;
                }
                else
                {
                    List<PointF> points = new List<PointF>();
                    List<byte> types = new List<byte>();
                    using (GraphicsPath path = new GraphicsPath())
                    {
                        using (Graphics graphics = character.ParentSubtitle.Graphics)
                        {
                            using (Font font = Font)
                            {
                                using (FontFamily family = font.FontFamily)
                                {
                                    using (StringFormat stringFormat = StringFormat.GenericTypographic)
                                    {
                                        if (IsGdiVerticalFont)
                                        {
                                            stringFormat.FormatFlags |= StringFormatFlags.DirectionVertical;
                                        }
                                        path.AddString(
                                            character.Char.ToString(),
                                            family,
                                            (int)FontStyle,
                                            Font.SizeInPoints * graphics.DpiY / 72,
                                            character.Position,
                                            stringFormat);
                                        stringFormat.Dispose();
                                    }
                                    family.Dispose();
                                }
                                font.Dispose();
                            }
                            graphics.Dispose();
                        }
                        RectangleF rectangle = new RectangleF(character.Position, GetOriginalSize(character));
                        foreach (PointF point in path.PathPoints)
                        {
                            points.Add(Transform(point, rectangle));
                        }
                        foreach (byte type in path.PathTypes)
                        {
                            types.Add(type);
                        }
                        path.Dispose();
                    }
                    return new GraphicsPath(points.ToArray(), types.ToArray());
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return null;
            }
        }

        public GraphicsPath GetWidenPath(KsgCharacter character)
        {
            try
            {
                if (char.IsWhiteSpace(character.Char))
                {
                    return null;
                }
                else
                {
                    GraphicsPath path = GetForePath(character);
                    using (Pen pen = new Pen(Color.Black, 2))
                    {
                        pen.ScaleTransform(BorderWidthX, BorderWidthY);
                        pen.LineJoin = LineJoin.Round;
                        path.Widen(pen);
                        pen.Dispose();
                    }
                    return path;
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
                return null;
            }
        }
    }
}
