﻿using System;
using System.Collections.Generic;
using System.Drawing;
using System.Reflection;
using System.Text;
using System.Windows.Forms;

namespace Kasuga
{
    public class KsgTreeView : TreeView
    {
        public KsgTreeView() : base()
        {
            try
            {
                _checkedNodes = new SortedSet<TreeNode>(new TreeNodeComparer());
                CheckBoxes = true;
                AfterCheck += new TreeViewEventHandler(KsgTreeView_AfterCheck);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private SortedSet<TreeNode> _checkedNodes;
        public event EventHandler CheckedElementsChanged;

        public object[] CheckedElements
        {
            get
            {
                try
                {
                    List<object> elements = new List<object>();
                    foreach (TreeNode node in _checkedNodes)
                    {
                        elements.Add(node.Tag);
                    }
                    return elements.ToArray();
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return null;
                }
            }
        }

        public void Initialize(KsgSubtitle subtitle)
        {
            try
            {
                _checkedNodes.Clear();
                CheckedElementsChanged.Invoke(this, EventArgs.Empty);
                Nodes.Clear();
                TreeNode subtitleNode = new TreeNode();
                foreach(KsgPart part in subtitle.Parts)
                {
                    TreeNode partNode = new TreeNode();
                    foreach (KsgPage page in part.Pages)
                    {
                        TreeNode pageNode = new TreeNode();
                        foreach(KsgLine line in page.Lines)
                        {
                            TreeNode lineNode = new TreeNode();
                            StringBuilder lineTextBuilder = new StringBuilder();
                            foreach (KsgWord word in line.Words)
                            {
                                TreeNode wordNode = new TreeNode();
                                StringBuilder wordTextBuilder = new StringBuilder();
                                foreach (KsgCharacter character in word.MainTextCharacters)
                                {
                                    TreeNode characterNode = new TreeNode();
                                    characterNode.Tag = character;
                                    characterNode.Text = "本文文字" + (word.MainTextCharacters.IndexOf(character) + 1).ToString() + ": " + character.Char.ToString();
                                    wordNode.Nodes.Add(characterNode);
                                    wordTextBuilder.Append(character.Char);
                                }
                                foreach (KsgCharacter character in word.RubyTextCharacters)
                                {
                                    TreeNode characterNode = new TreeNode();
                                    characterNode.Tag = character;
                                    characterNode.Text = "ルビ文字" + (word.RubyTextCharacters.IndexOf(character) + 1).ToString() + ": " + character.Char.ToString();
                                    wordNode.Nodes.Add(characterNode);
                                }
                                wordNode.Tag = word;
                                wordNode.Text = "単語" + (line.Words.IndexOf(word) + 1).ToString() + ": " + wordTextBuilder.ToString();
                                lineNode.Nodes.Add(wordNode);
                                lineTextBuilder.Append(wordTextBuilder.ToString());
                            }
                            lineNode.Tag = line;
                            lineNode.Text = "行" + (page.Lines.IndexOf(line) + 1).ToString() + ": " + lineTextBuilder.ToString();
                            pageNode.Nodes.Add(lineNode);
                        }
                        pageNode.Tag = page;
                        pageNode.Text = "ページ" + (part.Pages.IndexOf(page) + 1).ToString();
                        partNode.Nodes.Add(pageNode);
                    }
                    partNode.Tag = part;
                    partNode.Text = "パート" + (subtitle.Parts.IndexOf(part) + 1).ToString();
                    subtitleNode.Nodes.Add(partNode);
                }
                subtitleNode.Tag = subtitle;
                subtitleNode.Text = "字幕";
                Nodes.Add(subtitleNode);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void KsgTreeView_AfterCheck(object sender, TreeViewEventArgs e)
        {
            try
            {
                if (e.Node.Checked)
                {
                    _checkedNodes.Add(e.Node);
                }
                else
                {
                    _checkedNodes.Remove(e.Node);
                }
                CheckedElementsChanged.Invoke(this, EventArgs.Empty);
                Refresh();
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private class TreeNodeComparer : IComparer<TreeNode>
        {
            public int Compare(TreeNode node1, TreeNode node2)
            {

                int index1 = 0, index2 = 0;
                GetNodeIndexInTree(node1, node1.TreeView.TopNode, ref index1);
                GetNodeIndexInTree(node2, node2.TreeView.TopNode, ref index2);
                return index1 - index2;
            }

            private bool GetNodeIndexInTree(TreeNode targetNode, TreeNode searchNode, ref int count)
            {
                try
                {
                    if (targetNode == searchNode)
                    {
                        return true;
                    }
                    else
                    {
                        count++;
                        foreach (TreeNode childNode in searchNode.Nodes)
                        {
                            if (GetNodeIndexInTree(targetNode, childNode, ref count))
                            {
                                return true;
                            }
                        }
                        searchNode = searchNode.NextNode;
                        if (searchNode != null)
                        {
                            if (GetNodeIndexInTree(targetNode, searchNode, ref count))
                            {
                                return true;
                            }
                        }
                    }
                    return false;
                }
                catch (Exception exception)
                {
                    Debug.Show(
                        exception,
                        Assembly.GetExecutingAssembly(),
                        MethodBase.GetCurrentMethod());
                    return false;
                }
            }
        }
    }
}
