﻿using System;
using System.Reflection;
using System.Text;
using System.Windows.Forms;
using Kasuga.DirectShow;

namespace Kasuga
{
    public partial class MainForm : Form
    {
        public MainForm()
        {
            try
            {
                InitializeComponent();

                _audioPlayer = new AudioPlayer(Handle, WM_GRAPH_NOTIFY);
                PluginManager.LoadPlugins();
                _subtitle = new KsgSubtitle();
                timer.Enabled = true;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private KsgSubtitle _subtitle;
        private AudioPlayer _audioPlayer;
        private string _openFileName = string.Empty;
        private bool _updateSeekBarValue = false;
        private bool _isUpdatingSeekBarValue = false;
        private const int WM_GRAPH_NOTIFY = 0x8000 + 100;

        private void openAudioFileToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                OpenFileDialog dialog = new OpenFileDialog();
                dialog.Filter = "音声ファイル (*.mp3;*.wav)|*.mp3;*.wav|すべてのファイル (*.*)|*.*";
                if (dialog.ShowDialog() == System.Windows.Forms.DialogResult.OK)
                {
                    bool result = _audioPlayer.OpenFile(dialog.FileName);
                    if (result)
                    {
                        _audioPlayer.Rate = rateToolStripTrackBar.Value / 100.0;
                        _audioPlayer.Volume = volumeToolStripTrackBar.Value / 100.0;

                        PlayTimeSpan duration = _audioPlayer.Duration;
                        seekToolStripTrackBar.Maximum = (int)duration.TotalSeconds;
                        durationToolStripLabel.Text = (PlayTime.Zero + duration).ToAssTimeString();
                    }
                    else
                    {
                        seekToolStripTrackBar.Maximum = 0;
                        durationToolStripLabel.Text = PlayTime.Zero.ToAssTimeString();
                    }

                    playToolStripMenuItem.Enabled = result;
                    pauseToolStripMenuItem.Enabled = result;
                    stopToolStripMenuItem.Enabled = result;
                    skipToHeadToolStripMenuItem.Enabled = result;
                    seekForwardToolStripMenuItem.Enabled = result;
                    seekBackwardToolStripMenuItem.Enabled = result;
                    rateUpToolStripMenuItem.Enabled = result;
                    rateDownToolStripMenuItem.Enabled = result;
                    volumeUpToolStripMenuItem.Enabled = result;
                    volumeDownToolStripMenuItem.Enabled = result;
                    playToolStripButton.Enabled = result;
                    pauseToolStripButton.Enabled = result;
                    skipToHeadToolStripButton.Enabled = result;
                    seekToolStripTrackBar.Enabled = result;
                    rateToolStripTrackBar.Enabled = result;
                    volumeToolStripTrackBar.Enabled = result;

                    SwitchPlayForPause(true);
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void playToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                _audioPlayer.Run();
                SwitchPlayForPause(false);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void pauseToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                _audioPlayer.Pause();
                SwitchPlayForPause(true);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void stopToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                _audioPlayer.Stop();
                _audioPlayer.CurrentPosition = PlayTime.Zero;
                currentPositionToolStripLabel.Text = PlayTime.Zero.ToAssTimeString();
                SwitchPlayForPause(true);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void skipToHeadToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                _audioPlayer.CurrentPosition = PlayTime.Zero;
                currentPositionToolStripLabel.Text = PlayTime.Zero.ToAssTimeString();
                seekToolStripTrackBar.Value = 0;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void seekForwardToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                PlayTime currentPosition = _audioPlayer.CurrentPosition;
                if (currentPosition + new PlayTimeSpan(5) < PlayTime.Zero + _audioPlayer.Duration)
                {
                    _audioPlayer.CurrentPosition = currentPosition + new PlayTimeSpan(5);
                }
                else
                {
                    _audioPlayer.CurrentPosition = PlayTime.Zero + _audioPlayer.Duration;
                }
                currentPositionToolStripLabel.Text = currentPosition.ToAssTimeString();
                seekToolStripTrackBar.Value = (int)currentPosition.TotalSeconds;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void seekBackwardToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                PlayTime currentPosition = _audioPlayer.CurrentPosition;
                if (currentPosition - new PlayTimeSpan(5) > PlayTime.Zero)
                {
                    _audioPlayer.CurrentPosition = currentPosition - new PlayTimeSpan(5);
                }
                else
                {
                    _audioPlayer.CurrentPosition = PlayTime.Zero;
                }
                currentPositionToolStripLabel.Text = currentPosition.ToAssTimeString();
                seekToolStripTrackBar.Value = (int)currentPosition.TotalSeconds;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void rateUpToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                double value;
                double rate = _audioPlayer.Rate;
                if (rate + 0.1 < 10)
                {
                    value = rate + 0.1;
                }
                else
                {
                    value = 10;
                }
                _audioPlayer.Rate = value;
                int percent = (int)(value * 100);
                rateToolStripLabel.Text = "速度: " + percent.ToString() + "%";
                if (percent >= rateToolStripTrackBar.Minimum && percent <= rateToolStripTrackBar.Maximum)
                {
                    rateToolStripTrackBar.Value = percent;
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void rateDownToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                double value;
                double rate = _audioPlayer.Rate;
                if (rate - 0.1 > 0.1)
                {
                    value = rate - 0.1;
                }
                else
                {
                    value = 0.1;
                }
                _audioPlayer.Rate = value;
                int percent = (int)(value * 100);
                rateToolStripLabel.Text = "速度: " + percent.ToString() + "%";
                if (percent >= rateToolStripTrackBar.Minimum && percent <= rateToolStripTrackBar.Maximum)
                {
                    rateToolStripTrackBar.Value = percent;
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void volumeUpToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                double value;
                double volume = _audioPlayer.Volume;
                if (volume + 0.1 < 1)
                {
                    value = volume + 0.1;
                }
                else
                {
                    value = 1;
                }
                _audioPlayer.Volume = value;
                int percent = (int)(value * 100);
                volumeToolStripLabel.Text = "音量: " + percent.ToString() + "%";
                if (percent >= volumeToolStripTrackBar.Minimum && percent <= volumeToolStripTrackBar.Maximum)
                {
                    volumeToolStripTrackBar.Value = percent;
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void volumeDownToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                double value;
                double volume = _audioPlayer.Volume;
                if (volume - 0.1 > 0)
                {
                    value = volume - 0.1;
                }
                else
                {
                    value = 0;
                }
                _audioPlayer.Volume = value;
                int percent = (int)(value * 100);
                volumeToolStripLabel.Text = "音量: " + percent.ToString() + "%";
                if (percent >= volumeToolStripTrackBar.Minimum && percent <= volumeToolStripTrackBar.Maximum)
                {
                    volumeToolStripTrackBar.Value = percent;
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void seekToolStripTrackBar_Scroll(object sender, EventArgs e)
        {
            try
            {
                PlayTime currentPosition =  new PlayTime(seekToolStripTrackBar.Value);
                _audioPlayer.CurrentPosition = currentPosition;
                currentPositionToolStripLabel.Text = currentPosition.ToAssTimeString();
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void rateToolStripTrackBar_Scroll(object sender, EventArgs e)
        {
            try
            {
                int percent = rateToolStripTrackBar.Value;
                _audioPlayer.Rate = percent / 100.0;
                rateToolStripLabel.Text = "速度: " + percent.ToString() + "%";
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void volumeToolStripTrackBar_Scroll(object sender, EventArgs e)
        {
            try
            {
                int percent = volumeToolStripTrackBar.Value;
                _audioPlayer.Volume = percent / 100.0;
                volumeToolStripLabel.Text = "音量: " + percent.ToString() + "%";
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void timer_Tick(object sender, EventArgs e)
        {
            if (_updateSeekBarValue && !_isUpdatingSeekBarValue)
            {
                _isUpdatingSeekBarValue = true;
                PlayTime currentPosition = _audioPlayer.CurrentPosition;
                seekToolStripTrackBar.Value = (int)currentPosition.TotalSeconds;
                currentPositionToolStripLabel.Text = currentPosition.ToAssTimeString();
                _isUpdatingSeekBarValue = false;
            }
        }

        protected override void WndProc(ref Message m)
        {
            try
            {
                switch (m.Msg)
                {
                    case WM_GRAPH_NOTIFY:
                        bool complete = false;
                        _audioPlayer.OnEvent(m.WParam, m.LParam, ref complete);
                        if (complete)
                        {
                            SwitchPlayForPause(true);
                            seekToolStripTrackBar.Value = 0;
                            currentPositionToolStripLabel.Text = PlayTime.Zero.ToAssTimeString();
                        }
                        break;
                }
                base.WndProc(ref m);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void SwitchPlayForPause(bool makeEnabledToPlay)
        {
            try
            {
                playToolStripMenuItem.Visible = makeEnabledToPlay;
                pauseToolStripMenuItem.Visible = !makeEnabledToPlay;
                playToolStripButton.Visible = makeEnabledToPlay;
                pauseToolStripButton.Visible = !makeEnabledToPlay;
                _updateSeekBarValue = !makeEnabledToPlay;
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void versionInfoToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                Assembly assembly = Assembly.GetExecutingAssembly();
                string title, version, copyright;
                {
                    AssemblyTitleAttribute attribute =
                        (AssemblyTitleAttribute)
                        Attribute.GetCustomAttribute(assembly, typeof(AssemblyTitleAttribute));
                    title = attribute != null ? attribute.Title : string.Empty;
                }
                {
                    AssemblyFileVersionAttribute attribute =
                        (AssemblyFileVersionAttribute)
                        Attribute.GetCustomAttribute(assembly, typeof(AssemblyFileVersionAttribute));
                    version = attribute != null ? attribute.Version : string.Empty;
                }
                {
                    AssemblyCopyrightAttribute attribute =
                        (AssemblyCopyrightAttribute)
                        Attribute.GetCustomAttribute(assembly, typeof(AssemblyCopyrightAttribute));
                    copyright = attribute != null ? attribute.Copyright : string.Empty;
                }
                StringBuilder builder = new StringBuilder();
                builder.Append(title).Append(" ").Append(version).AppendLine();
                builder.AppendLine(copyright);
                MessageBox.Show(builder.ToString());
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void openToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                OpenFileDialog dialog = new OpenFileDialog();
                dialog.Filter = "春日プロジェクトファイル (*.ksg)|*.ksg|すべてのファイル (*.*)|*.*";
                if (dialog.ShowDialog() == System.Windows.Forms.DialogResult.OK)
                {
                    _subtitle = KsgSubtitle.Open(dialog.FileName);
                    _subtitle.Initialize(previewPanel);
                    _subtitle.PageArrange();
                    treeView.Initialize(_subtitle);
                    _openFileName = dialog.FileName;
                    saveToolStripMenuItem.Enabled = true;
                    saveToolStripButton.Enabled = true;
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void saveToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                _subtitle.Save(_openFileName);
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void saveAsToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                SaveFileDialog dialog = new SaveFileDialog();
                dialog.Filter = "春日プロジェクトファイル (*.ksg)|*.ksg|すべてのファイル (*.*)|*.*";
                if (dialog.ShowDialog() == System.Windows.Forms.DialogResult.OK)
                {
                    _subtitle.Save(dialog.FileName);
                    _openFileName = dialog.FileName;
                    saveToolStripMenuItem.Enabled = true;
                    saveToolStripButton.Enabled = true;
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void importFromLyricsTextToolStripMenuItem_Click(object sender, EventArgs e)
        {
            try
            {
                OpenFileDialog dialog = new OpenFileDialog();
                dialog.Filter = "テキストファイル (*.txt)|*.txt|すべてのファイル (*.*)|*.*";
                if (dialog.ShowDialog() == System.Windows.Forms.DialogResult.OK)
                {
                    _subtitle = KsgSubtitle.ReadTextFile(dialog.FileName);
                    _subtitle.Initialize(previewPanel);
                    _subtitle.PageArrange();
                    treeView.Initialize(_subtitle);
                    _openFileName = string.Empty;
                    saveToolStripMenuItem.Enabled = false;
                    saveToolStripButton.Enabled = false;
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void treeView_SelectedObjectsChanged(object sender, EventArgs e)
        {
            try
            {
                propertyGrid.SelectedObjects = treeView.CheckedElements;
                DrawForPreview();
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void propertyGrid_PropertyValueChanged(object sender, PropertyValueChangedEventArgs e)
        {
            try
            {
                object element = propertyGrid.SelectedObject;
                if (e.ChangedItem.Label == "ベース")
                {
                    propertyGrid.SelectedObject = null;
                    propertyGrid.SelectedObject = element;
                }
                DrawForPreview();
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }

        private void DrawForPreview()
        {
            try
            {
                object[] elements = propertyGrid.SelectedObjects;
                if (elements.Length <= 0)
                {
                    previewPanel.Refresh();
                }
                else
                {
                    KsgPage page = null;
                    if (elements[0] is KsgSubtitle)
                    {
                        page = ((KsgSubtitle)elements[0]).Pages[0];
                    }
                    else if (elements[0] is KsgPart)
                    {
                        page = ((KsgPart)elements[0]).Pages[0];
                    }
                    else if (elements[0] is KsgPage)
                    {
                        page = (KsgPage)elements[0];
                    }
                    else if (elements[0] is KsgLine)
                    {
                        page = ((KsgLine)elements[0]).ParentPage;
                    }
                    else if (elements[0] is KsgWord)
                    {
                        page = ((KsgWord)elements[0]).ParentPage;
                    }
                    else if (elements[0] is KsgCharacter)
                    {
                        page = ((KsgCharacter)elements[0]).ParentPage;
                    }
                    if (page != null)
                    {
                        page.DrawForPreview(true);
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.Show(
                    exception,
                    Assembly.GetExecutingAssembly(),
                    MethodBase.GetCurrentMethod());
            }
        }
    }
}
