﻿#ifndef KFX_TEXTDRAWER_H_INCLUDED
#define KFX_TEXTDRAWER_H_INCLUDED

#include<sstream>
#include"../../kernel/device/device_wrapper.h"
#include"../../kernel/pointer/scoped_ptr.h"
#include"../../graphics/geometory/geometory.h"
#include"../../debugutil/debug_timer.h"
#include"game_node.h"

namespace KFX{

///文字よせ設定
enum TextAlign{
	ta_left,///<左寄せ
	ta_center,///<中央揃え
	ta_right///<右寄せ
};

///テキストのカラーを保持する基底クラス
class TextColorBase{
	public:
		TextColorBase(){}
		virtual void SetRGB(int r,int g,int b)=0;
		virtual int R() const=0;
		virtual int G() const=0;
		virtual int B() const=0;
};

///フォントオブジェクトのベースとなる構造体
struct FontBase{
	int size;///<フォントサイズ
	std::wstring name;///<書体名
	bool bold;///<太字
	bool italic;///<イタリック
};



class TextDrawerFactory;
///テキストを画面上に表示するインターフェイスクラス
///@attention テキスト表示で扱う座標系は２D座標系で
///上側がマイナス、下側がプラスになっているので注意
class ITextDrawer{
	friend TextDrawerFactory;
protected:
	ITextDrawer(){};
public:
	///文字色を設定する
	///@param color 文字色
	virtual void SetTextColor(TextColorBase& color)=0;

	///フォントを設定する
	///@param font フォント
	virtual void SetFont(const FontBase& font)=0;
	
	///アライメントを設定する
	virtual void SetAlignment(TextAlign alignment)=0;

	///表示矩形を設定する
	///@param left 左端X座標
	///@param top 上端Y座標
	///@attention この座標系は２D座標で、プラスが下方向なので注意
	virtual void SetRect(int left,int top , int width, int height)=0;
	virtual void SetRect(const Rect& rect )=0;

	
	virtual void DrawText(const wchar_t* str)=0;

	///@brief テキストの表示
	///@param str 表示する文字列
	///@param x 表示X座標
	///@param y 表示Y座標
	virtual void DrawText(const wchar_t* str,int x, int y)=0;

	///@brief テキストの表示(カラー、フォント指定版)
	///@param str 表示する文字列
	///@param x 表示X座標
	///@param y 表示Y座標
	///@param color 文字色
	///@param font フォント情報
	virtual void DrawText(const wchar_t* str,int x, int y,TextColorBase& color,FontBase& font)=0;

	///矩形を計算して返す
	virtual Rect CalculateRect(const wchar_t* str)=0;
	virtual Rect CalculateRect(const wchar_t* str,FontBase& fontbase)=0;
	virtual Rect CalculateRect(const wchar_t* str,int x,int y)=0;
	virtual Rect CalculateRect(const wchar_t* str,int x,int y,FontBase& fontbase)=0;

	virtual ~ITextDrawer(){}
};

///テキスト表示クラスファクトリ
class TextDrawerFactory{
private:
	TextDrawerFactory(){}
	TextDrawerFactory(const TextDrawerFactory& );
	TextDrawerFactory& operator=(const TextDrawerFactory&);
public:
	
	static TextDrawerFactory& Instance();
	ITextDrawer* CreateTextDrawer(bool useDefault=false);
};


///文字列にて情報出力するノード
///ここもフォントをラッピングするなりして、機種依存を隠す
///@note セットされた文字列は、すぐ描画されるわけではなく
///画面更新時に一気に描画される
class TextInfoNode : public GameNode{
	private:
		ScopedPtr<ITextDrawer> _textDrawer;
		ScopedPtr<KDebugTimer> _debugTimer;
		std::wostringstream _str;
		Rect _r;
		bool _fpsFlg;
	public:
		TextInfoNode();
		void ClearText();
		///テキストの表示
		TextInfoNode& Out(const char* str);
		TextInfoNode& Out(const wchar_t* str);
		TextInfoNode& Out(const int num);
		TextInfoNode& Out(const float dec);
		TextInfoNode& Linefeed();
		void DisplayFPS(const bool flg);
		bool DisplayFPS()const;
		void Update();
};
}//end of namespace KFX

#endif