/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#define _XOPEN_SOURCE

#include <string.h>
#include <time.h>

#include "kz-hinadi.h"


static gboolean kz_hinadi_is_supported      (KzBookmarkFile *bookmark,
					     const gchar    *buf);
static gboolean kz_hinadi_parse_from_string (KzBookmarkFile *rbookmark,
					     const gchar    *buffer,
					     guint           length,
					     GError        **error);


static KzBookmarkFileType hinadi_file_type =
{
	/* .priority_hint = */ 0,
	/* .file_type     = */ "hina-di",
	/* .init          = */ NULL,
	/* .is_supported  = */ kz_hinadi_is_supported,
	/* .from_string   = */ kz_hinadi_parse_from_string,
	/* .to_string     = */ NULL,
};


KzBookmarkFileType *
kz_hinadi_get_file_types (gint idx)
{
	if (idx == 0)
		return &hinadi_file_type;
	else
		return NULL;
}


static gboolean
kz_hinadi_is_supported (KzBookmarkFile *bookmark, const gchar *buf)
{
	g_return_val_if_fail(buf, FALSE);

	if (g_str_has_prefix(buf, "HINA"))
		return TRUE;

	return FALSE;
}


static gchar *
kz_hinadi_get_encoding (gchar *src)
{
	gchar *encoding = NULL;
	gchar **lines = NULL;
	gchar *begin_pos, *end_pos, *line_end;;
	
	g_return_val_if_fail(src, NULL);

	lines = g_strsplit(src, "\r\n\r\n", 1);

	begin_pos =  strstr(lines[0], "charset=");
	if (begin_pos)
	{
		begin_pos += 8;
		line_end = strstr(begin_pos, "\r\n");
		end_pos = g_strstr_len(begin_pos, line_end - begin_pos, ";");
		if (end_pos)
			encoding = g_strndup(begin_pos, end_pos - begin_pos);
		else
			encoding = g_strndup(begin_pos, line_end - begin_pos);
	}
	else
		encoding = g_strdup("EUC-JP"); /* default value */

	g_strfreev(lines);
	return encoding;
}


static gboolean
kz_hinadi_parse_from_string (KzBookmarkFile *bookmark,
			     const gchar *buffer, guint length,
			     GError **error)
{
	gchar *dest = NULL;
	gsize read_len, dest_len;
	gchar **lines    = NULL;
	gchar **elements = NULL;
	guint line_number = 1;
	guint element_number;
	gchar timestr[20];
	gchar *encoding;

	g_return_val_if_fail(buffer, FALSE);

	/* convert to utf-8 encoding */
	encoding = kz_hinadi_get_encoding((gchar*)buffer);	
	dest = g_convert(buffer, length, "UTF-8", encoding,
			 &read_len, &dest_len, error);
	g_free(encoding);
	if (!dest)
		return FALSE;

	lines = g_strsplit(dest, "\r\n\r\n", -1);
	
	if (lines[1] == NULL)
	{
		g_strfreev(lines);
		return FALSE;
	}

	while(lines[line_number + 1] != NULL)
	{
		KzBookmark *item;
		struct tm t;
		
		elements = g_strsplit(lines[line_number], "\r\n", 20);
		element_number = 0;
		item = kz_bookmark_new();
		/* kz_bookmark_set_editable(item, FALSE); */
		while(elements[element_number+1] != NULL)
		{
			gchar *text;
			switch (elements[element_number][0])
			{
			 case 'L':
			 case 'l': /* Last-Modified */
				if(elements[element_number][13] == ':')
				{
					text = g_strchug(elements[element_number] + 15);
					strptime(text,
						 "%a, %d %b %Y %H:%M:%S %z", &t);
					kz_bookmark_set_last_modified(item, (guint)mktime(&t));
					strftime(timestr, 20, "%Y/%m/%d %H:%M", &t);
					kz_bookmark_set_description(item, timestr);
				}
				break;
			 case 'T':
			 case 't': /* Title */
				text = g_strchug(elements[element_number] + 6);
				kz_bookmark_set_title(item, text);
				break;
			 case 'U': 
			 case 'u':/* URL */
				text = g_strchug(elements[element_number] + 4);
				kz_bookmark_set_link(item, text);
				break;
			 default:
				break;
			}
			element_number++;
		}
		g_strfreev(elements);
		kz_bookmark_append(KZ_BOOKMARK(bookmark), item);
		g_object_unref(G_OBJECT(item));
		line_number++;
	}

	g_strfreev(lines);
	
	/* sorting */
	kz_bookmark_sort(KZ_BOOKMARK(bookmark),
			 KZ_BOOKMARK_SORT_LAST_MODIFIED);

	return TRUE;
}
