/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *  Copyright (C) 2004 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "prefs_history.h"

#include <string.h>
#include <glib/gi18n.h>
#include "kazehakase.h"
#include "utils/gtk-utils.h"
#include "kz-search.h"

#define DATA_KEY "KzPrefsHistory::info"


static GtkWidget     *prefs_history_create      (void);
static void           prefs_history_response    (GtkWidget *widget,
						 gint       response);


static KzPrefsWinPageEntry prefs_entry =
{
	/* .path          = */ N_("/History"),
	/* .priority_hint = */ 0,
	/* .ui_level      = */ ~KZ_UI_LEVEL_BEGINNER,
	/* .create        = */ prefs_history_create,
	/* .response      = */ prefs_history_response,
};


typedef struct _KzPrefsHistory
{
	GtkWidget     *main_vbox;
	GtkWidget     *store_cache_check;
	GtkWidget     *limit_days_check;
	GtkWidget     *limit_days_spin;
	GtkWidget     *optimize_label;
	GtkWidget     *optimize_button;
	GtkWidget     *max_results;
	GtkWidget     *num_summary;
	GtkWidget     *except_keyword;
	GtkWidget     *search_engine;
	gboolean       changed;
} KzPrefsHistory;


static void
cb_store_cache_toggled(GtkToggleButton *button, KzPrefsHistory *prefsui)
{
	gtk_widget_set_sensitive(prefsui->limit_days_check,
				 gtk_toggle_button_get_active(button));
	gtk_widget_set_sensitive(prefsui->limit_days_spin,
				 gtk_toggle_button_get_active(button));
	prefsui->changed = TRUE;
}

static void
cb_limit_days_toggled(GtkToggleButton *button, KzPrefsHistory *prefsui)
{
	gtk_widget_set_sensitive(prefsui->limit_days_spin,
				 gtk_toggle_button_get_active(button));
	prefsui->changed = TRUE;
}

static void
cb_changed (GtkWidget *widget, KzPrefsHistory *prefsui)
{
	prefsui->changed = TRUE;
}


static void
cb_optimize_button_clicked (GtkButton *button, KzPrefsHistory *prefsui)
{
	GPid pid;
	if (KZ_GET_SEARCH)
	{
		gtk_widget_set_sensitive(GTK_WIDGET(button), FALSE);
		pid = kz_search_optimize_index(KZ_GET_SEARCH);
		gtk_widget_set_sensitive(GTK_WIDGET(button), TRUE);
	}
}


static void
prefs_history_destroy (gpointer data)
{
	KzPrefsHistory *prefsui = data;

	g_free(prefsui);
}

static void
set_search_engine (GtkComboBox *combo)
{
	GDir *dir;
	gchar *search_engine_name;
	gint i = 0, active = 0;

	search_engine_name = KZ_CONF_GET_STR("History", "search_engine");
	gtk_combo_box_append_text(GTK_COMBO_BOX(combo), _("None"));
	dir = g_dir_open(KZ_SEARCH_MODULEDIR, 0, NULL);
	if (dir)
	{
		const gchar *name;
		while ((name = g_dir_read_name(dir)))
		{
			gchar *path;
			path = g_module_build_path(KZ_SEARCH_MODULEDIR, name);

			if (g_str_has_suffix(name, G_MODULE_SUFFIX) &&
			    g_file_test(path, G_FILE_TEST_EXISTS))
			{
				gchar *engine_name;
				size_t len = strlen(name);
				i++;
				engine_name = g_strndup(name+3, len - 4 - strlen(G_MODULE_SUFFIX));
				gtk_combo_box_append_text(combo, engine_name);
				if (!strcmp(engine_name, search_engine_name))
					active = i;
				g_free(engine_name);
			}
			g_free(path);
		}
		g_dir_close(dir);
	}

	if (search_engine_name)
		g_free(search_engine_name);
	gtk_combo_box_set_active(combo, active);
}

static GtkWidget *
prefs_history_create (void)
{
	KzPrefsHistory *prefsui = g_new0(KzPrefsHistory, 1);
	GtkWidget *main_vbox, *vbox, *hbox, *frame;
	GtkWidget *label, *check, *spin, *button;
	GtkWidget *entry, *combo;
	GtkAdjustment *adj;
	gboolean limit, store_cache = TRUE;
	gint limit_days, max_results, num_summary;
	gchar *keyword;

	main_vbox = gtk_vbox_new(FALSE, 0);
	prefsui->main_vbox = main_vbox;
	g_object_set_data_full(G_OBJECT(main_vbox), DATA_KEY,
			       prefsui, prefs_history_destroy);

	label = kz_prefs_ui_utils_create_title(_("History"));
	gtk_box_pack_start(GTK_BOX(main_vbox), label,
			   FALSE, FALSE, 0);
	gtk_widget_show(label);


	/*
	 *  Local Cache 
	 */
	KZ_CONF_GET("History", "store_cache", store_cache, BOOL);
	prefsui->store_cache_check = check =
		gtk_check_button_new_with_label(_("Store history cache on local disk"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check),
				     store_cache);
	g_signal_connect(check, "toggled",
			 G_CALLBACK(cb_store_cache_toggled), prefsui);
	frame = gtk_frame_new("");
	gtk_frame_set_label_widget(GTK_FRAME(frame), check);
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, FALSE, 2);
	gtk_widget_show(frame);
	gtk_widget_show(check);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
	gtk_container_add(GTK_CONTAINER(frame), vbox);
	gtk_widget_show(vbox);

	limit = TRUE;
	KZ_CONF_GET("History", "limit_cache", limit, BOOL);

	limit_days = 30;
	KZ_CONF_GET("History", "limit_days", limit_days, INT);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);
	gtk_widget_show(hbox);

	check = gtk_check_button_new_with_mnemonic(_("_Limit days of keeping the history cache"));
	prefsui->limit_days_check = check;
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), limit);
	gtk_widget_set_sensitive(check, store_cache);
	gtk_box_pack_start(GTK_BOX(hbox), check, FALSE, FALSE, 2);
	g_signal_connect(check, "toggled",
			 G_CALLBACK(cb_limit_days_toggled), prefsui);
	gtk_widget_show(check);

	prefsui->limit_days_spin = spin
		= gtk_spin_button_new_with_range(1, 1000, 1);
	adj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(spin));
	gtk_adjustment_set_value(adj, limit_days);
	gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 2);
	gtk_widget_set_sensitive(spin, limit && store_cache);
	g_signal_connect(spin, "value-changed",
			 G_CALLBACK(cb_changed), prefsui);
	gtk_widget_show(spin);

	/*
	 * optimize index
	 */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);
	gtk_widget_show(hbox);

	prefsui->optimize_label = label
		= gtk_label_new(_("Optimize history cache"));
	gtk_widget_set_sensitive(label, store_cache);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
	gtk_widget_show(label);

	prefsui->optimize_button = button
		= gtk_button_new_with_label (_("start"));
	gtk_widget_set_sensitive(button, store_cache);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 4);
	g_signal_connect(button, "clicked",
			 G_CALLBACK(cb_optimize_button_clicked), prefsui);
	gtk_widget_show(button);

	/*
	 * History Search
	 */
	frame = gtk_frame_new(_("Full-text search in history"));
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, FALSE, 2);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
	gtk_container_add(GTK_CONTAINER(frame), vbox);
	gtk_widget_show(vbox);
	
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);
	gtk_widget_show(hbox);
	label = gtk_label_new_with_mnemonic(_("Search engine name"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
	gtk_widget_show(label);

	prefsui->search_engine = combo = gtk_combo_box_new_text();
	set_search_engine(GTK_COMBO_BOX(combo));
	gtk_widget_show(combo);
	gtk_box_pack_start (GTK_BOX(hbox), combo, TRUE, TRUE, 0);
	g_signal_connect(combo, "changed",
			 G_CALLBACK(cb_changed), prefsui);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);
	gtk_widget_show(hbox);
	max_results = 20;
	KZ_CONF_GET("History", "max_results", max_results, INT);
	
	label = gtk_label_new_with_mnemonic(_("The _maximum number of shown results"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
	gtk_widget_show(label);
	prefsui->max_results = spin
		= gtk_spin_button_new_with_range(-1, 1000, 1);
	adj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(spin));
	gtk_adjustment_set_value(adj, max_results);
	gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 2);
	g_signal_connect(spin, "value-changed",
			 G_CALLBACK(cb_changed), prefsui);
	gtk_widget_show(spin);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);
	gtk_widget_show(hbox);
	num_summary = 128;
	KZ_CONF_GET("History", "num_summary", num_summary, INT);
	
	label = gtk_label_new_with_mnemonic(_("The number of words in _summary"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
	gtk_widget_show(label);
	prefsui->num_summary = spin
		= gtk_spin_button_new_with_range(-1, 1000, 1);
	adj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(spin));
	gtk_adjustment_set_value(adj, num_summary);
	gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 2);
	g_signal_connect(spin, "value-changed",
			 G_CALLBACK(cb_changed), prefsui);
	gtk_widget_show(spin);

	/* except keyword */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 4);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);
	gtk_widget_show(hbox);

	label = gtk_label_new_with_mnemonic(_("Except keyword in history search(delimiter for keyword is comma):"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
	gtk_widget_show(label);

	entry = gtk_entry_new();
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), entry);
	gtk_box_pack_start(GTK_BOX(hbox), entry, FALSE, FALSE, 2);
	gtk_widget_show(entry);

	keyword = KZ_CONF_GET_STR("History", "except_keyword");
	if(keyword)
	{
		gtk_entry_set_text(GTK_ENTRY(entry), keyword);
		g_free(keyword);
	}
	prefsui->except_keyword = entry;

	prefsui->changed = FALSE;

	return main_vbox;
}


static void
prefs_history_response (GtkWidget *widget, gint response)
{
	KzPrefsHistory *prefsui = g_object_get_data(G_OBJECT(widget), DATA_KEY);

	g_return_if_fail(prefsui);

	switch (response) {
	case GTK_RESPONSE_ACCEPT:
	case GTK_RESPONSE_APPLY:
	{
		GtkAdjustment *adj;
		gint limit_days, num_summary, max_results;
		gboolean limit, store_cache;
		const gchar *keyword;

		keyword = gtk_entry_get_text(GTK_ENTRY(prefsui->except_keyword));
		KZ_CONF_SET_STR("History", "except_keyword", keyword);
		
		if (!prefsui->changed)
			break;

		store_cache = gtk_toggle_button_get_active
				(GTK_TOGGLE_BUTTON(prefsui->store_cache_check));
		KZ_CONF_SET("History", "store_cache", store_cache, BOOL);
		
		limit = gtk_toggle_button_get_active
				(GTK_TOGGLE_BUTTON(prefsui->limit_days_check));
		KZ_CONF_SET("History", "limit_cache", limit, BOOL);

		adj = gtk_spin_button_get_adjustment
			(GTK_SPIN_BUTTON(prefsui->limit_days_spin));
		limit_days = gtk_adjustment_get_value(adj);
		KZ_CONF_SET("History", "limit_days", limit_days, INT);

		adj = gtk_spin_button_get_adjustment
			(GTK_SPIN_BUTTON(prefsui->max_results));
		max_results = gtk_adjustment_get_value(adj);
		KZ_CONF_SET("History", "max_results", max_results, INT);
		
		adj = gtk_spin_button_get_adjustment
			(GTK_SPIN_BUTTON(prefsui->num_summary));
		num_summary = gtk_adjustment_get_value(adj);
		KZ_CONF_SET("History", "num_summary", num_summary, INT);

		if (gtk_combo_box_get_active(GTK_COMBO_BOX(prefsui->search_engine)) == 0)
		{
			KZ_CONF_SET_STR("History", "search_engine", "");
		}
		else
		{
			gchar *search_engine;
			search_engine = gtk_combo_box_get_active_text
				(GTK_COMBO_BOX(prefsui->search_engine));
			if (search_engine)
			{
				KZ_CONF_SET_STR("History", "search_engine", search_engine);
				g_free(search_engine);
			}
		}

		prefsui->changed = FALSE;

		break;
	}
	case GTK_RESPONSE_REJECT:
		break;
	case KZ_RESPONSE_UI_LEVEL_MEDIUM:
		break;
	case KZ_RESPONSE_UI_LEVEL_EXPERT:
		break;
	case KZ_RESPONSE_UI_LEVEL_CUSTOM:
		break;
	default:
		break;
	}
}


KzPrefsWinPageEntry *
prefs_history_get_entry (gint idx)
{
	if (idx == 0)
		return &prefs_entry;
	else
		return NULL;
}
