/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2007 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <glib/gi18n.h>
#include <gmodule.h>

#include "kazehakase.h"
#include "glib-utils.h"
#include "kz-embed-module-impl.h"

typedef struct _KzEmbedModule KzEmbedModule;
typedef struct _KzEmbedModuleClass KzEmbedModuleClass;

struct _KzEmbedModule
{
	GTypeModule  object;
	GModule *library;

	gchar *path;

	KzEmbedModuleInitFunc init;
	KzEmbedModuleExitFunc exit;
	KzEmbedModuleCreateFunc create;
};

struct _KzEmbedModuleClass
{
	GTypeModuleClass parent_class;
};

GType kz_embed_module_get_type (void) G_GNUC_CONST;

G_DEFINE_TYPE (KzEmbedModule, kz_embed_module, G_TYPE_TYPE_MODULE)

#define KZ_TYPE_EMBED_MODULE      (kz_embed_module_get_type ())
#define KZ_EMBED_MODULE(module)   (G_TYPE_CHECK_INSTANCE_CAST ((module), KZ_TYPE_EMBED_MODULE, KzEmbedModule))

static GSList *loaded_embed;

static gboolean
kz_embed_module_load (GTypeModule *module)
{
	KzEmbedModule *s_module = KZ_EMBED_MODULE(module);
	gpointer initp, exitp, createp;

	s_module->library = g_module_open(s_module->path, G_MODULE_BIND_LAZY | G_MODULE_BIND_LOCAL);
	if (!s_module->library)
	{
		g_warning(g_module_error());
		return FALSE;
	}

	/* exact symbols from the lib */
	if (!g_module_symbol(s_module->library,
			     G_STRINGIFY(KZ_EMBED_MODULE_IMPL_INIT),
			     &initp) ||
	    !g_module_symbol(s_module->library,
			     G_STRINGIFY(KZ_EMBED_MODULE_IMPL_EXIT),
			     &exitp) ||
	    !g_module_symbol(s_module->library,
			     G_STRINGIFY(KZ_EMBED_MODULE_IMPL_CREATE),
			     &createp))
	{
		g_warning(g_module_error());
		g_module_close(s_module->library);

		return FALSE;
	}

	s_module->init   = initp;
	s_module->exit   = exitp;
	s_module->create = createp;

	s_module->init(module);

	return TRUE;
}

static void
kz_embed_module_unload (GTypeModule *module)
{
	KzEmbedModule *s_module = KZ_EMBED_MODULE(module);

	s_module->exit();

	g_module_close(s_module->library);
	s_module->library = NULL;

	s_module->init   = NULL;
	s_module->exit   = NULL;
	s_module->create = NULL;
}

static void
kz_embed_module_finalize (GObject *object)
{
	KzEmbedModule *module = KZ_EMBED_MODULE(object);

	g_free(module->path);

	G_OBJECT_CLASS(kz_embed_module_parent_class)->finalize(object);
}
static void
kz_embed_module_class_init (KzEmbedModuleClass *klass)
{
	GTypeModuleClass *module_class = G_TYPE_MODULE_CLASS(klass);
	GObjectClass *gobject_class = G_OBJECT_CLASS(klass);

	module_class->load   = kz_embed_module_load;
	module_class->unload = kz_embed_module_unload;

	gobject_class->finalize = kz_embed_module_finalize;
}

static void
kz_embed_module_init (KzEmbedModule *module)
{
}

static GtkWidget *
kz_embed_module_create (KzEmbedModule *module, const gchar *url)
{
	GtkWidget *embed = NULL;
	if (g_type_module_use(G_TYPE_MODULE(module)))
	{
		embed = module->create(url);
		g_type_module_unuse(G_TYPE_MODULE(module));
		return embed;
	}
	return embed;
}

GtkWidget *
kz_embed_new (const gchar *name, const gchar *url)
{
	GSList *l;
	gchar *module_path;
	KzEmbedModule *module;
	GtkWidget *embed = NULL;

	for (l = loaded_embed; l; l = l->next)
	{
		module = l->data;

		if (strcmp(G_TYPE_MODULE (module)->name, name) == 0)
			return kz_embed_module_create(module, url);
	}

	if (g_module_supported())
	{
		module_path = g_module_build_path(KZ_EMBED_MODULEDIR, name);

		if (module_path)
		{
			module = g_object_new(KZ_TYPE_EMBED_MODULE, NULL);

			g_type_module_set_name(G_TYPE_MODULE(module), name);
			module->path = g_strdup(module_path);

			loaded_embed = g_slist_prepend(loaded_embed, module);

			embed = kz_embed_module_create(module, url);
			g_free(module_path);
		}
	}

	return embed;
}

