/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2009 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <dbus/dbus-glib-bindings.h>

#include "kz-dbus-embed-agent.h"
#include "kz-marshalers.h"
#include "kz-embed.h"

enum {
    PROP_0,
    PROP_SOCKET_ADDRESS,
    PROP_CONNECTION,
    PROP_PROCESS_ID
};

enum {
    STOP_LOAD,
    RELOAD,
    LOAD_URI,
    GO_BACK,
    GO_FORWARD,
    COPY,
    CUT,
    PASTE,
    FIND,
    ZOOM,
    LAST_SIGNAL
};

static gint signals[LAST_SIGNAL] = {0};

typedef struct _KzDBusEmbedAgentPrivate	KzDBusEmbedAgentPrivate;
struct _KzDBusEmbedAgentPrivate
{
    gchar *socket_address;
    DBusGConnection *connection;
    GPid process_id;
    guint watch_id;
};

#define KZ_DBUS_EMBED_AGENT_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), KZ_TYPE_DBUS_EMBED_AGENT, KzDBusEmbedAgentPrivate))

static GObject *constructor  (GType type,
                              guint n_props,
                              GObjectConstructParam *props);
static void     dispose      (GObject      *object);
static void     set_property (GObject      *object,
                              guint         prop_id,
                              const GValue *value,
                              GParamSpec   *pspec);
static void     get_property (GObject      *object,
                              guint         prop_id,
                              GValue       *value,
                              GParamSpec   *pspec);

static void         load_uri         (KzEmbed      *embed,
                                      const gchar  *uri);
static void         reload           (KzEmbed      *embed,
                                      KzEmbedReloadFlag flags);
static void         stop_load        (KzEmbed      *embed);
static void         go_back          (KzEmbed      *embed);
static void         go_forward       (KzEmbed      *embed);
static void         cut              (KzEmbed      *embed);
static void         copy             (KzEmbed      *embed);
static void         paste            (KzEmbed      *embed);
static gboolean     find             (KzEmbed      *embed,
                                      const gchar  *keyword,
                                      gboolean      backward);
static void         zoom             (KzEmbed      *embed,
                                      gdouble       zoom);

static gboolean kz_dbus_embed_agent_ready     (KzDBusEmbedAgent *object, guint32 process_id, gchar **address, guint32 *id, gchar **engine_name, GError **error);
static gboolean kz_dbus_embed_agent_location  (KzDBusEmbedAgent *object, gchar *location, GError **error);
static gboolean kz_dbus_embed_agent_title     (KzDBusEmbedAgent *object, gchar *title, GError **error);
static gboolean kz_dbus_embed_agent_net_start (KzDBusEmbedAgent *object, GError **error);
static gboolean kz_dbus_embed_agent_net_stop  (KzDBusEmbedAgent *object, GError **error);
static gboolean kz_dbus_embed_agent_progress  (KzDBusEmbedAgent *object, gdouble ratio, GError **error);
static gboolean kz_dbus_embed_agent_dom_mouse_up (KzDBusEmbedAgent *object,
                                            gint event_context,
                                            gchar *link_uri,
                                            gchar *link_text,
                                            gchar *image_uri,
                                            gchar *frame_src,
                                            gint button,
                                            gint modifier,
                                            gint x,
                                            gint y,
                                            gboolean *ret,
                                            GError **error);
static gboolean kz_dbus_embed_agent_dom_mouse_down (KzDBusEmbedAgent *object,
                                              gint event_context,
                                              gchar *link_uri,
                                              gchar *link_text,
                                              gchar *image_uri,
                                              gchar *frame_src,
                                              gint button,
                                              gint modifier,
                                              gint x,
                                              gint y,
                                              gboolean *ret,
                                              GError **error);
static gboolean kz_dbus_embed_agent_dom_mouse_click (KzDBusEmbedAgent *object,
                                              gint event_context,
                                              gchar *link_uri,
                                              gchar *link_text,
                                              gchar *image_uri,
                                              gchar *frame_src,
                                              gint button,
                                              gint modifier,
                                              gint x,
                                              gint y,
                                              gboolean *ret,
                                              GError **error);

#include "kz-dbus-embed-agent-server-bindings.h"

static void kz_embed_iface_init   (KzEmbedIFace *iface);

G_DEFINE_TYPE_EXTENDED(KzDBusEmbedAgent, kz_dbus_embed_agent, GTK_TYPE_SOCKET, 0, G_IMPLEMENT_INTERFACE(KZ_TYPE_EMBED, kz_embed_iface_init));

static guint32
kz_dbus_embed_agent_get_embed_id(KzDBusEmbedAgent *agent)
{
    return gtk_socket_get_id(GTK_SOCKET(agent));
}

static void
child_watch_func (GPid pid, gint status, gpointer data)
{
    KzDBusEmbedAgentPrivate *priv = KZ_DBUS_EMBED_AGENT_GET_PRIVATE(data);

    if (WIFSIGNALED(status)) {
    } else if (WIFEXITED(status)){
    } else {
    }

    priv->watch_id = 0;
    priv->process_id = -1;
}

static void
spawn_kz_embed_process (KzDBusEmbedAgent *agent)
{
    gboolean success;
    GError *error = NULL;
    gchar *argv[2] = {0};
    KzDBusEmbedAgentPrivate *priv = KZ_DBUS_EMBED_AGENT_GET_PRIVATE(agent);

    argv[0] = "kz-embed-process";
    argv[1] = priv->socket_address;

    success = g_spawn_async(NULL,
                            argv,
                            NULL,
                            G_SPAWN_DO_NOT_REAP_CHILD,
                            NULL,
                            NULL,
                            &priv->process_id,
                            &error);
    if (error) {
        g_print("Error: %s\n", error->message);
        g_error_free(error);
    }
    priv->watch_id = g_child_watch_add(priv->process_id, child_watch_func, agent);
}

static gboolean
kz_dbus_embed_agent_ready (KzDBusEmbedAgent *object,
                           guint32 process_id,
                           gchar **address,
                           guint32 *embed_id,
                           gchar **engine_name,
                           GError **error)
{
    *engine_name = g_strdup("webkit_gtk");
    *embed_id = kz_dbus_embed_agent_get_embed_id(object);

    return TRUE;
}

static gboolean
kz_dbus_embed_agent_location (KzDBusEmbedAgent *object, gchar *location, GError **error)
{
    g_signal_emit_by_name(object, "kz-location", location);
    return TRUE;
}

static gboolean
kz_dbus_embed_agent_title (KzDBusEmbedAgent *object, gchar *title, GError **error)
{
    g_signal_emit_by_name(object, "kz-title", title);
    return TRUE;
}

static gboolean
kz_dbus_embed_agent_net_start (KzDBusEmbedAgent *object, GError **error)
{
    g_signal_emit_by_name(object, "kz-net-start");
    return TRUE;
}

static gboolean
kz_dbus_embed_agent_net_stop (KzDBusEmbedAgent *object, GError **error)
{
    g_signal_emit_by_name(object, "kz-net-stop");
    return TRUE;
}

static gboolean
kz_dbus_embed_agent_progress (KzDBusEmbedAgent *object, gdouble ratio, GError **error)
{
    g_signal_emit_by_name(KZ_EMBED(object), "kz-progress", ratio);
    return TRUE;
}

static KzEmbedEventMouse *
create_kz_embed_event_mouse (gint event_context,
                             gchar *link_uri,
                             gchar *link_text,
                             gchar *image_uri,
                             gchar *frame_src,
                             gint button,
                             gint modifier,
                             gint x,
                             gint y)
{
    KzEmbedEventMouse *kzevent;

    kzevent = (KzEmbedEventMouse *) kz_embed_event_new(KZ_EMBED_EVENT_MOUSE);
    kzevent->cinfo.context = event_context;
    kzevent->cinfo.link = g_strdup(link_uri);
    kzevent->cinfo.linktext = g_strdup(link_text);
    kzevent->cinfo.img = g_strdup(image_uri);
    kzevent->cinfo.frame_src = g_strdup(frame_src);
    kzevent->button = button;
    kzevent->modifier = modifier;
    kzevent->x = x;
    kzevent->y = y;

    return kzevent;
}

static gboolean
emit_dom_mouse_event (KzDBusEmbedAgent *object,
                      const gchar *signal_name,
                      gint event_context,
                      gchar *link_uri,
                      gchar *link_text,
                      gchar *image_uri,
                      gchar *frame_src,
                      gint button,
                      gint modifier,
                      gint x,
                      gint y)
{
    KzEmbedEventMouse *kzevent;
    gboolean ret;

    kzevent = create_kz_embed_event_mouse(event_context,
                                          link_uri,
                                          link_text,
                                          image_uri,
                                          frame_src,
                                          button,
                                          modifier,
                                          x,
                                          y);
    g_signal_emit_by_name(object, signal_name, kzevent, &ret);
    kz_embed_event_free((KzEmbedEvent *) kzevent);
    return ret;
}

static gboolean
kz_dbus_embed_agent_dom_mouse_up (KzDBusEmbedAgent *object,
                                  gint event_context,
                                  gchar *link_uri,
                                  gchar *link_text,
                                  gchar *image_uri,
                                  gchar *frame_src,
                                  gint button,
                                  gint modifier,
                                  gint x,
                                  gint y,
                                  gboolean *ret,
                                  GError **error)
{
    *ret = emit_dom_mouse_event(object,
                                "kz-dom-mouse-up",
                                event_context,
                                link_uri,
                                link_text,
                                image_uri,
                                frame_src,
                                button,
                                modifier,
                                x,
                                y);
    return TRUE;
}

static gboolean
kz_dbus_embed_agent_dom_mouse_down (KzDBusEmbedAgent *object,
                              gint event_context,
                              gchar *link_uri,
                              gchar *link_text,
                              gchar *image_uri,
                              gchar *frame_src,
                              gint button,
                              gint modifier,
                              gint x,
                              gint y,
                              gboolean *ret,
                              GError **error)
{
    *ret = emit_dom_mouse_event(object,
                                "kz-dom-mouse-down",
                                event_context,
                                link_uri,
                                link_text,
                                image_uri,
                                frame_src,
                                button,
                                modifier,
                                x,
                                y);
    return TRUE;
}

static gboolean
kz_dbus_embed_agent_dom_mouse_click (KzDBusEmbedAgent *object,
                            gint event_context,
                            gchar *link_uri,
                            gchar *link_text,
                            gchar *image_uri,
                            gchar *frame_src,
                            gint button,
                            gint modifier,
                            gint x,
                            gint y,
                            gboolean *ret,
                            GError **error)
{
    *ret = emit_dom_mouse_event(object,
                                "kz-dom-mouse-click",
                                event_context,
                                link_uri,
                                link_text,
                                image_uri,
                                frame_src,
                                button,
                                modifier,
                                x,
                                y);
    return TRUE;
}

static void
kz_embed_iface_init (KzEmbedIFace *iface)
{
    iface->load_uri               = load_uri;
    iface->view_source            = NULL;
    iface->view_current_page_source_in_new 
                                  = NULL;
    iface->can_cut_selection      = NULL; 
    iface->can_copy_selection     = NULL; 
    iface->can_paste              = NULL; 
    iface->cut_selection          = cut;
    iface->copy_selection         = copy;
    iface->paste                  = paste;
    iface->select_all             = NULL;
    iface->get_selection_string   = NULL;
    iface->find                   = find;
    iface->incremental_search     = NULL;
    iface->selection_is_collapsed = NULL;
    iface->get_links              = NULL;
    iface->copy_page              = NULL; 
    iface->shistory_get_pos       = NULL; 
    iface->shistory_get_nth       = NULL; 
    iface->reload                 = reload;
    iface->stop_load              = stop_load;
    iface->go_back                = go_back;
    iface->go_forward             = go_forward;
    iface->can_go_back            = NULL; 
    iface->can_go_forward         = NULL; 
    iface->can_go_nav_link        = NULL;
    iface->go_nav_link            = NULL;
    iface->append_nav_link        = NULL;
    iface->set_nav_link           = NULL;
    iface->set_nth_nav_link       = NULL;
    iface->get_nav_link           = NULL;
    iface->get_nth_nav_link       = NULL;
    iface->get_nav_links          = NULL;
    iface->go_history_index       = NULL;
    iface->get_body_text          = NULL;
#if 0
    iface->get_selection_source   = NULL;
#endif
    iface->set_encoding           = NULL; 
    iface->get_encoding           = NULL; 
    iface->print                  = NULL; 
    iface->print_preview          = NULL; 
    iface->get_printer_list       = NULL;
    iface->create_thumbnail       = NULL;
    iface->save_with_content      = NULL;
    iface->set_text_into_textarea = NULL;
    iface->get_text_from_textarea = NULL;
    iface->zoom                   = zoom;
    iface->get_zoom_ratio         = NULL;
    iface->set_text_size          = NULL;
    iface->get_text_size          = NULL;
    iface->get_html_with_contents = NULL;
    iface->set_history            = NULL;
    iface->get_history            = NULL; 
    iface->get_last_modified      = NULL;
    iface->fine_scroll            = NULL;
    iface->page_up                = NULL;
    iface->page_down              = NULL; 
    iface->get_allow_javascript   = NULL;
    iface->set_allow_javascript   = NULL; 
    iface->get_allow_images       = NULL;
    iface->set_allow_images       = NULL;
}

static void
kz_dbus_embed_agent_class_init (KzDBusEmbedAgentClass *klass)
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS(klass);
    object_class->constructor = constructor;
    object_class->dispose = dispose;
    object_class->set_property = set_property;
    object_class->get_property = get_property;

    signals[STOP_LOAD] = g_signal_new("stop-load",
                                      G_TYPE_FROM_CLASS(klass),
                                      G_SIGNAL_RUN_LAST,
                                      0,
                                      NULL, NULL,
                                      g_cclosure_marshal_VOID__VOID,
                                      G_TYPE_NONE, 0,
                                      G_TYPE_NONE);
    signals[RELOAD] = g_signal_new("reload",
                                   G_TYPE_FROM_CLASS(klass),
                                   G_SIGNAL_RUN_LAST,
                                   0,
                                   NULL, NULL,
                                   g_cclosure_marshal_VOID__VOID,
                                   G_TYPE_NONE, 0,
                                   G_TYPE_NONE);
    signals[LOAD_URI] = g_signal_new("load-uri",
                                     G_TYPE_FROM_CLASS(klass),
                                     G_SIGNAL_RUN_LAST,
                                     0,
                                     NULL, NULL,
                                     g_cclosure_marshal_VOID__STRING,
                                     G_TYPE_NONE, 1,
                                     G_TYPE_STRING);
    signals[GO_BACK] = g_signal_new("go-back",
                                    G_TYPE_FROM_CLASS(klass),
                                    G_SIGNAL_RUN_LAST,
                                    0,
                                    NULL, NULL,
                                    g_cclosure_marshal_VOID__VOID,
                                    G_TYPE_NONE, 0,
                                    G_TYPE_NONE);
    signals[GO_FORWARD] = g_signal_new("go-forward",
                                       G_TYPE_FROM_CLASS(klass),
                                       G_SIGNAL_RUN_LAST,
                                       0,
                                       NULL, NULL,
                                       g_cclosure_marshal_VOID__VOID,
                                       G_TYPE_NONE, 0,
                                       G_TYPE_NONE);
    signals[COPY] = g_signal_new("copy",
                                 G_TYPE_FROM_CLASS(klass),
                                 G_SIGNAL_RUN_LAST,
                                 0,
                                 NULL, NULL,
                                 g_cclosure_marshal_VOID__VOID,
                                 G_TYPE_NONE, 0,
                                 G_TYPE_NONE);
    signals[CUT] = g_signal_new("cut",
                                G_TYPE_FROM_CLASS(klass),
                                G_SIGNAL_RUN_LAST,
                                0,
                                NULL, NULL,
                                g_cclosure_marshal_VOID__VOID,
                                G_TYPE_NONE, 0,
                                G_TYPE_NONE);
    signals[PASTE] = g_signal_new("paste",
                                  G_TYPE_FROM_CLASS(klass),
                                  G_SIGNAL_RUN_LAST,
                                  0,
                                  NULL, NULL,
                                  g_cclosure_marshal_VOID__VOID,
                                  G_TYPE_NONE, 0,
                                  G_TYPE_NONE);
    signals[ZOOM] = g_signal_new("zoom",
                                 G_TYPE_FROM_CLASS(klass),
                                 G_SIGNAL_RUN_LAST,
                                 0,
                                 NULL, NULL,
                                 g_cclosure_marshal_VOID__DOUBLE,
                                 G_TYPE_NONE, 1,
                                 G_TYPE_DOUBLE);
    signals[FIND] = g_signal_new("find",
                                 G_TYPE_FROM_CLASS(klass),
                                 G_SIGNAL_RUN_LAST,
                                 0,
                                 NULL, NULL,
                                 _kz_marshal_VOID__STRING_BOOLEAN,
                                 G_TYPE_NONE, 2,
                                 G_TYPE_STRING,
                                 G_TYPE_BOOLEAN);

    g_object_class_install_property
        (object_class,
         PROP_SOCKET_ADDRESS,
         g_param_spec_string("socket-address",
             _("Socket Address"),
             _("Socket address"),
             NULL,
             G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
    g_object_class_install_property
        (object_class,
         PROP_CONNECTION,
         g_param_spec_pointer("connection",
             _("Connection"),
             _("Connection"),
             G_PARAM_READWRITE));
    g_object_class_install_property
        (object_class,
         PROP_PROCESS_ID,
         g_param_spec_uint("process-id",
             _("Process ID"),
             _("The ID of embed process"),
             0, G_MAXUINT,
             0,
             G_PARAM_READWRITE));

    dbus_g_object_type_install_info(KZ_TYPE_DBUS_EMBED_AGENT, &dbus_glib_kz_dbus_embed_agent_object_info);
	g_type_class_add_private(object_class, sizeof(KzDBusEmbedAgentPrivate));
}

static GObject*
constructor (GType                  type,
             guint                  n_props,
             GObjectConstructParam *props)
{
	GObject *object;
	GObjectClass *klass = G_OBJECT_CLASS(kz_dbus_embed_agent_parent_class);

	object = klass->constructor(type, n_props, props);

    spawn_kz_embed_process(KZ_DBUS_EMBED_AGENT(object));

	return object;
}

static void
dispose (GObject *object)
{
    KzDBusEmbedAgentPrivate *priv = KZ_DBUS_EMBED_AGENT_GET_PRIVATE(object);

    if (priv->connection) {
        dbus_g_connection_unref(priv->connection);
        priv->connection = NULL;
    }
    if (priv->process_id > 0) {
        kill(priv->process_id, SIGKILL);
        g_spawn_close_pid(priv->process_id);
        priv->process_id = -1;
    }
    if (priv->watch_id > 0) {
        g_source_remove(priv->watch_id);
        priv->watch_id = 0;
    }

    g_free(priv->socket_address);
    priv->socket_address = NULL;
}

static void
set_property (GObject         *object,
              guint            prop_id,
              const GValue    *value,
              GParamSpec      *pspec)
{
    KzDBusEmbedAgentPrivate *priv = KZ_DBUS_EMBED_AGENT_GET_PRIVATE(object);

    switch (prop_id)
    {
    case PROP_SOCKET_ADDRESS:
        priv->socket_address = g_value_dup_string(value);
        break;
    case PROP_CONNECTION:
        priv->connection = dbus_g_connection_ref(g_value_get_pointer(value));
        break;
    case PROP_PROCESS_ID:
        priv->process_id = g_value_get_uint(value);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}


static void
get_property (GObject         *object,
              guint            prop_id,
              GValue          *value,
              GParamSpec      *pspec)
{
    KzDBusEmbedAgentPrivate *priv = KZ_DBUS_EMBED_AGENT_GET_PRIVATE(object);

    switch (prop_id)
    {
    case PROP_SOCKET_ADDRESS:
        g_value_set_string(value, priv->socket_address);
        break;
    case PROP_CONNECTION:
        g_value_set_pointer(value, priv->connection);
        break;
    case PROP_PROCESS_ID:
        g_value_set_uint(value, priv->process_id);
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
kz_dbus_embed_agent_init (KzDBusEmbedAgent *object)
{
    KzDBusEmbedAgentPrivate *priv = KZ_DBUS_EMBED_AGENT_GET_PRIVATE(object);

    priv->socket_address = NULL;
    priv->connection = NULL;
    priv->process_id = -1;
}

GtkWidget *
kz_dbus_embed_agent_new (const gchar *socket_address)
{
    return GTK_WIDGET(g_object_new(KZ_TYPE_DBUS_EMBED_AGENT,
                                   "socket-address", socket_address,
                                   NULL));
}

static void
load_uri (KzEmbed *embed, const gchar *uri)
{
    g_signal_emit(embed, signals[LOAD_URI], 0, uri);
}

static void
stop_load (KzEmbed *embed)
{
    g_signal_emit(embed, signals[STOP_LOAD], 0);
}

static void
reload (KzEmbed *embed, KzEmbedReloadFlag flags)
{
    g_signal_emit(embed, signals[RELOAD], 0);
}

static void
go_back (KzEmbed *embed)
{
    g_signal_emit(embed, signals[GO_BACK], 0);
}

static void
go_forward (KzEmbed *embed)
{
    g_signal_emit(embed, signals[GO_FORWARD], 0);
}

static void
copy (KzEmbed *embed)
{
    g_signal_emit(embed, signals[COPY], 0);
}

static void
cut (KzEmbed *embed)
{
    g_signal_emit(embed, signals[CUT], 0);
}

static void
paste (KzEmbed *embed)
{
    g_signal_emit(embed, signals[PASTE], 0);
}

static gboolean
find (KzEmbed *embed, const gchar *keyword, gboolean backward)
{
    g_signal_emit(embed, signals[FIND], 0, keyword, FALSE);
    return FALSE;
}

static void
zoom (KzEmbed *embed, gdouble zoom)
{
    g_signal_emit(embed, signals[ZOOM], 0, zoom);
}

void
kz_dbus_embed_agent_set_connection (KzDBusEmbedAgent *agent,
                                    DBusGConnection  *connection)
{
    KzDBusEmbedAgentPrivate *priv = KZ_DBUS_EMBED_AGENT_GET_PRIVATE(agent);
    priv->connection = dbus_g_connection_ref(connection);
}

/* 
vi:ts=4:nowrap:ai:expandtab:sw=4
*/
