/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-linksdialog.h"

#include <stdio.h>
#include <gdk/gdkkeysyms.h>

#include "intl.h"
#include "gobject-utils.h"
#include "kz-link.h"
#include "mozilla.h"

enum {
	RESPONSE_SAVE
};

static void     kz_links_dialog_class_init         (KzLinksDialogClass *klass);
static void     kz_links_dialog_init               (KzLinksDialog  *kzlinks);
static void     kz_links_dialog_dispose            (GObject        *object);
static void     kz_links_dialog_response           (GtkDialog      *dialog,
						    gint            arg);

static void     cb_parent_destroy                  (GtkWidget      *widget,
						    KzLinksDialog  *kzlinks);
static void     tree_sel_open_selected             (GtkTreeModel   *model,
						    GtkTreePath    *path,
						    GtkTreeIter    *iter,
						    gpointer        data);
static gboolean cb_tree_view_key_press             (GtkWidget      *widget,
						    GdkEventKey    *event,
						    KzLinksDialog  *kzlinks);
static gboolean cb_tree_view_button_press          (GtkWidget      *widget,
						    GdkEventButton *event,
						    KzLinksDialog  *kzlinks);
 
static GtkDialogClass *parent_class = NULL;


KZ_OBJECT_GET_TYPE(kz_links_dialog, "KzLinksDialog", KzLinksDialog,
		   kz_links_dialog_class_init, kz_links_dialog_init,
		   GTK_TYPE_DIALOG)

static void
kz_links_dialog_class_init (KzLinksDialogClass *klass)
{
	GObjectClass *gobject_class;
	GtkDialogClass *dialog_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *) klass;
	dialog_class  = (GtkDialogClass *) klass;

	/* GtkObject signals */
	gobject_class->dispose = kz_links_dialog_dispose;

	/* GtkDialog signals */
	dialog_class->response = kz_links_dialog_response;
}

static void
kz_links_dialog_init (KzLinksDialog *kzlinks)
{
	GtkWidget *main_vbox, *scrwin, *tree_view;
	GtkListStore *store;
	GtkCellRenderer *cell;
	GtkTreeViewColumn *col;
	GtkTreeSelection *selection;

	kzlinks->kz      = NULL;
	kzlinks->kzembed = NULL;

	main_vbox = GTK_DIALOG(kzlinks)->vbox;
	gtk_window_set_default_size(GTK_WINDOW(kzlinks), 600, 450);

	gtk_dialog_add_buttons(GTK_DIALOG(kzlinks),
			       GTK_STOCK_OPEN,  GTK_RESPONSE_YES,
			       GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
			       GTK_STOCK_SAVE,  RESPONSE_SAVE,
			       NULL);

	/* scrolled window */
	scrwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
        gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrwin),
					    GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(main_vbox), scrwin, TRUE, TRUE, 0);
	gtk_widget_show(scrwin);

	/* tree view */
	store = gtk_list_store_new(2, G_TYPE_STRING, G_TYPE_STRING);
	tree_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree_view), TRUE);
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);

	cell = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new_with_attributes
			("Title", cell, "text", 0, NULL);
	gtk_tree_view_column_set_sizing(col, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_fixed_width (col, 200);
	gtk_tree_view_column_set_resizable(col, TRUE);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), col);

	cell = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new_with_attributes
			("URL", cell, "text", 1, NULL);
	gtk_tree_view_column_set_sizing(col, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_fixed_width (col, 400);
	gtk_tree_view_column_set_resizable(col, TRUE);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), col);

	g_signal_connect(G_OBJECT(tree_view), "key-press-event",
			 G_CALLBACK(cb_tree_view_key_press), kzlinks);
	g_signal_connect(G_OBJECT(tree_view), "button-press-event",
			 G_CALLBACK(cb_tree_view_button_press), kzlinks);

	gtk_container_add(GTK_CONTAINER(scrwin), tree_view);
	gtk_widget_show(tree_view);

	kzlinks->scrolled_window = scrwin;
	kzlinks->tree_view       = GTK_TREE_VIEW(tree_view);
	kzlinks->list_store      = store;
}


static void
kz_links_dialog_dispose (GObject *object)
{
	KzLinksDialog *kzlinks = KZ_LINKS_DIALOG(object);

	if (kzlinks->kz)
	{
		g_signal_handlers_disconnect_by_func
			(G_OBJECT(kzlinks->kz),
			 G_CALLBACK(cb_parent_destroy),
			 kzlinks);
		kzlinks->kz = NULL;
	}

	if (kzlinks->kzembed)
	{
		g_signal_handlers_disconnect_by_func
			(G_OBJECT(kzlinks->kzembed),
			 G_CALLBACK(cb_parent_destroy),
			 kzlinks);
		kzlinks->kzembed = NULL;
	}

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}


GtkWidget *
kz_links_dialog_new (KzWindow *kz, KzMozEmbed *kzembed, gboolean selected_only)
{
	GtkWidget *widget;
	KzLinksDialog *kzlinks;
	GList *list = NULL, *node;
	gboolean result;
	gchar buf[256];

	kzlinks = KZ_LINKS_DIALOG(g_object_new(KZ_TYPE_LINKS_DIALOG, NULL));
	widget = GTK_WIDGET(kzlinks);

	g_return_val_if_fail(KZ_IS_WINDOW(kz), widget);
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), widget);

	kzlinks->kz      = kz;
	kzlinks->kzembed = kzembed;

	g_signal_connect(G_OBJECT(kz), "destroy",
			 G_CALLBACK(cb_parent_destroy), kzlinks);
	g_signal_connect(G_OBJECT(kzembed), "destroy",
			 G_CALLBACK(cb_parent_destroy), kzlinks);


	g_snprintf(buf, sizeof(buf), _("Extracted links  - %s"),
		   kz_moz_embed_get_title(kzembed));
	gtk_window_set_title(GTK_WINDOW(kzlinks), buf);

	result = kz_moz_embed_get_links(kzembed, &list, selected_only);
	if (!result || !list) return widget;

	for (node = list; node; node = g_list_next(node))
	{
		KzLink *kzlink = node->data;
		GtkTreeIter iter;

		if (!kzlink->url || !*kzlink->url) continue;

		if (kz_link_filter(kzlink)) continue;

		gtk_list_store_append(kzlinks->list_store, &iter);
		gtk_list_store_set(kzlinks->list_store, &iter,
				   0, kzlink->title,
				   1, kzlink->url,
				   -1);
	}

	g_list_foreach(list, (GFunc) kz_link_unref, NULL);
	g_list_free(list);

	return widget;
}


static gchar *last_saved_file = NULL;


static void
kz_links_dialog_save (KzLinksDialog *kzlinks, const gchar *filename)
{
	FILE *fp;
	GtkTreeModel *model = GTK_TREE_MODEL(kzlinks->list_store);
	GtkTreeIter iter;
	gboolean exist;

	fp = fopen(filename, "wt");
	if (!fp)
	{
		g_warning("Can't open %s for write!", filename);
#warning FIXME! Show error dialog!
		return;
	}

	exist = gtk_tree_model_get_iter_first(model, &iter);
	for (; exist; exist = gtk_tree_model_iter_next(model, &iter))
	{
		gchar *url;

		gtk_tree_model_get(model, &iter,
				   1, &url,
				   -1);
		if (!url) continue;
		if (!*url)
		{
			g_free(url);
			continue;
		}

		fputs(url, fp);
		fputc('\n', fp);
		g_free(url);
	}

	fclose(fp);
}


static void
cb_filesel_response (GtkDialog *dialog, gint arg, KzLinksDialog *kzlinks)
{
	GtkFileSelection *filesel = GTK_FILE_SELECTION(dialog);
	const gchar *file;

	file = gtk_file_selection_get_filename(filesel);

	switch (arg) {
	case GTK_RESPONSE_OK:
		if (file && *file)
			kz_links_dialog_save(kzlinks, file);
	case GTK_RESPONSE_CANCEL:
		if (file && *file)
			g_free(last_saved_file);
		last_saved_file = g_strdup(file);
		gtk_widget_destroy(GTK_WIDGET(dialog));
		break;
	default:
		break;
	}
}


static void
kz_links_dialog_save_dialog (KzLinksDialog *kzlinks)
{
	GtkWidget *filesel;

	filesel = gtk_file_selection_new (NULL);
	if (last_saved_file)
		gtk_file_selection_set_filename(GTK_FILE_SELECTION(filesel),
						last_saved_file);
	gtk_window_set_transient_for(GTK_WINDOW(filesel),
				     GTK_WINDOW(kzlinks));
	g_signal_connect(G_OBJECT(filesel), "response",
			 G_CALLBACK(cb_filesel_response), kzlinks);
	g_signal_connect(G_OBJECT(filesel), "destroy",
			 G_CALLBACK(gtk_main_quit), NULL);
	gtk_grab_add(GTK_WIDGET(filesel));
	gtk_widget_show(filesel);
	gtk_main();
}


static void
kz_links_dialog_response (GtkDialog *dialog, gint arg)
{
	KzLinksDialog *kzlinks;
	GtkTreeSelection *selection;

	g_return_if_fail (KZ_LINKS_DIALOG(dialog));

	kzlinks = KZ_LINKS_DIALOG(dialog);
	if (!kzlinks->kz || !kzlinks->kzembed) return;

	switch (arg) {
	case GTK_RESPONSE_YES:
		selection = gtk_tree_view_get_selection(kzlinks->tree_view);
		gtk_tree_selection_selected_foreach
			(selection, tree_sel_open_selected, kzlinks);
		break;
	case GTK_RESPONSE_CLOSE:
		gtk_widget_destroy(GTK_WIDGET(dialog));
		break;
	case RESPONSE_SAVE:
		kz_links_dialog_save_dialog(kzlinks);
		break;
	default:
		break;
	}
}


static void
cb_parent_destroy(GtkWidget *widget, KzLinksDialog *kzlinks)
{
	g_return_if_fail (KZ_IS_LINKS_DIALOG(kzlinks));
	gtk_widget_destroy(GTK_WIDGET(kzlinks));
}


static void
tree_sel_open_selected (GtkTreeModel *model,
			GtkTreePath *path, GtkTreeIter *iter,
			gpointer data)
{
	KzLinksDialog *kzlinks = data;
	gchar *url;

	g_return_if_fail(KZ_IS_LINKS_DIALOG(kzlinks));

	gtk_tree_model_get(GTK_TREE_MODEL(kzlinks->list_store),
			   iter, 1, &url, -1);

	if (url && *url)
		kz_window_open_new_tab_with_parent(kzlinks->kz, url,
						   GTK_WIDGET(kzlinks->kzembed));

	g_free(url);
}


static gboolean
cb_tree_view_button_press (GtkWidget *widget, GdkEventButton *event,
			   KzLinksDialog *kzlinks)
{
	GtkTreeSelection *selection;

	g_return_val_if_fail(GTK_IS_TREE_VIEW(widget), FALSE);
	g_return_val_if_fail(KZ_IS_LINKS_DIALOG(kzlinks), FALSE);

	if (event->type == GDK_2BUTTON_PRESS)
	{
		selection = gtk_tree_view_get_selection(kzlinks->tree_view);
		gtk_tree_selection_selected_foreach
			(selection, tree_sel_open_selected, kzlinks);
		return TRUE;
	}

	return FALSE;
}


static gboolean
cb_tree_view_key_press (GtkWidget *widget, GdkEventKey *event,
			KzLinksDialog *kzlinks)
{
	GtkTreeSelection *selection;

	g_return_val_if_fail(GTK_IS_TREE_VIEW(widget), FALSE);
	g_return_val_if_fail(KZ_IS_LINKS_DIALOG(kzlinks), FALSE);

	switch (event->keyval) {
	case GDK_Return:
	case GDK_space:
		selection = gtk_tree_view_get_selection(kzlinks->tree_view);
		gtk_tree_selection_selected_foreach
			(selection, tree_sel_open_selected, kzlinks);
		return TRUE;
		break;
	default:
		break;
	}

	return FALSE;
}
