/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2004 Hiroyuki Ikezoe
 *  Copyright (C) 2003 - 2004 Takuro Ashie
 *  Copyright (C) 2004 Hidetaka Iwai
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-smart-bookmark-action.h"

#include <stdlib.h>
#include <gdk/gdkkeysyms.h>
#include <sys/types.h>
#include <regex.h>

#include "gtk24backports.h"

#include "kazehakase.h"
#include "intl.h"
#include "utils/utils.h"
#include "gobject-utils.h"
#include "kz-mozembed.h"
#include "kz-entry.h"

enum {
	PROP_0,
	PROP_KZ_WINDOW,
	PROP_KZ_BOOKMARK,
};

static void kz_smart_bookmark_action_class_init       (KzSmartBookmarkActionClass *class);
static void kz_smart_bookmark_action_init             (KzSmartBookmarkAction      *action);
static void kz_smart_bookmark_action_dispose          (GObject *obj);
static void kz_smart_bookmark_action_set_property     (GObject           *object,
						       guint              prop_id,
						       const GValue      *value,
						       GParamSpec        *pspec);
static void kz_smart_bookmark_action_get_property     (GObject           *object,
						       guint              prop_id,
						       GValue            *value,
						       GParamSpec        *pspec);

static void kz_smart_bookmark_action_activate         (GtkAction         *action);
static void kz_smart_bookmark_action_connect_proxy    (GtkAction         *action,
						       GtkWidget         *proxy);
static void kz_smart_bookmark_action_disconnect_proxy (GtkAction         *action,
						       GtkWidget         *proxy);

static gboolean cb_entry_key_press                       (GtkWidget *widget,
							  GdkEventKey *event,
							  KzSmartBookmarkAction *action);

static gchar *kz_bookmark_get_smart_uri (KzBookmark *bookmark, const gchar *uri);

static KzEntryActionClass *parent_class = NULL;


KZ_OBJECT_GET_TYPE(kz_smart_bookmark_action,
		   "KzSmartBookmarkAction",
		   KzSmartBookmarkAction,
		   kz_smart_bookmark_action_class_init,
		   kz_smart_bookmark_action_init,
		   KZ_TYPE_ENTRY_ACTION)


static void
kz_smart_bookmark_action_class_init (KzSmartBookmarkActionClass *klass)
{
	GObjectClass *object_class;
	GtkActionClass *action_class;
	KzEntryActionClass *entry_class;

	parent_class  = g_type_class_peek_parent(klass);
	object_class  = G_OBJECT_CLASS(klass);
	action_class  = GTK_ACTION_CLASS(klass);
	entry_class   = KZ_ENTRY_ACTION_CLASS(klass);

	object_class->set_property     = kz_smart_bookmark_action_set_property;
	object_class->get_property     = kz_smart_bookmark_action_get_property;
	object_class->dispose          = kz_smart_bookmark_action_dispose;

	action_class->activate         = kz_smart_bookmark_action_activate;
	action_class->connect_proxy    = kz_smart_bookmark_action_connect_proxy;
	action_class->disconnect_proxy = kz_smart_bookmark_action_disconnect_proxy;

	g_object_class_install_property
		(object_class,
		 PROP_KZ_WINDOW,
		 g_param_spec_object ("kz-window",
				      _("KzWindow"),
				      _("The KzWindow"),
				      KZ_TYPE_WINDOW,
				      G_PARAM_READWRITE |
				      G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property
		(object_class,
		 PROP_KZ_BOOKMARK,
		 g_param_spec_object ("kz-bookmark",
				      _("KzBookmark"),
				      _("The Bookmark object"),
				      KZ_TYPE_BOOKMARK,
				      G_PARAM_READWRITE |
				      G_PARAM_CONSTRUCT_ONLY));
}


static void
kz_smart_bookmark_action_init (KzSmartBookmarkAction *action)
{
	action->kz            = NULL;
	action->bookmark      = NULL;
}


static void
kz_smart_bookmark_action_dispose (GObject *obj)
{
	KzSmartBookmarkAction *action = KZ_SMART_BOOKMARK_ACTION(obj);

	if (action->kz)
	{
		g_object_unref(action->kz);
		action->kz = NULL;
	}
	if (action->bookmark)
	{
		g_object_unref(action->bookmark);
		action->bookmark = NULL;
	}

	if (G_OBJECT_CLASS(parent_class)->dispose)
		G_OBJECT_CLASS(parent_class)->dispose(obj);
}


static void
kz_smart_bookmark_action_set_property (GObject         *object,
				       guint            prop_id,
				       const GValue    *value,
				       GParamSpec      *pspec)
{
	KzSmartBookmarkAction *action = KZ_SMART_BOOKMARK_ACTION(object);
  
	switch (prop_id)
	{
	case PROP_KZ_WINDOW:
		action->kz = g_object_ref(g_value_get_object(value));
		break;
	case PROP_KZ_BOOKMARK:
		action->bookmark = g_object_ref(g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


static void
kz_smart_bookmark_action_get_property (GObject         *object,
				       guint            prop_id,
				       GValue          *value,
				       GParamSpec      *pspec)
{
	KzSmartBookmarkAction *action = KZ_SMART_BOOKMARK_ACTION(object);

	switch (prop_id)
	{
	case PROP_KZ_WINDOW:
		g_value_set_object(value, action->kz);
		break;
	case PROP_KZ_BOOKMARK:
		g_value_set_object(value, action->bookmark);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


static void
kz_smart_bookmark_action_activate (GtkAction *action)
{
	KzSmartBookmarkAction *smart_action;
	const gchar *uri;
	gchar *smart_uri;
	gboolean new_tab;

	g_return_if_fail(KZ_IS_SMART_BOOKMARK_ACTION(action));

	if (GTK_ACTION_CLASS(parent_class)->activate)
		GTK_ACTION_CLASS(parent_class)->activate(action);

	smart_action = KZ_SMART_BOOKMARK_ACTION(action);

	uri = kz_entry_action_get_text(KZ_ENTRY_ACTION(action));
	smart_uri = kz_bookmark_get_smart_uri(smart_action->bookmark, uri);

	KZ_CONF_GET("Global", "entry_open_in_new_tab", new_tab, BOOL);
	if (new_tab)
		kz_window_open_new_tab(smart_action->kz, smart_uri);
	else	
		kz_window_load_url(smart_action->kz, smart_uri);
	g_free(smart_uri);
}


static void
kz_smart_bookmark_action_connect_proxy (GtkAction *action, GtkWidget *proxy)
{
	GtkEntry *entry;

	GTK_ACTION_CLASS (parent_class)->connect_proxy (action, proxy);

	entry = kz_entry_action_get_entry_widget(KZ_ENTRY_ACTION(action), proxy);
	if (entry)
	{
		KzSmartBookmarkAction *smart_action;
		KzBookmark *bookmark;
		const gchar *title, *stock_id;

		smart_action = KZ_SMART_BOOKMARK_ACTION(action);
		bookmark = smart_action->bookmark;
		g_signal_connect(G_OBJECT(entry), "key-press-event",
				 G_CALLBACK(cb_entry_key_press), action);
		title = kz_bookmark_get_title(bookmark);
		kz_entry_set_backtext(KZ_ENTRY(entry), 
				      title);
#warning FIXME! set proper icon
		if (!strncmp(title, "Google", 6))
			stock_id = KZ_STOCK_GOOGLE;
		else
			stock_id = KZ_STOCK_HISTORY;
		kz_entry_set_icon_from_stock(KZ_ENTRY(entry),
					     stock_id,
					     GTK_ICON_SIZE_MENU);
	}
}


static void
kz_smart_bookmark_action_disconnect_proxy (GtkAction *action, GtkWidget *proxy)
{
	GtkEntry *entry;

	entry = kz_entry_action_get_entry_widget(KZ_ENTRY_ACTION(action), proxy);
	if (entry)
	{
		g_signal_handlers_disconnect_by_func
			(G_OBJECT(entry),
			 G_CALLBACK(cb_entry_key_press),
			 action);
	}

	GTK_ACTION_CLASS (parent_class)->disconnect_proxy (action, proxy);
}


KzSmartBookmarkAction *
kz_smart_bookmark_action_new (KzWindow *kz, KzBookmark *bookmark)
{
	KzSmartBookmarkAction *action;
	gchar *name = g_strconcat("SmartBookmark:",
				  kz_bookmark_get_title(bookmark),
				  NULL);

	action = KZ_SMART_BOOKMARK_ACTION(
			g_object_new(KZ_TYPE_SMART_BOOKMARK_ACTION,
				     "name",        name,
				     "label",       _("Entry"),
				     "tooltip",     NULL,
				     "stock_id",    GTK_STOCK_NEW,
				     "kz-window",   kz,
				     "kz-bookmark", bookmark,
				     NULL));
	g_free(name);

	return action;
}


static gboolean
cb_entry_key_press (GtkWidget *widget, GdkEventKey *event,
		    KzSmartBookmarkAction *action)
{
	gboolean new_tab;

	if ((event->keyval == GDK_Return || event->keyval == GDK_ISO_Enter)
	    && (event->state & GDK_CONTROL_MASK))
	{
		const gchar *uri;
		gchar *smart_uri;

		KZ_CONF_GET("Global", "entry_open_in_new_tab", new_tab, BOOL);

		uri = kz_entry_action_get_text(KZ_ENTRY_ACTION(action));

		smart_uri = kz_bookmark_get_smart_uri(action->bookmark, uri);

		if (new_tab)
			kz_window_load_url(action->kz, smart_uri);
		else 
			kz_window_open_new_tab(action->kz, smart_uri);
		g_free(smart_uri);

		return TRUE;
	}

	return FALSE;
}

#warning FIXME!
static gchar *
kz_bookmark_get_smart_uri (KzBookmark *bookmark, const gchar *uri)
{
	gchar *text_encoded, *smart_uri;
	text_encoded = url_encode(uri);
	smart_uri = g_strdup_printf(kz_bookmark_get_link(bookmark), 
				    text_encoded);
	g_free(text_encoded);
	
	return smart_uri;
}
