/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2004 Hidetaka Iwai
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <stdlib.h>
#include <regex.h>

#include "prefs_location_entry.h"

#include "kazehakase.h"

#include "utils/utils.h"
#include "utils/intl.h"
#include "utils/gtk-utils.h"

#define DATA_KEY "KzPrefsLocationEntry::info"
#define REGEX_ENTRY_WIDTH 140
#define URI_ENTRY_WIDTH 180

enum
{
	COLUMN_TERMINATOR = -1,
	COLUMN_KIND,
	COLUMN_REGEX,
	COLUMN_URI,
	COLUMN_ENCODE,
	COLUMN_EDITABLE,
	N_COLUMNS
};


typedef struct _KzPrefsLocationEntry
{
	GtkWidget     *main_vbox;
	GtkTreeView   *tree_view;
	GtkListStore  *store;
	GtkWidget     *regex_entry, *uri_entry;
	GtkWidget     *up, *down;
	GtkWidget     *new, *add, *remove;
	gboolean       changed;
	gboolean       lock_entry_handlers;
} KzPrefsLocationEntry;


static GtkWidget     *prefs_location_entry_create          (void);
static void           prefs_location_entry_response        (GtkWidget *widget,
							     gint       response);
static void           prefs_location_entry_set_sensitive   (KzPrefsLocationEntry *prefsui);
static void           prefs_location_entry_set_entry       (KzPrefsLocationEntry *prefsui);
static void           prefs_location_entry_destroy         (gpointer data);


static KzPrefsWinPageEntry prefs_entry =
{
	path:            N_("/Location Entry"),
	priority_hint:   0,
	ui_level:        KZ_UI_LEVEL_EXPERT,
	create:          prefs_location_entry_create,
	response:        prefs_location_entry_response,
};


static void
cb_up_button (GtkButton *button, KzPrefsLocationEntry *prefsui)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter, prev;
	GtkTreePath *treepath;

	g_return_if_fail(prefsui);

	selection = gtk_tree_view_get_selection (prefsui->tree_view);
	if (!gtk_tree_selection_get_selected (selection, &model, &iter))
		return;

	treepath = gtk_tree_model_get_path(model, &iter);
	if (!gtk_tree_path_prev(treepath)) goto ERROR;
	gtk_tree_model_get_iter(model, &prev, treepath);

	gtkutil_list_store_swap(prefsui->store, &iter, &prev);

	prefsui->changed = TRUE;

ERROR:
	gtk_tree_path_free(treepath);

	prefs_location_entry_set_sensitive (prefsui);
}


static void
cb_down_button (GtkButton *button, KzPrefsLocationEntry *prefsui)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter, next;

	g_return_if_fail(prefsui);

	selection = gtk_tree_view_get_selection (prefsui->tree_view);
	if (!gtk_tree_selection_get_selected (selection, &model, &iter))
		return;

	next = iter;
	if (!gtk_tree_model_iter_next(model, &next)) return;

	gtkutil_list_store_swap(prefsui->store, &iter, &next);

	prefsui->changed = TRUE;

	prefs_location_entry_set_sensitive (prefsui);
}


static void
cb_new_button (GtkButton *button, KzPrefsLocationEntry *prefsui)
{
	GtkTreeSelection *selection;

	g_return_if_fail(prefsui);

	selection = gtk_tree_view_get_selection(prefsui->tree_view);
	gtk_tree_selection_unselect_all(selection);

	prefs_location_entry_set_entry(prefsui);
	prefs_location_entry_set_sensitive(prefsui);
}


static void
cb_add_button (GtkButton *button, KzPrefsLocationEntry *prefsui)
{
	GtkTreeIter iter;
	GtkTreePath *treepath;
	const gchar *regex, *uri;

	g_return_if_fail(prefsui);

	regex = gtk_entry_get_text(GTK_ENTRY(prefsui->regex_entry));
	uri   = gtk_entry_get_text(GTK_ENTRY(prefsui->uri_entry));

	gtk_list_store_append(prefsui->store, &iter);
	gtk_list_store_set(prefsui->store, &iter,
			   COLUMN_REGEX,    regex,
			   COLUMN_URI,      uri,
			   COLUMN_ENCODE,   FALSE,
			   COLUMN_EDITABLE, TRUE,
			   COLUMN_TERMINATOR);

	treepath = gtk_tree_model_get_path(GTK_TREE_MODEL(prefsui->store),
					   &iter);
	gtk_tree_view_set_cursor(prefsui->tree_view,
				 treepath, NULL, FALSE);
	gtk_tree_path_free(treepath);

	prefsui->changed = TRUE;

	prefs_location_entry_set_sensitive (prefsui);
}


static void
cb_remove_button (GtkButton *button, KzPrefsLocationEntry *prefsui)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter, next;
	GtkTreePath *treepath;
	gboolean select;

	g_return_if_fail(prefsui);

	selection = gtk_tree_view_get_selection (prefsui->tree_view);
	if (!gtk_tree_selection_get_selected (selection, &model, &iter))
		return;

        /* get next row to select */
	next = iter;
	select = gtk_tree_model_iter_next(model, &next);
	if (select)
	{
		treepath = gtk_tree_model_get_path(model, &next);
	}
	else
	{
		treepath = gtk_tree_model_get_path(model, &iter);
		select = gtk_tree_path_prev(treepath);
	}
	if (select)
		gtk_tree_view_set_cursor(prefsui->tree_view,
					 treepath, NULL, FALSE);
	gtk_tree_path_free(treepath);

	/* remove the row */
	gtk_list_store_remove(prefsui->store, &iter);

	prefsui->changed = TRUE;

	prefs_location_entry_set_sensitive (prefsui);
}


static void
cb_regex_edited (GtkCellRendererText *cell,
		 const gchar *path_str,
		 const gchar *new_text,
		 KzPrefsLocationEntry *prefsui)
{
        GtkTreeIter  iter;

        gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(prefsui->store),
					    &iter, path_str);
	gtk_list_store_set(prefsui->store, &iter,
			   COLUMN_REGEX, new_text,
			   COLUMN_TERMINATOR);

	prefsui->changed = TRUE;

	prefs_location_entry_set_entry(prefsui);
	prefs_location_entry_set_sensitive(prefsui);
}


static void
cb_uri_edited (GtkCellRendererText *cell,
	       const gchar *path_str,
	       const gchar *new_text,
	       KzPrefsLocationEntry *prefsui)
{
        GtkTreeIter  iter;

        gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(prefsui->store),
					    &iter, path_str);
	gtk_list_store_set(prefsui->store, &iter,
			   COLUMN_URI, new_text,
			   COLUMN_TERMINATOR);

	prefsui->changed = TRUE;

	prefs_location_entry_set_entry(prefsui);
	prefs_location_entry_set_sensitive(prefsui);
}

static void
cb_encode_toggled (GtkCellRendererToggle *cell,
			  const gchar *path_str,
			  KzPrefsLocationEntry *prefsui)
{
        GtkTreeIter  iter;
	GValue val = { 0 };
	gboolean flag;

	gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(prefsui->store),
					    &iter, path_str);

	gtk_tree_model_get_value(GTK_TREE_MODEL(prefsui->store), &iter,
				 COLUMN_ENCODE, &val);
	flag = !(g_value_get_boolean(&val));

	gtk_list_store_set(prefsui->store, &iter,
			   COLUMN_ENCODE, flag,
			   COLUMN_TERMINATOR);

	prefsui->changed = TRUE;

	prefs_location_entry_set_entry(prefsui);
	prefs_location_entry_set_sensitive(prefsui);
}

static void
cb_cursor_changed(GtkTreeView *tree_view, KzPrefsLocationEntry *prefsui)
{
	g_return_if_fail(prefsui);

	prefsui->lock_entry_handlers = TRUE;
	prefs_location_entry_set_entry(prefsui);
	prefs_location_entry_set_sensitive (prefsui);
	prefsui->lock_entry_handlers = FALSE;
}


static void
cb_regex_entry_changed(GtkEditable *editable, KzPrefsLocationEntry *prefsui)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;

	g_return_if_fail(prefsui);

	if (prefsui->lock_entry_handlers) return;

	selection = gtk_tree_view_get_selection (prefsui->tree_view);
	if (gtk_tree_selection_get_selected (selection, &model, &iter))
	{
		const gchar *regex;

		regex = gtk_entry_get_text(GTK_ENTRY(prefsui->regex_entry));
		gtk_list_store_set(GTK_LIST_STORE(model), &iter,
				   COLUMN_REGEX, regex,
				   COLUMN_TERMINATOR);
		prefsui->changed = TRUE;
	}

	prefs_location_entry_set_sensitive (prefsui);
}


static void
cb_uri_entry_changed(GtkEditable *editable, KzPrefsLocationEntry *prefsui)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;

	g_return_if_fail(prefsui);

	if (prefsui->lock_entry_handlers) return;

	selection = gtk_tree_view_get_selection (prefsui->tree_view);
	if (gtk_tree_selection_get_selected (selection, &model, &iter))
	{
		const gchar *uri;

		uri = gtk_entry_get_text(GTK_ENTRY(prefsui->uri_entry));
		gtk_list_store_set(GTK_LIST_STORE(model), &iter,
				   COLUMN_URI, uri,
				   COLUMN_TERMINATOR);
		prefsui->changed = TRUE;
	}

	prefs_location_entry_set_sensitive (prefsui);
}

static GtkWidget *
prefs_location_entry_create (void)
{
	KzPrefsLocationEntry *prefsui = g_new0(KzPrefsLocationEntry, 1);
	GtkWidget *main_vbox, *list_hbox, *hbox, *vbox1, *vbox2;
	GtkWidget *label, *scrwin, *tree_view;
	GtkWidget *button, *arrow, *entry;
	GtkListStore *store;
	GtkCellRenderer *cell;
	GtkTreeViewColumn *column;
	GList *list, *node;
	gint index;

	prefsui->changed = FALSE;
	prefsui->lock_entry_handlers = FALSE;

	main_vbox = gtk_vbox_new(FALSE, 0);
	prefsui->main_vbox = main_vbox;
	g_object_set_data_full(G_OBJECT(main_vbox), DATA_KEY,
			       prefsui, prefs_location_entry_destroy);

	label = kz_prefs_ui_utils_create_title(_("Location Entry"));
	gtk_box_pack_start(GTK_BOX(main_vbox), label,
			   FALSE, FALSE, 0);
	gtk_widget_show(label);

	/* description */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(main_vbox), hbox,
			   FALSE, FALSE, 0);
	gtk_widget_show(hbox);

	label = gtk_label_new(_("Kazehakase can change its behaviour "
				"according to regular expressions"));
	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
	gtk_box_pack_start(GTK_BOX(hbox), label,
			   FALSE, FALSE, 10);
	gtk_widget_show(label);

	/* hbox for list and up/down buttons */
	list_hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(main_vbox), list_hbox,
			   TRUE, TRUE, 0);
	gtk_widget_show(list_hbox);

	/* scrolled window */
	scrwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrwin),
					    GTK_SHADOW_IN);
	gtk_container_set_border_width(GTK_CONTAINER(scrwin), 5);
	gtk_box_pack_start(GTK_BOX(list_hbox), scrwin, TRUE, TRUE, 0);
	gtk_widget_show(scrwin);

	/* tree view */
	store = gtk_list_store_new(N_COLUMNS,
				   G_TYPE_STRING,
				   G_TYPE_STRING,
				   G_TYPE_STRING,
				   G_TYPE_BOOLEAN,
				   G_TYPE_BOOLEAN);

	prefsui->store = store;

	tree_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
	prefsui->tree_view = GTK_TREE_VIEW(tree_view);
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree_view), TRUE);

	/* Regex column */
	cell = gtk_cell_renderer_text_new();
	g_signal_connect(G_OBJECT(cell), "edited",
			 G_CALLBACK(cb_regex_edited), prefsui);
	column = gtk_tree_view_column_new_with_attributes
			(_("Regex"), cell,
			 "text",     COLUMN_REGEX,
			 "editable", COLUMN_EDITABLE,
			 NULL);
	gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_fixed_width (column, REGEX_ENTRY_WIDTH);
	gtk_tree_view_column_set_resizable(column, TRUE);
	gtk_tree_view_append_column(prefsui->tree_view, column);

	/* URI column */
	cell = gtk_cell_renderer_text_new();
	g_signal_connect(G_OBJECT(cell), "edited",
			 G_CALLBACK(cb_uri_edited), prefsui);
	column = gtk_tree_view_column_new_with_attributes
			(_("URI"), cell,
			 "text",     COLUMN_URI,
			 "editable", COLUMN_EDITABLE,
			 NULL);
	gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_fixed_width (column, URI_ENTRY_WIDTH);
	gtk_tree_view_column_set_resizable(column, TRUE);
	gtk_tree_view_append_column(prefsui->tree_view, column);

        /* ENCODE column */
	cell = gtk_cell_renderer_toggle_new();
	g_signal_connect(G_OBJECT(cell), "toggled",
			 G_CALLBACK(cb_encode_toggled), prefsui);

	gtk_cell_renderer_toggle_set_radio
		(GTK_CELL_RENDERER_TOGGLE(cell), FALSE);

	column = gtk_tree_view_column_new_with_attributes
		(_("ENCODE"), cell,
		 "active", COLUMN_ENCODE,
		 "activatable", COLUMN_EDITABLE,
		 NULL);
	gtk_tree_view_column_set_resizable(column, TRUE);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);

        /* set default value */
	list = kz_profile_enum_key(kz_global_profile, "LocationEntry", TRUE);
	for (node = list; node; node = g_list_next(node))
	{

		const gchar *regex_key = node->data;
		gchar key[32];
		GtkTreeIter iter;
		gchar *first;
		gchar *regex, *uri;
		gboolean encode = FALSE;

		if (!key_seems_sequential(regex_key, "regex")) continue;
		regex = kz_profile_get_string(kz_global_profile, "LocationEntry", regex_key);

		first = strpbrk(regex_key, "0123456789");
		if(!first)
		{
			g_warning("KzPrefsLocationEntry: unexpected error: "
				  "URI regex key has no number!");
			continue;
		}

		index = strtol(first, (char**)'\0', 10);

		g_snprintf(key, G_N_ELEMENTS(key), "uri%d", index);
		uri = kz_profile_get_string(kz_global_profile, "LocationEntry", key);

		g_snprintf(key, G_N_ELEMENTS(key), "encode%d", index);
		kz_profile_get_value(kz_global_profile, "LocationEntry", key,
				     &encode, sizeof(encode),
				     KZ_PROFILE_VALUE_TYPE_BOOL);

		gtk_list_store_append(store, &iter);
		gtk_list_store_set(store, &iter,
				   COLUMN_REGEX,     regex,
				   COLUMN_URI,       uri,
				   COLUMN_ENCODE, encode,
				   COLUMN_EDITABLE,  TRUE,
				   COLUMN_TERMINATOR);
		g_free(regex);
		g_free(uri);
	}
	g_list_free(list);

	gtk_container_add(GTK_CONTAINER(scrwin), tree_view);
	gtk_widget_show(tree_view);

	/* button box */
	vbox1 = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(list_hbox), vbox1, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox1), 0);
	gtk_widget_show(vbox1);

	vbox2 = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), vbox2, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox2), 0);
	gtk_widget_show(vbox2);

	/* up button */
	button = gtk_button_new();
	prefsui->up = button;
	arrow = gtk_arrow_new(GTK_ARROW_UP, GTK_SHADOW_NONE);
	gtk_container_add(GTK_CONTAINER(button), arrow);
	gtk_widget_show(arrow);
	gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, FALSE, 5);
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(cb_up_button), prefsui);
	gtk_widget_show(button);

	/* down button */
	button = gtk_button_new();
	prefsui->down = button;
	arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_NONE);
	gtk_container_add(GTK_CONTAINER(button), arrow);
	gtk_widget_show(arrow);
	gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, FALSE, 5);
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(cb_down_button), prefsui);
	gtk_widget_show(button);

	/* entry box */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
	gtk_widget_show(hbox);

	prefsui->regex_entry = entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
	gtk_widget_set_size_request(entry, REGEX_ENTRY_WIDTH, -1);
	g_signal_connect(G_OBJECT(entry), "changed",
			 G_CALLBACK(cb_regex_entry_changed), prefsui);
	gtk_widget_show(entry);

	prefsui->uri_entry = entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
	gtk_widget_set_size_request(entry, URI_ENTRY_WIDTH, -1);
	g_signal_connect(G_OBJECT(entry), "changed",
			 G_CALLBACK(cb_uri_entry_changed), prefsui);
	gtk_widget_show(entry);

	/* button box */
	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
	gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
	gtk_widget_show(hbox);

	/* clear button */
	button = gtk_button_new_from_stock(GTK_STOCK_NEW);
	prefsui->new = button;
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(cb_new_button), prefsui);
	gtk_widget_show(button);

	/* add button */
	button = gtk_button_new_from_stock(GTK_STOCK_ADD);
	prefsui->add = button;
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(cb_add_button), prefsui);
	gtk_widget_show(button);

	/* remove button */
	button = gtk_button_new_from_stock(GTK_STOCK_REMOVE);
	prefsui->remove = button;
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(cb_remove_button), prefsui);
	gtk_widget_show(button);

	/* signal handler for GtkTreeView */
	g_signal_connect(G_OBJECT(tree_view), "cursor-changed",
			 G_CALLBACK(cb_cursor_changed), prefsui);

	prefs_location_entry_set_sensitive (prefsui);

	return main_vbox;
}

static void
prefs_location_entry_apply (KzPrefsLocationEntry *prefsui)
{
	GtkTreeModel *model = GTK_TREE_MODEL(prefsui->store);
	GtkTreeIter iter;
	GSList *new_list = NULL;
	GList *node, *list;
	gboolean exist;
	gint index;
	gchar regex_key[32], key[32];
	gboolean encode;

	if (!prefsui->changed) return;

	/* first, remove all old entries */
	list = kz_profile_enum_key(kz_global_profile, "LocationEntry", TRUE);
	for (node = list; node; node = g_list_next(node))
	{
		const gchar *key = node->data;

		if (!key_seems_sequential(key, "regex") &&
		    !key_seems_sequential(key, "uri")   &&
		    !key_seems_sequential(key, "encode")) continue;

		kz_profile_delete_key(kz_global_profile, "LocationEntry", key);
	}
	g_list_free(list);
	
	exist = gtk_tree_model_get_iter_first(model, &iter);
	for(index = 0; exist; exist = gtk_tree_model_iter_next(model, &iter), ++index)
	{
		gchar *regex = NULL, *uri = NULL;

		gtk_tree_model_get(model, &iter,
				   COLUMN_REGEX,     &regex,
				   COLUMN_URI,       &uri,
				   COLUMN_ENCODE, &encode,
				   COLUMN_TERMINATOR);

		g_snprintf(regex_key, G_N_ELEMENTS(regex_key), "regex%d", index);

		kz_profile_set_value(kz_global_profile, "LocationEntry",
				     regex_key, regex, sizeof(regex),
				     KZ_PROFILE_VALUE_TYPE_STRING);

		g_snprintf(key, G_N_ELEMENTS(key), "uri%d", index);
		kz_profile_set_value(kz_global_profile, "LocationEntry",
				     key, uri, sizeof(uri),
				     KZ_PROFILE_VALUE_TYPE_STRING);

		g_snprintf(key, G_N_ELEMENTS(key), "encode%d", index);
		kz_profile_set_value(kz_global_profile, "LocationEntry",
				     key, &encode, sizeof(encode),
				     KZ_PROFILE_VALUE_TYPE_BOOL);

		g_free(regex);
		g_free(uri);
	}

	g_slist_foreach(new_list, (GFunc) g_object_unref, NULL);
	g_slist_free(new_list);

	prefsui->changed = FALSE;

	prefs_location_entry_set_sensitive (prefsui);
}


static void
prefs_location_entry_response (GtkWidget *widget, gint response)
{
	KzPrefsLocationEntry *prefsui = g_object_get_data(G_OBJECT(widget), DATA_KEY);

	g_return_if_fail(prefsui);

	switch (response) {
	case GTK_RESPONSE_ACCEPT:
	case GTK_RESPONSE_APPLY:
		prefs_location_entry_apply(prefsui);
		break;
	case GTK_RESPONSE_REJECT:
		break;
	case KZ_RESPONSE_UI_LEVEL_MEDIUM:
		break;
	case KZ_RESPONSE_UI_LEVEL_EXPERT:
		break;
	case KZ_RESPONSE_UI_LEVEL_CUSTOM:
		break;
	default:
		break;
	}
}


static void
prefs_location_entry_set_sensitive (KzPrefsLocationEntry *prefsui)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter, next;
	gboolean select;
	gboolean can_up = FALSE, can_down = FALSE;
	gboolean can_add = FALSE, can_new = TRUE;
	const gchar *regex, *uri;

	g_return_if_fail(prefsui);

	selection = gtk_tree_view_get_selection (prefsui->tree_view);
	select = gtk_tree_selection_get_selected (selection, &model, &iter);

	if (select)
	{
		GtkTreePath *treepath;

		next = iter;
		can_down = gtk_tree_model_iter_next(model, &next);

		treepath = gtk_tree_model_get_path(model, &iter);
		can_up = gtk_tree_path_prev(treepath);
		gtk_tree_path_free(treepath);
	}

	regex = gtk_entry_get_text(GTK_ENTRY(prefsui->regex_entry));
	uri   = gtk_entry_get_text(GTK_ENTRY(prefsui->uri_entry));

	if (regex && *regex && uri && *uri)
		can_add = TRUE;

	if ((!regex || !*regex) && (!uri || !*uri) && !select)
		can_new = FALSE;

	gtk_widget_set_sensitive(prefsui->up,     can_up);
	gtk_widget_set_sensitive(prefsui->down,   can_down);
	gtk_widget_set_sensitive(prefsui->new,    can_new);
	gtk_widget_set_sensitive(prefsui->add,    can_add);
	gtk_widget_set_sensitive(prefsui->remove, select);
}


static void
prefs_location_entry_set_entry (KzPrefsLocationEntry *prefsui)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *regex = NULL, *uri = NULL;
	gboolean encode;

	g_return_if_fail(prefsui);

	selection = gtk_tree_view_get_selection (prefsui->tree_view);

	if (gtk_tree_selection_get_selected (selection, &model, &iter))
	{
		gtk_tree_model_get(model, &iter,
				   COLUMN_REGEX,     &regex,
				   COLUMN_URI,       &uri,
				   COLUMN_ENCODE, &encode,
				   COLUMN_TERMINATOR);

		gtk_entry_set_text(GTK_ENTRY(prefsui->regex_entry), regex);
		gtk_entry_set_text(GTK_ENTRY(prefsui->uri_entry), uri);

		g_free(regex);
		g_free(uri);
	}
	else
	{
		gtk_entry_set_text(GTK_ENTRY(prefsui->regex_entry), "");
		gtk_entry_set_text(GTK_ENTRY(prefsui->uri_entry), "");
	}
}


static void
prefs_location_entry_destroy (gpointer data)
{
	KzPrefsLocationEntry *prefsui = data;

	g_free(prefsui);
}


KzPrefsWinPageEntry *
prefs_location_entry_get_entry (gint idx)
{
	if (idx == 0)
		return &prefs_entry;
	else
		return NULL;
}
