/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 - 2004 Hiroyuki Ikezoe
 *  Copyright (C) 2003 - 2004 Takuro Ashie
 *  Copyright (C) 2004 Hidetaka Iwai
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*  
 *  This file is based on kz-location-entry.c.
 *
 */
#include "kz-history-search-action.h"

#include <stdlib.h>
#include <gdk/gdkkeysyms.h>
#include <sys/types.h>
#include <ctype.h>

#include "kazehakase.h"
#include "intl.h"
#include "utils/utils.h"
#include "gobject-utils.h"
#include "glib-utils.h"
#include "egg-toolbutton.h"
#include "kz-mozembed.h"
#include "kz-entry.h"

#define DESCRIPTION_LEN 1024
#define ESTRAIER_URI "http://estraier.sourceforge.net/"
#define DTD   "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\" \"http://www.w3.org/TR/html4/strict.dtd\">"
#define HEAD  "<head>\n" \
	      "  <title>Full-text search in history</title>\n" \
	      "  <style type=\"text/css\">\n" \
	      "  %s\n" \
              "  </style>\n" \
	      "</head>\n"
#define STYLE "body {\n" \
	      "background-color: #ccc;\n" \
	      "}\n" \
	      "div.summary {\n" \
	      "padding: 0.2em;\n" \
	      "float: right;\n" \
	      "}\n" \
	      "div.content {\n" \
	      "background-color: #eee;\n" \
	      "margin: 1.2em;\n" \
	      "border-style: solid;\n" \
	      "border-color: black;\n" \
	      "border-width: 1px;\n" \
	      "}\n" \
	      "div.header {\n" \
	      "padding: 0.2em;\n" \
	      "background-color: #ccf;\n" \
	      "border-bottom: solid 1px black;\n" \
	      "margin: 0em;\n" \
	      "}\n" \
	      "span.title {\n" \
	      "font-weight: bold;\n"\
	      "}\n" \
	      "span.keyword {\n" \
	      "color: #ff0000;\n" \
	      "}\n" \
	      "img.thumbnail {\n" \
	      "width: 128px;\n" \
	      "float: left;\n" \
	      "margin: 0.2em;\n" \
	      "}\n" \
	      "div.footer {\n" \
	      "text-align: right;\n" \
	      "text-size: 90%;\n" \
	      "border-top: solid 1px #888;\n" \
	      "clear: both;\n" \
	      "}\n" 
#define HEADER  ""
#define CONTENT "<div class=\"content\">\n" \
	        "  <div class=\"header\"><span class=\"title\"><a href=\"%s\">%s</a></span></div>\n" \
		"    <div class=\"summary\"><img src=\"%s\" class=\"thumbnail\">\n" \
		"    <span class=\"sentence\">%s</span>\n" \
		"  </div>\n" \
		"  <div class=\"footer\">\n" \
		"    <span class=\"cache\"><a href=\"%s\">cache</a></span>\n" \
		"    <span class=\"date\">%s</span>\n" \
		"  </div>\n" \
		"</div>\n"
#define FOOTER  "<div class=\"footer\">\n" \
	        "Powered by <a href=\"%s\">Estraier</a> version %s\n" \
		"</div>\n"

enum {
	PROP_0,
	PROP_KZ_WINDOW,
};

static void kz_history_search_action_class_init       (KzHistorySearchActionClass *class);
static void kz_history_search_action_init             (KzHistorySearchAction      *action);
static void kz_history_search_action_dispose          (GObject *obj);
static void kz_history_search_action_set_property     (GObject      *object,
						       guint         prop_id,
						       const GValue *value,
						       GParamSpec   *pspec);
static void kz_history_search_action_get_property     (GObject      *object,
						       guint         prop_id,
						       GValue       *value,
						       GParamSpec   *pspec);

static void kz_history_search_action_activate         (EggAction    *action);
static void kz_history_search_action_connect_proxy    (EggAction    *action,
						       GtkWidget    *proxy);
static void kz_history_search_action_disconnect_proxy (EggAction    *action,
						       GtkWidget    *proxy);

static void kz_history_search_action_open_result      (KzHistorySearchAction *action);

static gboolean cb_entry_key_press                       (GtkWidget *widget,
							  GdkEventKey *event,
							  KzHistorySearchAction *action);

static EggEntryActionClass *parent_class = NULL;


KZ_OBJECT_GET_TYPE(kz_history_search_action,
		   "KzHistorySearchAction",
		   KzHistorySearchAction,
		   kz_history_search_action_class_init,
		   kz_history_search_action_init,
		   EGG_TYPE_ENTRY_ACTION)


static void
kz_history_search_action_class_init (KzHistorySearchActionClass *klass)
{
	GObjectClass *object_class;
	EggActionClass *action_class;

	parent_class  = g_type_class_peek_parent(klass);
	object_class  = G_OBJECT_CLASS(klass);
	action_class  = EGG_ACTION_CLASS(klass);

	object_class->set_property     = kz_history_search_action_set_property;
	object_class->get_property     = kz_history_search_action_get_property;
	object_class->dispose          = kz_history_search_action_dispose;

	action_class->activate         = kz_history_search_action_activate;
	action_class->connect_proxy    = kz_history_search_action_connect_proxy;
	action_class->disconnect_proxy = kz_history_search_action_disconnect_proxy;

	g_object_class_install_property
		(object_class,
		 PROP_KZ_WINDOW,
		 g_param_spec_object ("kz-window",
				      _("KzWindow"),
				      _("The KzWindow to add a home button"),
				      KZ_TYPE_WINDOW,
				      G_PARAM_READWRITE |
				      G_PARAM_CONSTRUCT_ONLY));
}


static void
kz_history_search_action_init (KzHistorySearchAction *action)
{
	action->kz = NULL;
}


static void
kz_history_search_action_dispose (GObject *obj)
{
	KzHistorySearchAction *action = KZ_HISTORY_SEARCH_ACTION(obj);

	if (action->kz)
	{
		g_object_unref(action->kz);
		action->kz = NULL;
	}

	if (G_OBJECT_CLASS(parent_class)->dispose)
		G_OBJECT_CLASS(parent_class)->dispose(obj);
}


static void
kz_history_search_action_set_property (GObject         *object,
				       guint            prop_id,
				       const GValue    *value,
				       GParamSpec      *pspec)
{
	KzHistorySearchAction *action = KZ_HISTORY_SEARCH_ACTION(object);
  
	switch (prop_id)
	{
	case PROP_KZ_WINDOW:
		action->kz = g_object_ref(g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


static void
kz_history_search_action_get_property (GObject         *object,
				       guint            prop_id,
				       GValue          *value,
				       GParamSpec      *pspec)
{
	KzHistorySearchAction *action = KZ_HISTORY_SEARCH_ACTION(object);

	switch (prop_id)
	{
	case PROP_KZ_WINDOW:
		g_value_set_object(value, action->kz);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


static void
kz_history_search_action_activate (EggAction *action)
{
	KzHistorySearchAction *entry_action;

	g_return_if_fail(KZ_IS_HISTORY_SEARCH_ACTION(action));

	if (EGG_ACTION_CLASS(parent_class)->activate)
		EGG_ACTION_CLASS(parent_class)->activate(action);

	entry_action = KZ_HISTORY_SEARCH_ACTION(action);
	kz_history_search_action_open_result(entry_action);
}


static void
kz_history_search_action_connect_proxy (EggAction *action, GtkWidget *proxy)
{
	GtkEntry *entry;

	EGG_ACTION_CLASS (parent_class)->connect_proxy (action, proxy);

	entry = egg_entry_action_get_entry_widget(EGG_ENTRY_ACTION(action), proxy);
	if (entry)
	{
		kz_entry_set_backtext(KZ_ENTRY(entry), 
				      _("Search in History"));
		g_signal_connect(G_OBJECT(entry), "key-press-event",
				 G_CALLBACK(cb_entry_key_press), action);
	}
}


static void
kz_history_search_action_disconnect_proxy (EggAction *action, GtkWidget *proxy)
{
	GtkEntry *entry;

	entry = egg_entry_action_get_entry_widget(EGG_ENTRY_ACTION(action), proxy);
	if (entry)
	{
		g_signal_handlers_disconnect_by_func
			(G_OBJECT(entry),
			 G_CALLBACK(cb_entry_key_press),
			 action);
	}

	EGG_ACTION_CLASS (parent_class)->disconnect_proxy (action, proxy);
}


KzHistorySearchAction *
kz_history_search_action_new (KzWindow *kz)
{
	KzHistorySearchAction *action;

	action = KZ_HISTORY_SEARCH_ACTION(
			g_object_new(KZ_TYPE_HISTORY_SEARCH_ACTION,
				     "name",      "HistorySearch",
				     "label",     _("Location Entry"),
				     "tooltip",   NULL,
				     "stock_id",  GTK_STOCK_NEW,
				     "kz-window", kz,
				     NULL));
	return action;
}


static gboolean
cb_entry_key_press (GtkWidget *widget, GdkEventKey *event,
		    KzHistorySearchAction *action)
{
	if ((event->keyval == GDK_Return || event->keyval == GDK_ISO_Enter)
	    && (event->state & GDK_CONTROL_MASK))
	{
		kz_history_search_action_open_result(action);

		return TRUE;
	}

	return FALSE;
}


static gboolean
execute_search_command(const gchar *search_text, gint *standard_output)
{
	gboolean ret;
	const gchar *estxview = "estxview ";
	gchar *command;
	gint argc;
	gchar **argv = NULL;
	GSpawnFlags flags;
	GPid pid;
	gint err;

	command = g_strconcat(estxview,
			      "-max 20 ",
			      g_get_home_dir(), "/.kazehakase/history_index ",
			      search_text,
			      NULL);

	g_shell_parse_argv(command,
			   &argc,
			   &argv,
			   NULL);

	flags = G_SPAWN_SEARCH_PATH; 
	ret = g_spawn_async_with_pipes(NULL,
				       argv,
				       NULL,
				       flags,
				       NULL,
				       NULL,
				       &pid,
				       NULL,
				       standard_output,
				       &err,
				       NULL);
	g_strfreev(argv);
	g_free(command);

	return ret;
}


static gchar *
get_attr(const gchar *line, const gchar *attr_name)
{
	gchar *pos1, *pos2, *attr = NULL;
	size_t len = strlen(attr_name);
	gchar *dummy = g_strdup_printf("%s=\"", attr_name);

	pos1 = strstr(line, dummy);

	if (pos1)
		pos2 = strchr(pos1 + len + 2, '"');
	if (pos1 && pos2)
	{
		attr = g_strndup(pos1 + len + 2,
				 pos2 - pos1 - len - 2);
	}
	g_free(dummy);
	return attr;
}


static gchar *
get_element(const gchar *line)
{
	gchar *pos1, *pos2, *element = NULL;
	pos1 = strchr(line, '>');
	if (pos1)
		pos2 = strchr(pos1, '<');
	if (pos1 && pos2)
	{
		element = g_strndup(pos1 + 1,
				    pos2 - pos1 -1);
	}
	return element;
}


static void
show_result (gint out, GtkWidget *embed, gint n_text)
{
	GIOChannel *io;
	gchar *line;
	gsize length;
	gchar *title = NULL, *uri = NULL, *date = NULL;
	gchar *cache_link = NULL;
	gchar **search_word = NULL;
	gint n_search_word = 0;
	gchar *estversion = NULL;
	GString *desc, *html;
	GtkMozEmbed *mozembed;

	g_return_if_fail(GTK_IS_MOZ_EMBED(embed));
	mozembed = GTK_MOZ_EMBED(embed);

	io = g_io_channel_unix_new(out);
	g_io_channel_set_encoding(io, NULL, NULL);
	
	html = g_string_sized_new(0);
	search_word = g_new0(gchar*, n_text + 1);

	g_string_append(html, DTD"\n");
	g_string_append(html, "<html>\n");
	g_string_append_printf(html, HEAD, STYLE); 
	g_string_append(html, "<body>\n");

#if 1
	while (g_io_channel_read_line(io, &line, &length, NULL, NULL) == G_IO_STATUS_NORMAL)
	{
		if (!strncmp(line, "<document", 9))
		{
		}
		else if (!strncmp(line, "</document>", 11))
		{
			g_string_append_printf(html,
					       CONTENT,
					       uri,
					       title,
					       "", /* thumbnail */
					       desc->str,
					       cache_link,
					       date);

			g_string_free(desc, TRUE);
			g_free(title);
			g_free(uri);
			g_free(date);
			g_free(cache_link);
		}
		else if (!strncmp(line, "<uri>", 5))
		{
			gchar *dirname, *orig_uri;
			size_t len;
			cache_link = get_element(line);
			dirname = g_strconcat(g_get_home_dir(), 
					      "/.kazehakase/history/",
					      NULL);
			len = strlen(dirname);
			orig_uri = create_uri_from_filename(cache_link + len);
			uri = url_decode(orig_uri);
			g_free(orig_uri);
			g_free(dirname);
		}
		else if (!strncmp(line, "<title>", 7))
		{
			title = get_element(line);
		}
		else if (!strncmp(line, "<date>", 6))
		{
			date = get_element(line);
		}
		else if (!strncmp(line, "<text>", 6))
		{
			desc = g_string_sized_new(0);
		}
		else if (!strncmp(line, "<word", 5))
		{
			gchar *attr, *element;
			
			if (desc->len > DESCRIPTION_LEN)
				continue;
			
			attr = get_attr(line, "normal");
			element = get_element(line);
			if (element)
			{
				gboolean ascii = TRUE, keyword = FALSE;
				gint i, len;
				len = strlen(element);
				for (i = 0; i < len; i++)
				{
					if (!isascii(element[i]))
					{
						ascii = FALSE;
						break;
					}
				}
				for (i = 0; i < n_text; i++)
				{
					if (!strcmp(search_word[i], attr))
					{
						keyword = TRUE;
						break;
					}
				}
				if (keyword)
				{
					g_string_append_printf(desc,
							       "<span class=\"keyword %d\">%s</span>",
							       i,
							       element);
				}
				else
				{
					g_string_append(desc, element);
				}
				if (ascii)
					g_string_append_c(desc, ' ');
				g_free(attr);
				g_free(element);
			}
		}
		else if (!strncmp(line, "<query", 6))
		{
			search_word[n_search_word] = get_element(line);
			n_search_word++;
		}
		else if (!strncmp(line, "<estxview", 9))
		{
			estversion = get_attr(line, "version");
		}
		g_free(line);
	}
#endif

	g_io_channel_unref(io);

	g_string_append_printf(html, FOOTER, ESTRAIER_URI, estversion);
	g_string_append(html, "</body></html>");

	gtk_moz_embed_render_data(mozembed,
				  html->str,
				  html->len,
				  "file://",
				  "text/html");
	if (search_word)
		g_strfreev(search_word);
	if (estversion)
		g_free(estversion);
	g_string_free(html, TRUE);
}


static void
kz_history_search_action_open_result (KzHistorySearchAction *action)
{
	const gchar *text;
	gboolean new_tab;
	gint out;
	gint n_text = 0;
	gchar **text_array = NULL;
	GtkWidget *embed;
	
	KZ_CONF_GET("Global", "entry_open_in_new_tab", new_tab, BOOL);

	text = egg_entry_action_get_text(EGG_ENTRY_ACTION(action));
	if (!text) return;

	text_array = g_strsplit(text, " ", -1);
	while (text_array[n_text] != NULL) n_text++;
	g_strfreev(text_array);

	if (!execute_search_command(text, &out))
		return;

#warning FIXME!
/*	if (new_tab)
		embed = kz_window_open_new_tab(action->kz, "");
	else */
		embed = KZ_WINDOW_CURRENT_PAGE(action->kz);

	show_result(out, embed, n_text);
}
