/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003-2004 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include "utils.h"
#include "config.h"
#include "kazehakase.h"

void make_dir(const gchar *dirname)
{
	gchar *temp;
	temp = g_strdup_printf("%s/.%s/%s", g_get_home_dir(), PACKAGE, dirname);
	if (!g_file_test(temp, G_FILE_TEST_IS_DIR))
	{
		mkdir(temp, 0755);
	}
	g_free(temp);	
}


gboolean
key_seems_sequential (const gchar *key, const gchar *prefix)
{
	gint len = strlen(prefix);
	const gchar *tail = key + len;
	gint i;
	gboolean valid;

	g_return_val_if_fail(key && *key, FALSE);
	g_return_val_if_fail(prefix && *prefix, FALSE);

	len = strlen(prefix);
	tail = key + len;

	if (strncmp(key, prefix, len)) return FALSE;

	len = strlen(tail);
	if (len <= 0) return FALSE;

	valid = TRUE;
	for (i = 0; i < len; i++)
	{
		if (!isdigit(tail[i]))
		{
			valid = FALSE;
			break;
		}
	}

	return valid;
}


gchar *
remove_tag (const gchar *string, gsize len)
{
	GString *work_string;
	gchar *begin_pos;
	guint i = 0;
	
	work_string = g_string_new(NULL);
	begin_pos = strstr(string, "<");
	while (string[i] != '\0' && i < len)
	{
		if (string[i] == '<')
		{
			while (string[i] != '>' && string[i] != '\0' && i < len)
			{
				i++;
			}
		}
		else
			g_string_append_c(work_string, string[i]);
		i++;
	}
	
	return g_string_free(work_string, FALSE);
}


gchar *
create_filename_from_uri (const gchar *uri)
{
	gchar *filename;
	gint len, i = 0;
	
	filename = g_strdup(uri);
	len = strlen(filename);

	while (filename[i] != '\0' && i < len)
	{
		if (filename[i] == '/')
			filename[i] = '_';
		i++;
	}

	return filename;
}


gchar *
create_filename_with_path_from_uri (const gchar *uri)
{
	gchar *filename;
	gchar *pos;
	gchar *scheme;
	
	pos = strstr(uri, "://");
	
	if (!pos)
	{
		pos = (gchar*)uri;
		scheme = g_strdup("");
	}
	else
	{
		scheme = g_strndup(uri, pos - uri);
		pos += 3;
	}

	if (g_str_has_suffix(uri, "/"))
	{
		filename = g_strconcat(scheme, "/", pos, "_", NULL);
	}
	else
	{
		gchar *pos2 = strchr(uri, '?');
		if (pos2)
		{
			gchar *string = g_strndup(pos, pos2 - pos);
			filename = g_strconcat(scheme,
					       "/",
					       string, 
					       "_/",
					       pos2 + 1,
					       NULL);
			g_free(string);
		}
		else
		{
			filename = g_strconcat(scheme,
						"/",
						pos,
						NULL);
		}
	}

	g_free(scheme);
	
	return filename;
}


gchar *
create_uri_from_filename (const gchar *filename)
{
	gchar *uri;
	gchar *pos;
	gchar *scheme;
	gchar *path;
	
	pos = strstr(filename, "/");
	
	if (!pos)
		return NULL;

	scheme = g_strndup(filename, pos - filename);

	if (g_str_has_suffix(filename, "/_"))
	{
		path = g_strndup(pos + 1, strlen(pos) - 2); 
	}
	else
	{
		gchar *pos2 = g_strrstr(filename, "_/");
		if (pos2)
		{
			gchar *string = g_strndup(pos + 1, pos2 - pos - 1);
			path = g_strconcat(string, 
					   "?",
					   pos2 + 2,
					   NULL);
			
			g_free(string);
		}
		else
		{
			path = g_strdup(pos + 1);
		}
	}

	if (!strcmp(scheme, "file"))
	{
		uri = g_strconcat(scheme, ":///", path, NULL);
	}
	else
	{
		uri = g_strconcat(scheme, "://", path, NULL);
	}
	
	g_free(scheme);
	g_free(path);

	return uri;
}


gchar *
url_decode(const gchar *src)
{
	GString *dest;
	gint len, i=0;

	if (!src) return NULL;
	len = strlen(src);
	dest = g_string_sized_new(len);

	while (src[i] != '\0' && i < len)
	{
		if (src[i] == '%')
		{
			if (i + 2 <= len &&
			    g_ascii_isxdigit(src[i+1]) && 
			    g_ascii_isxdigit(src[i+2]))
			{				
				g_string_append_c(dest,
						  g_ascii_xdigit_value(src[i+1]) * 16 + 
						  g_ascii_xdigit_value(src[i+2]));
				i+=2;
			}
		}
		else
		{
			g_string_append_c(dest, src[i]);
		}
		i++;
	}

	/* Free gstring and reserve its data.*/
	return g_string_free(dest, FALSE);
}


gchar *
create_profile_key_from_uri (const gchar *string)
{
	gchar *key, *pos;
	gint len, i = 0;
	
	if (!string)
		return NULL;
	pos = strchr(string, '?');
	len = strlen(string);
	if (pos)
		len = pos - string;
	key = g_strndup(string, len);

	while (key[i] != '\0' && i < len)
	{
		if (key[i] == '=')
			key[i] = '_';
		i++;
	}

	return key;
}

/* return hex-encoded UTF-8 data
 * please free returned gchar* if unnecessary
 */
gchar *
url_encode(const gchar* utf8src)
{
	GString *dest;
	const gchar *ch = utf8src;
	unsigned char buf;

	if (!utf8src) return "";

	dest = g_string_sized_new(strlen(utf8src));

	while(*ch != '\0')
	{
		if (((*ch >= 'A') && (*ch <= 'Z')) ||
		    ((*ch >= 'a') && (*ch <= 'z')) ||
		    ((*ch >= '0') && (*ch <= '9')))
		{
			g_string_append_c(dest, *ch);
		}
		else if (*ch == ' ')
		{
			g_string_append_c(dest, '+');

		}
		else
		{
			g_string_append_c(dest, '%');
			buf = (*ch >> 4) & 0x0f;
			g_string_append_c(dest,
					( (buf < 10) ? buf + '0' : buf + ('A' - 10)) );
			buf = *ch & 0x0f;
			g_string_append_c(dest,
					( (buf < 10) ? buf + '0' : buf + ('A' - 10)) );
		}
		ch++;
	}
	
	/* Free gstring and reserve its data.*/
	return g_string_free(dest, FALSE);
}


gchar *
complement_scheme (const gchar* url)
{
	gchar *file;

	if (g_file_test(url, G_FILE_TEST_EXISTS))
	{
		if (!g_path_is_absolute(url))
		{
			gchar *current= g_get_current_dir();
			file = g_strdup_printf("file://%s/%s",
					       current,
					       url);
			g_free(current);
		}
		else
		{
			file = g_strdup_printf("file://%s",
					       url);
		}
		return file;
	}
	else
		return g_strdup(url);
}

gchar *
xml_get_content (const gchar *buffer)
{
	gchar *pos1, *pos2, *pos3;
	gchar *name;
	gchar *content = NULL;
	
	pos1 = strchr(buffer, '>');
	pos3 = strchr(buffer, ' ');
	if (pos1)
	{
		gchar *string;
		guint len;
		if (pos3 && pos1 > pos3)
			len = pos3 - buffer - 1;
		else
			len = pos1 - buffer - 1;
		name = g_strndup(buffer + 1, len); 
		string = g_strconcat("</", name, NULL);
		pos2 = strstr(pos1, string);
		if (pos2)
			content = g_strndup(pos1 +1 , pos2 - pos1 - 1);
		g_free(string);
		g_free(name);
	}
	return content;
}

gchar *
xml_get_attr (const gchar *buffer, const gchar *attr_name)
{
	gchar *pos1, *pos2;
	gchar *string1, *string2;
	gchar *attr = NULL;
	guint len;

	pos1 = strchr(buffer, '>');
	if (pos1)
		string1 = g_strndup(buffer, pos1 - buffer);
	else 
		return NULL;
	
	string2 = g_strdup_printf("%s=\"", attr_name);
	len = strlen(string2);
	pos1 = strstr(string1, string2); 
	if (pos1)
	{
		pos2 = strchr(pos1 + len, '"'); 
		if (pos2)
			attr = g_strndup(pos1 + len, pos2 - pos1 -len); 
	}
	g_free(string1);
	g_free(string2);
	
	return attr;

}


void
purge_history_file(const gchar *path, time_t limit_seconds)
{
	GDir *gdir;
	const gchar *dir;

	gdir = g_dir_open(path, 0, NULL);

	if (!gdir) return;

	while((dir = g_dir_read_name(gdir)))
	{
		gchar *dirname;
		dirname = g_strconcat(path, dir, NULL);
		if (g_file_test(dirname, G_FILE_TEST_IS_DIR))
		{
			purge_history_file(dirname, limit_seconds);
		}
		else
		{
			struct stat st;
			time_t t;
			GTimeVal now;
	
			g_get_current_time(&now);
			t = (time_t)now.tv_sec;
		
			if (stat(dirname, &st) == 0)
			{
				if (st.st_mtime < t - limit_seconds)
				{
					unlink(dirname);
				}
			}
		}
		g_free(dirname);
	}
	g_dir_close(gdir);
}

