/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "prefs_lang.h"

#include <string.h>

#include "kazehakase.h"
#include "kz-dlist.h"
#include "mozilla-prefs.h"
#include "utils/intl.h"

#define DATA_KEY "KzPrefsLang::info"

static GtkWidget     *prefs_lang_create          (void);
static void           prefs_lang_response        (GtkWidget *widget,
						  gint       response);


static KzPrefsWinPageEntry prefs_entry =
{
	path:            N_("/Language"),
	priority_hint:   0,
	ui_level:        KZ_UI_LEVEL_ALL,
	create:          prefs_lang_create,
	response:        prefs_lang_response,
};


typedef struct _KzPrefsLang
{
	GtkWidget     *main_vbox;
	GtkWidget     *option_menu_lang;
	GtkWidget     *option_menu_autodetect;
	GtkWidget     *lang_dlist;
	gboolean       enabled_list_updated;
} KzPrefsLang;

#define CREATE_MENU(menu, array, n_array, field)                               \
{                                                                              \
	gint i;                                                                \
                                                                               \
	menu = gtk_menu_new ();                                                \
                                                                               \
	for (i = 0; i < n_array; i++)                                          \
	{                                                                      \
		GtkWidget *item;                                               \
                                                                               \
		item = gtk_menu_item_new_with_label (_(array[i].field));       \
		gtk_menu_shell_append (GTK_MENU_SHELL(menu), item);            \
		gtk_widget_show (item);                                        \
	}                                                                      \
}


#define SET_DEFAULT_HISTORY(menu, key, array, n_array, field)                  \
{                                                                              \
	gchar *str;                                                            \
	gint i;                                                                \
                                                                               \
	if (mozilla_prefs_get_string(key, &str))                               \
	{                                                                      \
		for (i = 0; i < n_array; i++)                                  \
		{                                                              \
			if (!strcmp(str, array[i].field))                      \
			{                                                      \
				gtk_option_menu_set_history(GTK_OPTION_MENU(menu), i); \
				break;                                         \
			}                                                      \
		}                                                              \
		g_free(str);                                                   \
	}                                                                      \
}


static void
prefs_lang_destroy (gpointer data)
{
	KzPrefsLang *prefsui = data;

	g_free(prefsui);
}


static void
cb_enabled_list_updated (KzDList *dlist, KzPrefsLang *prefsui)
{
	prefsui->enabled_list_updated = TRUE;
}


static GtkWidget *
prefs_lang_create (void)
{
	KzPrefsLang *prefsui = g_new0(KzPrefsLang, 1);
	GtkWidget *main_vbox, *vbox, *hbox;
	GtkWidget *label, *table, *option_menu, *menu, *frame,*dlist;
	gchar *str, **langs;
	gint i;

	prefsui->enabled_list_updated = FALSE;

	main_vbox = gtk_vbox_new (FALSE, 0);
	prefsui->main_vbox = main_vbox;
	g_object_set_data_full (G_OBJECT(main_vbox),
				DATA_KEY, prefsui,
				(GDestroyNotify) prefs_lang_destroy);

	label = kz_prefs_ui_utils_create_title(_("Language"));
	gtk_box_pack_start(GTK_BOX(main_vbox), label,
			   FALSE, FALSE, 0);
	gtk_widget_show(label);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), vbox,
			   TRUE, TRUE, 0);
	gtk_widget_show(vbox);

	table = gtk_table_new(2, 2, FALSE);
	gtk_box_pack_start(GTK_BOX(vbox), table,
			   FALSE, FALSE, 0);
	gtk_widget_show(table);

	/*
	 *  Default Encoding
	 */
	frame = gtk_frame_new(NULL);
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 5);
	gtk_widget_show(frame);

	label = gtk_label_new_with_mnemonic(_("_Default Encoding"));
	gtk_label_set_use_underline(GTK_LABEL(label), TRUE);
	gtk_frame_set_label_widget(GTK_FRAME(frame), label);
	gtk_widget_show(label);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
	gtk_container_add(GTK_CONTAINER(frame), hbox);
	gtk_widget_show(hbox);

	option_menu = gtk_option_menu_new();
	prefsui->option_menu_lang = option_menu;
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), option_menu);
	gtk_box_pack_start(GTK_BOX(hbox), option_menu, FALSE, FALSE, 0);
	gtk_widget_show(option_menu);

	CREATE_MENU(menu, languages, n_languages, name);

	gtk_option_menu_set_menu (GTK_OPTION_MENU(option_menu), menu);
	SET_DEFAULT_HISTORY(option_menu, "intl.charset.default",
			    languages, n_languages, code);
	gtk_widget_show(menu);

	/*
	 *  Auto detect
	 */
	frame = gtk_frame_new(NULL);
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 5);
	gtk_widget_show(frame);

	label = gtk_label_new_with_mnemonic(_("A_utodetect Encoding"));
	gtk_frame_set_label_widget(GTK_FRAME(frame), label);
	gtk_widget_show(label);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
	gtk_container_add(GTK_CONTAINER(frame), hbox);
	gtk_widget_show(hbox);

	option_menu = gtk_option_menu_new();
	prefsui->option_menu_autodetect = option_menu;
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), option_menu);
	gtk_box_pack_start(GTK_BOX(hbox), option_menu, FALSE, FALSE, 0);
	gtk_widget_show(option_menu);

	CREATE_MENU(menu, encoding_autodetectors, n_encoding_autodetectors, title);
	gtk_option_menu_set_menu (GTK_OPTION_MENU(option_menu), menu);
	SET_DEFAULT_HISTORY(option_menu, "intl.charset.detector",
			    encoding_autodetectors, n_encoding_autodetectors,
			    name);
	gtk_widget_show(menu);

	/*
	 *  Accept Language
	 */
	frame = gtk_frame_new(_("Accept Language"));
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 5);
	gtk_widget_show(frame);

	dlist = kz_dlist_new(_("Available Languages"),
			     _("Enabled Languages"));
	prefsui->lang_dlist = dlist;
	gtk_container_set_border_width(GTK_CONTAINER(dlist), 5);
	gtk_container_add(GTK_CONTAINER(frame), dlist);
	gtk_widget_show(dlist);

	for (i = 0; i < n_accept_languages; i++)
	{
		const gchar *name = accept_languages[i].name;
		const gchar *code = accept_languages[i].code;

		kz_dlist_append_available_item(KZ_DLIST(dlist), name, code);
	}

	mozilla_prefs_get_string("intl.accept_languages", &str);
	langs = g_strsplit(str, ",", -1);
	g_free(str);
	if (langs)
	{
		for (i = 0; langs[i]; i++)
		{
			g_strstrip(langs[i]);
			kz_dlist_column_add_by_id(KZ_DLIST(dlist), langs[i]);
		}
		g_strfreev(langs);
	}
	g_signal_connect(G_OBJECT(dlist), "enabled-list-updated",
			 G_CALLBACK(cb_enabled_list_updated), prefsui);

	return main_vbox;
}


#define GET_HISTORY(option_menu) \
	(gtk_option_menu_get_history(GTK_OPTION_MENU(option_menu)))


static void
prefs_lang_apply (KzPrefsLang *prefsui)
{
	gint i, j;

	g_return_if_fail(prefsui);

	i = GET_HISTORY(prefsui->option_menu_lang);
	j = GET_HISTORY(prefsui->option_menu_autodetect);
	if (i >= 0 && i < n_languages)
	{
		mozilla_prefs_set_string("intl.charset.default",
					 languages[i].code);
	}
	if (j >= 0 && j < n_encoding_autodetectors)
	{
		if (mozilla_prefs_set_string("intl.charset.detector",
					     encoding_autodetectors[j].name));
	}

	if (prefsui->enabled_list_updated)
	{
		KzDList *dlist = KZ_DLIST(prefsui->lang_dlist);
		gint i, num = kz_dlist_get_n_enabled_items(dlist);
		gchar *langs = strdup("");

		for (i = 0; i < num; i++)
		{
			gchar *lang, *tmp;
			lang = kz_dlist_get_enabled_id(dlist, i);
			tmp = langs;
			if (i == 0)
				langs = g_strconcat(langs, lang, NULL);
			else
				langs = g_strconcat(langs, ",", lang, NULL);
			g_free(tmp);
			g_free(lang);
		}
		mozilla_prefs_set_string("intl.accept_languages", langs);
		g_free(langs);
		prefsui->enabled_list_updated = FALSE;
	}
}


static void
prefs_lang_response (GtkWidget *widget, gint response)
{
	KzPrefsLang *prefsui = g_object_get_data(G_OBJECT(widget), DATA_KEY);

	g_return_if_fail(prefsui);

	switch (response) {
	case GTK_RESPONSE_ACCEPT:
	case GTK_RESPONSE_APPLY:
		prefs_lang_apply(prefsui);
		break;
	case GTK_RESPONSE_REJECT:
		break;
	case KZ_RESPONSE_UI_LEVEL_MEDIUM:
		break;
	case KZ_RESPONSE_UI_LEVEL_EXPERT:
		break;
	case KZ_RESPONSE_UI_LEVEL_CUSTOM:
		break;
	default:
		break;
	}
}


KzPrefsWinPageEntry *
prefs_lang_get_entry (gint idx)
{
	if (idx == 0)
		return &prefs_entry;
	else
		return NULL;
}
