#
#   Copyright (C) 2006 Eriko Sato
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2, or (at your option)
#   any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copyED of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#

module Kz
  class SandBox
    def initialize(kz)
      @kz = kz
      @binding = binding
    end

    def evaluate(statements, file=__FILE__, line=__LINE__)
      eval(statements, @binding, file, line)
    end
  end

  class RubyDialog
    def initialize(kz)
      @kz = kz
      @sandbox = SandBox.new(@kz)
      @default_font_size = 14
      init_dialog
      @dialog.show_all
      @entry.grab_focus
    end

    def _(str)
      Kz.gettext(str)
    end

    def init_dialog
      @dialog = Gtk::Dialog.new(_("Ruby dialog"))
      @dialog.set_size_request(400, 300)
      @dialog.signal_connect("destroy") do |widget, event|
        false
      end
      init_history
      init_output_area
      init_input_area
      init_buttons
      init_redirector
    end

    def init_history
      @history = []
    end

    def init_output_area
      init_text_view
    end

    def init_text_view
      @view = Gtk::TextView.new
      @view.wrap_mode = Gtk::TextTag::WRAP_WORD_CHAR
      @view.editable = false
      @buffer = @view.buffer
      init_mark
      init_tags
      @buffer.signal_connect_after("insert_text") do |widget, iter, text, len|
        start = @buffer.get_iter_at_offset(iter.offset - len)
        @buffer.apply_tag(@all_tag, start, iter)
        false
      end
      sw = add_scrooled_window(@view)
      @dialog.vbox.pack_start(sw, true, true, 0)
    end

    def init_input_area
      init_history_spin_button
      init_input_entry

      input_hbox = Gtk::HBox.new
      input_hbox.pack_start(@history_spin, false, true, 0)
      input_hbox.pack_start(@entry, true, true, 0)
      @dialog.vbox.pack_start(input_hbox, false, true, 0)
    end

    def init_history_spin_button
      adjustment = Gtk::Adjustment.new(@history.size, 0, @history.size, 1, 4, 0)
      @history_spin = Gtk::SpinButton.new(adjustment, 1, 0)
      @history_spin.signal_connect("value-changed") do |widget, type|
        update_input_entry if widget.value < @history.size
        false
      end
    end

    def init_input_entry
      @entry = Gtk::Entry.new
      @entry.signal_connect("key_press_event") do |widget, event|
        handle_input(event)
      end
      @entry
    end

    def update_input_entry
      @entry.text = @history[@history_spin.value]
      @entry.position = -1
    end

    def init_mark
      @end_mark = @buffer.create_mark("end", @buffer.end_iter, true)
    end

    def init_tags
      result_tag_prop = {:foreground => "HotPink"}
      @result_tag = @buffer.create_tag("result", result_tag_prop)
      input_tag_prop = {:foreground => "Green"}
      @input_tag = @buffer.create_tag("input", input_tag_prop)
      output_tag_prop = {:foreground => "Blue"}
      @output_tag = @buffer.create_tag("output", output_tag_prop)
      all_tag_prop = {
        :family => "monospace",
        :size_points => @default_font_size
      }
      @all_tag = @buffer.create_tag("all", all_tag_prop)
    end

    def handle_input(event)
      handled = false
      case event.keyval
      when Gdk::Keyval::GDK_Return
        text = @entry.text.strip
        unless text.empty?
          eval_print(text)
          @history << text
          @history_spin.adjustment.upper = @history.size
          @history_spin.value = @history.size
        end
        @entry.text = ""
      when Gdk::Keyval::GDK_p
        handled = previous_history if event.state.control_mask?
      when Gdk::Keyval::GDK_Up
        handled = previous_history
        handled = true
      when Gdk::Keyval::GDK_n
        handled = next_history if event.state.control_mask?
      when Gdk::Keyval::GDK_Down
        handled = next_history
        handled = true
      end
      handled
    end

    def previous_history
      handled = false
      if @history_spin.value > 0
        @history_spin.value -= 1
        update_input_entry
        handled = true
      end
      handled
    end

    def next_history
      handled = false
      if @history_spin.value < @history.size
        update_input_entry
        @history_spin.value += 1
        handled = true
      end
      handled
    end

    def eval_print(text)
      @buffer.insert(@buffer.end_iter, ">> ")
      @buffer.insert(@buffer.end_iter, text, @input_tag)
      @buffer.insert(@buffer.end_iter, "\n")
      result = eval_text(text).inspect
      @buffer.insert(@buffer.end_iter, "=> ")
      @buffer.insert(@buffer.end_iter, result, @result_tag)
      @buffer.insert(@buffer.end_iter, "\n")
      @buffer.move_mark(@end_mark, @buffer.end_iter)
      @view.scroll_to_mark(@end_mark, 0, false, 0, 1)
    end

    def add_scrooled_window(widget)
      sw = Gtk::ScrolledWindow.new
      sw.border_width = 5
      sw.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC)
      sw.add(widget)
      sw
    end
    
    def init_buttons
      init_font_size_spin_button
      init_exit_button
    end

    def init_font_size_spin_button
      adjustment = Gtk::Adjustment.new(@default_font_size, 8, 72, 1, 4, 0)
      button = Gtk::SpinButton.new(adjustment, 1, 0)
      button.signal_connect("value-changed") do |widget, type|
        @all_tag.size_points = widget.value
        false
      end
      @dialog.action_area.add(button)
      @dialog.action_area.set_child_secondary(button, true)
    end
    
    def init_exit_button
      button = Gtk::Button.new("Exit")
      button.signal_connect("clicked") do |widget, event|
        @dialog.destroy
      end
      @dialog.action_area.add(button)
    end

    def init_redirector
      @buffer_out = Object.new
      @buffer_out.instance_variable_set("@buffer", @buffer)
      @buffer_out.instance_variable_set("@output_tag", @output_tag)
      class << @buffer_out
        def write(str)
          @buffer.insert(@buffer.end_iter, str, @output_tag)
        end
      end
    end

    def eval_text(text)
      redirect do
        @sandbox.evaluate(text)
      end
    rescue Exception
      $!
    end

    def redirect
      stdout = $stdout
      $stdout = @buffer_out
      yield
    ensure
      $stdout = stdout
    end
  end
end
