/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

#include <string.h>
#include <glib/gi18n.h>

#include "gtk-utils.h"
#include "kz-feed-info.h"
#include "kz-icons.h"

enum {
	PROP_0,
	PROP_KZ_WINDOW
};

static void dispose      (GObject            *object);
static void set_property (GObject            *object,
                          guint               prop_id,
                          const GValue       *value,
                          GParamSpec         *pspec);
static void get_property (GObject            *object,
                          guint               prop_id,
                          GValue             *value,
                          GParamSpec         *pspec);

static void kz_feed_info_setup_for_with_feed    (KzFeedInfo *info);
static void kz_feed_info_setup_for_without_feed (KzFeedInfo *info);


static void cb_add_feed_menu_activate (GtkWidget      *widget,
				       KzNavi         *navi);
static void cb_feed_info_button_press (GtkWidget      *widget,
				       GdkEventButton *event,
				       KzFeedInfo     *info);

G_DEFINE_TYPE(KzFeedInfo, kz_feed_info, GTK_TYPE_IMAGE_MENU_ITEM)

static void
kz_feed_info_class_init (KzFeedInfoClass *klass)
{
	GObjectClass *gobject_class;

	gobject_class = G_OBJECT_CLASS(klass);

	/* GtkObject signals */
	gobject_class->dispose      = dispose;
	gobject_class->set_property = set_property;
	gobject_class->get_property = get_property;

	g_object_class_install_property
		(gobject_class,
		 PROP_KZ_WINDOW,
		 g_param_spec_object("kz-window",
				     _("KzWindow"),
				     _("The parent kazehakase window"),
				     KZ_TYPE_WINDOW,
				     G_PARAM_READWRITE |
				     G_PARAM_CONSTRUCT_ONLY));
}

static void
kz_feed_info_init (KzFeedInfo *info)
{
	/* widgets */
	info->kz = NULL;
	
	info->feed_list = NULL;
}


static void
set_property (GObject *object, guint prop_id,
              const GValue *value, GParamSpec *pspec)
{
	KzFeedInfo *info = KZ_FEED_INFO(object);
  
	switch (prop_id)
	{
	case PROP_KZ_WINDOW:
		info->kz = g_object_ref(g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}

static void
get_property (GObject *object, guint prop_id,
              GValue *value, GParamSpec *pspec)
{
	KzFeedInfo *info = KZ_FEED_INFO(object);

	switch (prop_id)
	{
	case PROP_KZ_WINDOW:
		g_value_set_object(value, info->kz);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}


GtkWidget *
kz_feed_info_new (KzWindow *kz)
{
	KzFeedInfo *info;
	GtkWidget *image, *accel_label;
	gchar *stock_id = KZ_STOCK_RSS_FEED;

	info = g_object_new(KZ_TYPE_FEED_INFO,
			    "kz-window", kz,
			    NULL);
	image = gtk_image_new_from_stock(stock_id, GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(info), image);
	gtk_widget_show(image);

	g_signal_connect(info, "button_press_event",
			 G_CALLBACK(cb_feed_info_button_press), info);

	accel_label = gtk_accel_label_new("");
	gtk_accel_label_set_accel_widget(GTK_ACCEL_LABEL(accel_label),
					 GTK_WIDGET(info));
	gtk_misc_set_alignment(GTK_MISC(accel_label), 0.0, 0.5);

	gtk_widget_show(accel_label);
	gtk_container_add(GTK_CONTAINER(info), accel_label);
	
	gtk_widget_set_tooltip_text(GTK_WIDGET(info), _("Subscribe this page's feed"));

	kz_feed_info_setup_for_without_feed(info);
	
	return GTK_WIDGET(info);
}

void
kz_feed_info_change_state (KzFeedInfo *info)
{
	if (kz_embed_get_nav_links(KZ_WINDOW_CURRENT_EMBED(info->kz),
				   KZ_EMBED_LINK_RSS))
		kz_feed_info_setup_for_with_feed(info);
	else
		kz_feed_info_setup_for_without_feed(info);
}

static void
dispose (GObject *object)
{
	KzFeedInfo *info = KZ_FEED_INFO(object);

	if (info->kz)
	{
		g_object_unref(info->kz);
		info->kz = NULL;
	}

	if (info->feed_list)
	{
		g_slist_free(info->feed_list);
		info->feed_list = NULL;
	}
	
	g_signal_handlers_disconnect_by_func(info,
					     G_CALLBACK(cb_feed_info_button_press),
					     NULL);

	if (G_OBJECT_CLASS(kz_feed_info_parent_class)->dispose)
		(*G_OBJECT_CLASS(kz_feed_info_parent_class)->dispose) (object);
}

static void
kz_feed_info_setup_for_with_feed (KzFeedInfo *info)
{
	gtk_widget_show(GTK_WIDGET(info));
}

static void
kz_feed_info_setup_for_without_feed (KzFeedInfo *info)
{
	gtk_widget_hide(GTK_WIDGET(info));
}

static void
cb_add_feed_menu_activate (GtkWidget *widget, KzNavi *navi)
{
	kz_navi_add_bookmark(navi);
}

static void
cb_feed_info_button_press (GtkWidget *widget,
			   GdkEventButton *event,
			   KzFeedInfo *info)
{
	GList *nav_links, *nav_link;
	GtkWidget *menu, *menuitem;
	KzEmbed *embed;
	gchar *label;

	embed = KZ_WINDOW_CURRENT_EMBED(info->kz);
	if (!embed) return;
	
	switch (event->button) {
	case 1:
	case 3:
		nav_links = kz_embed_get_nav_links(embed, KZ_EMBED_LINK_RSS);
		if (!nav_links) return;
		
		menu = gtk_menu_new();

		for (nav_link = nav_links;
		     nav_link;
		     nav_link = g_list_next(nav_link))
		{
			KzNavi *navi = nav_link->data;
			if (!(navi && navi->uri))
				continue;

			if (navi->title)
				label = g_strdup_printf(_("Add '%s' feed to bookmarks"),
							navi->title);
			else
				label = g_strdup_printf(_("Add the feed to bookmarks"));
			
			menuitem = gtk_menu_item_new_with_label(label);
			g_free(label);

			g_signal_connect(menuitem, "activate",
					 G_CALLBACK(cb_add_feed_menu_activate),
					 navi);
			gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);
			gtk_widget_show(menuitem);
		}
		
		gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
			       event->button, event->time);
		break;
	default:
		break;
	}

}
