/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-tablabel.h"

#include <stdlib.h>
#include <string.h>
#include "gobject-utils.h"
#include "gtk-utils.h"
#include "intl.h"
#include "kz-actions-tab.h"


struct _KzTabLabelPriv
{
	gint start_x, start_y;
	gboolean moved;
};


static void     kz_tab_label_class_init         (KzTabLabelClass *klass);
static void     kz_tab_label_init               (KzTabLabel *kztab);
static void     kz_tab_label_dispose            (GObject *object);
static void     kz_tab_label_finalize           (GObject *object);
static void     kz_tab_label_realize            (GtkWidget *widget);
static void     kz_tab_label_unrealize          (GtkWidget *widget);
static gboolean kz_tab_label_button_press       (GtkWidget *widget,
						 GdkEventButton *event);
static gboolean kz_tab_label_motion_notify      (GtkWidget *widget,
						 GdkEventMotion *event);
static gboolean kz_tab_label_button_release     (GtkWidget *widget,
						 GdkEventButton *event);
static gboolean kz_tab_label_scroll_event       (GtkWidget *widget,
						 GdkEventScroll *event);
static gboolean kz_tab_label_drag_motion        (GtkWidget *widget,
						 GdkDragContext *drag_context,
						 gint x, gint y,
						 guint time);
static void     kz_tab_label_drag_data_get      (GtkWidget        *widget,
						 GdkDragContext   *context,
						 GtkSelectionData *seldata,
						 guint             info,
						 guint             time);
static void     kz_tab_label_drag_data_received (GtkWidget *widget,
						 GdkDragContext *drag_context,
						 gint x, gint y,
						 GtkSelectionData *data,
						 guint info,
						 guint time);

static void     kz_tab_label_move_page          (KzTabLabel *kztab,
						 KzTabLabel *dest_kztab);

static void     cb_close_button_clicked         (GtkWidget *button,
						 KzTabLabel *kztab);

/* callbacks for mozilla */
static void cb_title_changed (GtkMozEmbed *embed, KzTabLabel *kztab);
static void cb_net_start     (GtkMozEmbed *embed, KzTabLabel *kztab);
static void cb_net_stop      (GtkMozEmbed *embed, KzTabLabel *kztab);
static void cb_destroy       (GtkObject *object, KzTabLabel *kztab);


static GtkEventBoxClass *parent_class = NULL;

enum {
	TARGET_KAZEHAKASE_TAB,
	TARGET_NETSCAPE_URL,
	TARGET_TEXT_URI_LIST,
	TARGET_TEXT_PLAIN,
	TARGET_STRING
};

static GtkTargetEntry url_drag_types [] = 
{
        { "_KAZEHAKASE_TAB", 0, TARGET_KAZEHAKASE_TAB},
        { "_NETSCAPE_URL",   0, TARGET_NETSCAPE_URL},
	{ "text/uri-list",   0, TARGET_TEXT_URI_LIST},
	{ "text/plain",      0, TARGET_TEXT_PLAIN},
	{ "STRING",          0, TARGET_STRING}
};
static guint n_url_drag_types = G_N_ELEMENTS (url_drag_types);
 
KZ_OBJECT_GET_TYPE(kz_tab_label, "KzTabLabel", KzTabLabel,
		   kz_tab_label_class_init, kz_tab_label_init,
		   GTK_TYPE_EVENT_BOX)

static void
kz_tab_label_class_init (KzTabLabelClass *klass)
{
	GObjectClass *gobject_class;
	GtkWidgetClass *widget_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *) klass;
	widget_class  = (GtkWidgetClass *) klass;

	/* GtkObject signals */
	gobject_class->dispose  = kz_tab_label_dispose;
	gobject_class->finalize = kz_tab_label_finalize;

	/* GtkWidget signales */
	widget_class->realize              = kz_tab_label_realize;
	widget_class->unrealize            = kz_tab_label_unrealize;
	widget_class->button_press_event   = kz_tab_label_button_press;
	widget_class->scroll_event         = kz_tab_label_scroll_event;
	widget_class->motion_notify_event  = kz_tab_label_motion_notify;
	widget_class->button_release_event = kz_tab_label_button_release;
	widget_class->drag_motion          = kz_tab_label_drag_motion;
	widget_class->drag_data_get        = kz_tab_label_drag_data_get;
	widget_class->drag_data_received   = kz_tab_label_drag_data_received;
}

static void
kz_tab_label_init (KzTabLabel *kztab)
{
	GtkWidget *image;
	gint i;

	kztab->kz           = NULL;
	kztab->kzembed      = NULL;
	kztab->state        = KZ_TAB_LABEL_STATE_NORMAL;
	kztab->hbox         = gtk_hbox_new(0, FALSE);
	kztab->label        = gtk_label_new(NULL);
	kztab->close_button = gtk_button_new();
	kztab->tooltips     = gtk_tooltips_new();
	for (i = 0; i < KZ_TAB_LABEL_N_STATE; i++)
		kztab->style[i] = NULL;

	kztab->priv = g_new0(KzTabLabelPriv, 1);
	kztab->priv->start_x = 0;
	kztab->priv->start_y = 0;
	kztab->priv->moved   = FALSE;

	/*  hbox */
	gtk_container_add (GTK_CONTAINER(kztab), kztab->hbox);
	gtk_widget_show(kztab->hbox);

	/* label */
	gtk_widget_set_size_request(kztab->label, 50, -1);
	gtk_widget_show (kztab->label);

	/* button */
	gtk_widget_set_size_request(kztab->close_button, 14, 14);
	gtk_button_set_relief(GTK_BUTTON(kztab->close_button), GTK_RELIEF_NONE);
	g_signal_connect(G_OBJECT(kztab->close_button), "clicked", 
			 G_CALLBACK(cb_close_button_clicked), kztab);

	gtk_box_pack_start(GTK_BOX(kztab->hbox), kztab->label, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(kztab->hbox), kztab->close_button, FALSE, FALSE, 0);

	image = gtk_image_new_from_stock(GTK_STOCK_CLOSE,
					 GTK_ICON_SIZE_MENU);
	gtk_container_add(GTK_CONTAINER(kztab->close_button), image);

	gtk_widget_show(image);
	gtk_widget_show (kztab->close_button);

	gtk_drag_source_set(GTK_WIDGET(kztab),
			    GDK_BUTTON1_MASK | GDK_BUTTON2_MASK | GDK_BUTTON3_MASK,
			    url_drag_types, n_url_drag_types,
			    GDK_ACTION_ASK  | GDK_ACTION_COPY
			    | GDK_ACTION_MOVE | GDK_ACTION_LINK);
	gtk_drag_dest_set(GTK_WIDGET(kztab), 
			  GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_MOTION | 
			  GTK_DEST_DEFAULT_DROP,
                          url_drag_types,n_url_drag_types,
			  GDK_ACTION_MOVE);

	kz_tab_label_set_text(kztab, NULL);
}

static void
kz_tab_label_dispose (GObject *object)
{
	KzTabLabel *kztab;

	g_return_if_fail (KZ_IS_TAB_LABEL(object));

	kztab = KZ_TAB_LABEL(object);

	if (kztab->kzembed)
	{
		g_signal_handlers_disconnect_by_func(G_OBJECT(kztab->kzembed),
						     cb_net_start, kztab);
		g_signal_handlers_disconnect_by_func(G_OBJECT(kztab->kzembed),
						     cb_net_stop, kztab);
		g_signal_handlers_disconnect_by_func(G_OBJECT(kztab->kzembed),
						     cb_title_changed, kztab);
		g_signal_handlers_disconnect_by_func(G_OBJECT(kztab->kzembed),
						     cb_destroy, kztab);
	}
	kztab->kzembed = NULL;

	if (kztab->tooltips)
	{
		g_object_unref(kztab->tooltips);
	}
	kztab->tooltips = NULL;

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}

KZ_OBJECT_FINALIZE (kz_tab_label, KzTabLabel)

GtkWidget *
kz_tab_label_new (KzWindow *kz, KzMozEmbed *kzembed)
{
	KzTabLabel *kztab = KZ_TAB_LABEL(g_object_new(KZ_TYPE_TAB_LABEL, NULL));
	gchar *title;

	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);
	g_return_val_if_fail(KZ_IS_MOZ_EMBED(kzembed), NULL);

	kztab->kz      = kz;
	kztab->kzembed = kzembed;

	/* set signals */
	g_signal_connect(G_OBJECT(kzembed), "title",
			 G_CALLBACK(cb_title_changed), kztab);
	g_signal_connect(G_OBJECT(kzembed), "net-start",
			 G_CALLBACK(cb_net_start), kztab);
	g_signal_connect(G_OBJECT(kzembed), "net-stop",
			 G_CALLBACK(cb_net_stop), kztab);
	g_signal_connect(G_OBJECT(kzembed), "destroy",
			 G_CALLBACK(cb_destroy), kztab);

	/* set label text */
	title = kz_moz_embed_ensure_title(kzembed);
	kz_tab_label_set_text(kztab, title);
	g_free(title);

	return GTK_WIDGET(kztab);
}

void
kz_tab_label_show_close_button (KzTabLabel *kztab)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));
	gtk_widget_show(kztab->close_button);
}

void
kz_tab_label_hide_close_button (KzTabLabel *kztab)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));
	gtk_widget_hide(kztab->close_button);
}

void
kz_tab_label_set_text(KzTabLabel *kztab, const gchar *text)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));

	if (!text || !*text) text = _("No title");

	gtk_label_set_text(GTK_LABEL(kztab->label), text);
	gtk_tooltips_set_tip(kztab->tooltips, GTK_WIDGET(kztab),
			     text, NULL);
}

void
kz_tab_label_set_state (KzTabLabel *kztab,
			KzTabLabelState state)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));

	if (!GTK_WIDGET_REALIZED(kztab)) return;

	switch(state)
	{
	case KZ_TAB_LABEL_STATE_LOADING:
		gtk_widget_modify_style(GTK_WIDGET(kztab->label),
					kztab->style[KZ_TAB_LABEL_STATE_LOADING]);
		break;
	case KZ_TAB_LABEL_STATE_NORMAL:
	default:
		gtk_widget_modify_style(GTK_WIDGET(kztab->label),
					kztab->style[KZ_TAB_LABEL_STATE_NORMAL]);
		break;
	}
}

KzTabLabelState
kz_tab_label_get_state (KzTabLabel *kztab)
{
	g_return_val_if_fail(KZ_IS_TAB_LABEL(kztab), KZ_TAB_LABEL_STATE_NORMAL);
	return kztab->state;
}

static void
kz_tab_label_realize (GtkWidget *widget)
{
	KzTabLabel *kztab = KZ_TAB_LABEL(widget);
	gint i;

	if (GTK_WIDGET_CLASS (parent_class)->realize)
		GTK_WIDGET_CLASS (parent_class)->realize(widget);

	/* new rc styles */
	if (!kztab->style[KZ_TAB_LABEL_STATE_LOADING])
	{
		GtkRcStyle *style = gtk_rc_style_new();

		/* set colors */
		for(i = 0; i< 5; i++)
		{
			style->color_flags[i]   = GTK_RC_FG;
			style->fg[i].red   = 0xffff;
			style->fg[i].blue  = 0x0000;
			style->fg[i].green = 0x0000;
		}
		kztab->style[KZ_TAB_LABEL_STATE_LOADING] = style;
	}

	if (!kztab->style[KZ_TAB_LABEL_STATE_NORMAL])
	{
		kztab->style[KZ_TAB_LABEL_STATE_NORMAL]
			= gtk_widget_get_modifier_style(widget);
		gtk_rc_style_ref(kztab->style[KZ_TAB_LABEL_STATE_NORMAL]);
	}

	if (kz_moz_embed_is_loading(kztab->kzembed))
		kz_tab_label_set_state (kztab, KZ_TAB_LABEL_STATE_LOADING);
}

static void
kz_tab_label_unrealize (GtkWidget *widget)
{
	KzTabLabel *kztab = KZ_TAB_LABEL(widget);
	gint i;

	for (i = 0; i < KZ_TAB_LABEL_N_STATE; i++)
	{
		if (kztab->style[i])
		{
			gtk_rc_style_unref(kztab->style[i]);
			kztab->style[i] = NULL;
		}
	}

	if (GTK_WIDGET_CLASS (parent_class)->unrealize)
		return GTK_WIDGET_CLASS (parent_class)->unrealize(widget);
}

static gboolean
kz_tab_label_button_press (GtkWidget *widget, GdkEventButton *event)
{
	KzTabLabel *kztab = KZ_TAB_LABEL(widget);

	kztab->priv->start_x = event->x;
	kztab->priv->start_y = event->y;
	kztab->priv->moved = FALSE;

	if (GTK_WIDGET_CLASS(parent_class)->button_press_event)
		GTK_WIDGET_CLASS(parent_class)->button_press_event(widget, event);
	return FALSE;
}

static gboolean
kz_tab_label_motion_notify (GtkWidget *widget, GdkEventMotion *event)
{
	KzTabLabel *kztab = KZ_TAB_LABEL(widget);

	if (abs(event->x - kztab->priv->start_x) > 2 ||
	    abs(event->y - kztab->priv->start_y) > 2)
	{
		kztab->priv->moved = TRUE;
	}

	if (GTK_WIDGET_CLASS(parent_class)->motion_notify_event)
		return GTK_WIDGET_CLASS(parent_class)->motion_notify_event(widget, event);
	return FALSE;
}

static gboolean
kz_tab_label_button_release (GtkWidget *widget, GdkEventButton *event)
{
	KzTabLabel *kztab = KZ_TAB_LABEL(widget);

	if (event->button == 2 && !kztab->priv->moved)
	{
		/* Load URL in clipboard */
		/* return TRUE; */
	}
	else if (event->button == 3 && !kztab->priv->moved)
	{
		kz_tab_label_popup_menu_modal(kztab,
					      event->button,
					      event->time);
		return TRUE;
	}

	kztab->priv->start_x = 0;
	kztab->priv->start_y = 0;
	kztab->priv->moved = FALSE;

	if (GTK_WIDGET_CLASS(parent_class)->button_release_event)
		return GTK_WIDGET_CLASS(parent_class)->button_release_event(widget, event);
	return FALSE;
}


static gboolean
kz_tab_label_scroll_event (GtkWidget *widget, GdkEventScroll *event)
{
	KzTabLabel *kztab;
	GtkNotebook *notebook;
	gboolean retval = FALSE;

	kztab = KZ_TAB_LABEL(widget);
	notebook = GTK_NOTEBOOK(kztab->kz->notebook);

	switch (event->direction) {
	case GDK_SCROLL_UP:
	case GDK_SCROLL_LEFT:
		gtk_notebook_prev_page(notebook);
		retval = TRUE;
		break;
	case GDK_SCROLL_DOWN:
	case GDK_SCROLL_RIGHT:
		gtk_notebook_next_page(notebook);
		retval = TRUE;
		break;
	default:
		g_warning ("Invalid scroll direction!");
		break;
	}

	if (GTK_WIDGET_CLASS(parent_class)->scroll_event)
		return GTK_WIDGET_CLASS(parent_class)->scroll_event(widget, event)
			|| retval;
	return retval;
}


static gboolean
kz_tab_label_drag_motion (GtkWidget *widget,
			  GdkDragContext *drag_context,
			  gint x, gint y,
			  guint time)
{
	KzTabLabel *kztab;
	KzMozEmbed *current_kzembed;
	gint page_num;

	g_return_val_if_fail(KZ_TAB_LABEL(widget), FALSE);
	kztab = KZ_TAB_LABEL(widget);

	g_return_val_if_fail (KZ_IS_MOZ_EMBED(kztab->kzembed), FALSE);

	current_kzembed = KZ_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kztab->kz));

	if (KZ_MOZ_EMBED(kztab->kzembed) == current_kzembed)
		return FALSE;

	page_num = gtk_notebook_page_num(GTK_NOTEBOOK(kztab->kz->notebook),
					 GTK_WIDGET(kztab->kzembed));
	gtk_notebook_set_current_page(GTK_NOTEBOOK(kztab->kz->notebook),
				      page_num);
	return FALSE;
}

static void
kz_tab_label_drag_data_get (GtkWidget        *widget,
			    GdkDragContext   *context,
			    GtkSelectionData *data,
			    guint             info,
			    guint             time)
{
	KzTabLabel *kztab;
	const gchar *uri, *title;

	g_return_if_fail(KZ_TAB_LABEL(widget));
	kztab = KZ_TAB_LABEL(widget);

	g_return_if_fail (KZ_IS_MOZ_EMBED(kztab->kzembed));

	uri   = kz_moz_embed_get_location(kztab->kzembed);
	title = kz_moz_embed_get_title(kztab->kzembed);

	if (!uri || !*uri) return;

	switch (info)
	{
	case TARGET_KAZEHAKASE_TAB:
		gtk_selection_data_set(data, data->target,
				       8, "dummy", strlen("dummy"));
		break;
	case TARGET_NETSCAPE_URL:
	{
		gchar *str;
		if (title && *title)
		{
			gchar *title_locale;
			gsize bytes_read, bytes_written;
			GError *error = NULL;

			title_locale = g_locale_from_utf8(title, -1,
							  &bytes_read,
							  &bytes_written,
							  &error);
			if (error)
			{
				g_warning("kz_tab_label_drag_data_get(): %s",
					 error->message);
				g_error_free(error);
			}
			if (title_locale && *title_locale)
			{
				str = g_strconcat(uri, "\n", title_locale,
						  NULL);
				g_free(title_locale);
			}
			else
			{
				str = g_strdup (uri);
			}
		}
		else
		{
			str = g_strdup (uri);
		}
		gtk_selection_data_set(data, data->target,
				       8, str, strlen(str));
		g_free(str);
		break;
	}
	case TARGET_TEXT_URI_LIST:
	case TARGET_TEXT_PLAIN:
	case TARGET_STRING:
		gtk_selection_data_set(data, data->target,
				       8, uri, strlen(uri));
		break;
	default:
		break;
	}
}

static void
kz_tab_label_drag_data_received (GtkWidget *widget,
				 GdkDragContext *drag_context,
				 gint x, gint y,
				 GtkSelectionData *data,
				 guint info,
				 guint time)
{
	KzTabLabel *kztab;

	g_return_if_fail(KZ_TAB_LABEL(widget));
	kztab = KZ_TAB_LABEL(widget);

	g_return_if_fail (KZ_IS_MOZ_EMBED(kztab->kzembed));

	switch (info)
	{
	case TARGET_KAZEHAKASE_TAB:
	{
		GtkWidget *src_widget = gtk_drag_get_source_widget(drag_context);
		KzTabLabel *src_kztab;

		if (!KZ_IS_TAB_LABEL(src_widget)) return;
		src_kztab = KZ_TAB_LABEL(src_widget);

		kz_tab_label_move_page(src_kztab, kztab);
		return;
	}

	case TARGET_NETSCAPE_URL:
	case TARGET_TEXT_URI_LIST:
	case TARGET_TEXT_PLAIN:
	case TARGET_STRING:
	{
		gchar *string, **strings;

		if (data->length < 0) return;
		string = g_alloca (data->length + 1);

		memcpy (string, data->data, data->length);
		string[data->length] = '\0';

		strings = g_strsplit(data->data, "\n", 2);
		gtk_moz_embed_load_url(GTK_MOZ_EMBED(kztab->kzembed),
				       strings[0]);
		g_strfreev(strings);
		break;
	}

	default:
		break;
	}
}

/* move kztab to dest_kztab's position */
static void
kz_tab_label_move_page(KzTabLabel *kztab, KzTabLabel *dest_kztab)
{
	GtkWidget *src_page;
	GtkNotebook *notebook;
	gint dest_pos;

	g_return_if_fail(dest_kztab->kz && dest_kztab->kzembed);
	g_return_if_fail(kztab->kz && kztab->kzembed);

	notebook = GTK_NOTEBOOK(dest_kztab->kz->notebook);
	dest_pos = gtk_notebook_page_num(notebook,
					 GTK_WIDGET(dest_kztab->kzembed));

	if (dest_kztab->kz != kztab->kz)
	{
		src_page = GTK_WIDGET(kztab->kzembed);
		kz_window_move_tab(dest_kztab->kz, GTK_WIDGET(src_page));
	}
	else
	{
		src_page = GTK_WIDGET(kztab->kzembed);
	}

	gtk_notebook_reorder_child (notebook, src_page, dest_pos);
	kz_window_update_nav_buttons
		(kztab->kz, KZ_MOZ_EMBED(KZ_WINDOW_CURRENT_PAGE(kztab->kz)));
}

static void
cb_close_button_clicked (GtkWidget *button, KzTabLabel *kztab)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));
	kz_window_close_tab(kztab->kz, GTK_WIDGET(kztab->kzembed));
}

/* callbacks for mozilla */

static void
cb_title_changed (GtkMozEmbed *embed, KzTabLabel *kztab)
{
	gchar *title;

	g_return_if_fail(KZ_IS_MOZ_EMBED(embed));
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));

	title = kz_moz_embed_ensure_title(kztab->kzembed);
	kz_tab_label_set_text(kztab, title);
	g_free(title);
}

static void
cb_net_start (GtkMozEmbed *embed, KzTabLabel *kztab)
{
	gchar *title;

	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));
	kz_tab_label_set_state(kztab, KZ_TAB_LABEL_STATE_LOADING);

	title = kz_moz_embed_ensure_title(kztab->kzembed);
	kz_tab_label_set_text(kztab, title);
	g_free(title);
}

static void
cb_net_stop (GtkMozEmbed *embed, KzTabLabel *kztab)
{
	gchar *title;

	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));
	kz_tab_label_set_state(kztab, KZ_TAB_LABEL_STATE_NORMAL);

	title = kz_moz_embed_ensure_title(kztab->kzembed);
	kz_tab_label_set_text(kztab, title);
	g_free(title);
}

static void
cb_destroy (GtkObject *object, KzTabLabel *kztab)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));
	kztab->kzembed = NULL;
}



/*****************************************************************************
 *                                                                           *
 *                               Popup menu                                  *
 *                                                                           *
 *****************************************************************************/
static void
cb_reload (KzTabLabel *kztab, guint action, gpointer data)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));

	gtk_moz_embed_reload(GTK_MOZ_EMBED(kztab->kzembed),
			     GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
}

static void
cb_stop (KzTabLabel *kztab, guint action, gpointer data)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));

	gtk_moz_embed_stop_load(GTK_MOZ_EMBED(kztab->kzembed));
}

static void
cb_close (KzTabLabel *kztab, guint action, gpointer data)
{
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));

	kz_window_close_tab(kztab->kz, GTK_WIDGET(kztab->kzembed));
}

static void
cb_close_all (KzTabLabel *kztab, guint action, gpointer data)
{
	GtkNotebook *notebook;
	gint num, pos;
	gint i;

	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));
	notebook = GTK_NOTEBOOK(kztab->kz->notebook);

	num = gtkutil_notebook_get_n_pages(notebook);
	pos = gtk_notebook_page_num(notebook, GTK_WIDGET(kztab->kzembed));

	if (num < 2 || pos < 0) return;

	for (i = num - 1; i >= 0; i--)
	{
		GtkWidget *widget = KZ_WINDOW_NTH_PAGE(kztab->kz, i);

		if (i == pos) continue;
		if (i > pos && action == 0) continue;
		if (i < pos && action == 1) continue;

		kz_window_close_tab_idle(kztab->kz, widget);
	}
}

static void
cb_copy (KzTabLabel *kztab, guint action, gpointer data)
{
	const gchar *text = NULL;

	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));

	switch (action) {
	case 0:
		text = kz_moz_embed_get_title(KZ_MOZ_EMBED(kztab->kzembed));
		break;
	case 1:
		text = kz_moz_embed_get_location(KZ_MOZ_EMBED(kztab->kzembed));
		break;
	default:
		return;
		break;
	}

	gtkutil_copy_text(text);
}

static void
cb_popup_menu_hide ()
{
	gtk_main_quit();
}

static GtkItemFactoryEntry popup_menu[] =
{
	{N_("/Reloa_d"),      NULL, G_CALLBACK(cb_reload),  0, NULL, NULL},
	{N_("/_Stop"),        NULL, G_CALLBACK(cb_stop),    0, NULL, NULL},

	{N_("/---"),          NULL, NULL, 0, "<Separator>", NULL},

	{N_("/_Close"),                  NULL, G_CALLBACK(cb_close),     0, NULL, NULL},
	{N_("/Close all _left tab"),     NULL, G_CALLBACK(cb_close_all), 0, NULL, NULL},
	{N_("/Close all _right tab"),    NULL, G_CALLBACK(cb_close_all), 1, NULL, NULL},
	{N_("/Close with_out this tab"), NULL, G_CALLBACK(cb_close_all), 2, NULL, NULL},

	{N_("/---"),          NULL, NULL, 0, "<Separator>", NULL},

	{N_("/Copy _Title"),  NULL, G_CALLBACK(cb_copy),  0, NULL, NULL},
	{N_("/Copy _URL"),    NULL, G_CALLBACK(cb_copy),  1, NULL, NULL},
};

static void
set_popup_menu_sensitive (KzTabLabel *kztab, GtkWidget *menu)
{
   GtkWidget *item;
   GtkItemFactory *ifactory;
   gint pos, num;

   pos = gtk_notebook_page_num(GTK_NOTEBOOK(kztab->kz->notebook),
			       GTK_WIDGET(kztab->kzembed));
   num = gtkutil_notebook_get_n_pages(GTK_NOTEBOOK(kztab->kz->notebook));

   ifactory = gtk_item_factory_from_widget(menu);

   item = gtk_item_factory_get_item(ifactory, "/Stop");
   gtk_widget_set_sensitive(item, kz_moz_embed_is_loading(kztab->kzembed));
   item = gtk_item_factory_get_item(ifactory, "/Close all left tab");
   gtk_widget_set_sensitive(item, !(pos == 0));
   item = gtk_item_factory_get_item(ifactory, "/Close all right tab");
   gtk_widget_set_sensitive(item, !(pos == num -1));
   item = gtk_item_factory_get_item(ifactory, "/Close without this tab");
   gtk_widget_set_sensitive(item, num > 1);
}

void
kz_tab_label_popup_menu_modal (KzTabLabel *kztab, guint button, guint time)
{
	GtkItemFactory *factory = NULL;
	GtkWidget *widget;

	factory = gtk_item_factory_new (GTK_TYPE_MENU,
					"<PopupMenu>", NULL);
#ifdef ENABLE_NLS
	gtk_item_factory_set_translate_func (factory,
					     (GtkTranslateFunc) gettext,
					     NULL, NULL);
#endif /* ENABLE_NLS */
	gtk_item_factory_create_items (factory,
				       G_N_ELEMENTS(popup_menu),
				       popup_menu,
				       kztab);
	widget = gtk_item_factory_get_widget (factory, "<PopupMenu>");
	set_popup_menu_sensitive(kztab, widget);
	g_signal_connect(G_OBJECT(widget), "hide",
			 G_CALLBACK(cb_popup_menu_hide), kztab);
	gtk_menu_popup(GTK_MENU(widget), NULL, NULL,
		       NULL, NULL, button, time);
	gtk_main();
	gtk_widget_destroy(widget);
	g_object_unref(factory);
}
