/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003-2004 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-navigation-action.h"

#include <glib/gi18n.h>
#include "kazehakase.h"
#include "kz-favicon.h"
#include "kz-icons.h"

enum {
	PROP_0,
	PROP_KZ_WINDOW
};


static void dispose          (GObject *obj);
static void set_property     (GObject           *object,
                              guint              prop_id,
                              const GValue      *value,
                              GParamSpec        *pspec);
static void get_property     (GObject           *object,
                              guint              prop_id,
                              GValue            *value,
                              GParamSpec        *pspec);
static void connect_proxy    (GtkAction         *action,
                              GtkWidget         *proxy);
static void activate         (GtkAction         *action);

static void cb_menu_activated   (GtkMenuToolButton *b,
                                 KzNavigationAction *action);

G_DEFINE_TYPE(KzNavigationAction, kz_navigation_action, GTK_TYPE_ACTION)

static void
kz_navigation_action_class_init (KzNavigationActionClass *klass)
{
	GObjectClass *object_class;
	GtkActionClass *action_class;

	object_class = G_OBJECT_CLASS(klass);
	action_class = GTK_ACTION_CLASS(klass);

	object_class->set_property     = set_property;
	object_class->get_property     = get_property;
	object_class->dispose          = dispose;

	action_class->toolbar_item_type = GTK_TYPE_MENU_TOOL_BUTTON;
	action_class->connect_proxy     = connect_proxy;
	action_class->activate          = activate;

	g_object_class_install_property
		(object_class,
		 PROP_KZ_WINDOW,
		 g_param_spec_object ("kz-window",
				      _("KzWindow"),
				      _("The KzWindow to add a navigation button"),
				      KZ_TYPE_WINDOW,
				      G_PARAM_READWRITE |
				      G_PARAM_CONSTRUCT_ONLY));
}


static void
kz_navigation_action_init (KzNavigationAction *action)
{
	action->kz = NULL;
}


static void
dispose (GObject *obj)
{
	KzNavigationAction *action = KZ_NAVIGATION_ACTION(obj);

	if (action->kz)
		g_object_unref(action->kz);
	action->kz = NULL;

	if (G_OBJECT_CLASS(kz_navigation_action_parent_class)->dispose)
		G_OBJECT_CLASS(kz_navigation_action_parent_class)->dispose(obj);
}


static void
set_property (GObject         *object,
              guint            prop_id,
              const GValue    *value,
              GParamSpec      *pspec)
{
	KzNavigationAction *action = KZ_NAVIGATION_ACTION(object);
  
	switch (prop_id)
	{
	case PROP_KZ_WINDOW:
		action->kz = g_value_dup_object(value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


static void
get_property (GObject         *object,
              guint            prop_id,
              GValue          *value,
              GParamSpec      *pspec)
{
	KzNavigationAction *action = KZ_NAVIGATION_ACTION(object);

	switch (prop_id)
	{
	case PROP_KZ_WINDOW:
		g_value_set_object(value, action->kz);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void
connect_proxy (GtkAction *action, GtkWidget *proxy)
{
	GtkWidget *menu;

	g_signal_connect (proxy, "show-menu",
			  G_CALLBACK (cb_menu_activated), action);
	
	if (GTK_IS_MENU_TOOL_BUTTON(proxy))
	{
		menu = gtk_menu_tool_button_get_menu(GTK_MENU_TOOL_BUTTON(proxy));
		if (!menu)
		{
			menu = gtk_menu_new();
			gtk_menu_tool_button_set_menu(GTK_MENU_TOOL_BUTTON(proxy), menu);
		}
	}

	(* GTK_ACTION_CLASS (kz_navigation_action_parent_class)->connect_proxy) (action, proxy);
}


static void
activate (GtkAction *action)
{
	KzWeb *web;

	g_return_if_fail(KZ_NAVIGATION_ACTION(action));

	web = KZ_WINDOW_CURRENT_WEB(KZ_NAVIGATION_ACTION(action)->kz);

	if (!web) return;

	kz_web_go_back(web);
}

static void
cb_menu_item_activate (GtkWidget *menu, KzNavigationAction *action)
{
	gint go_nth;

	KzWeb *kzweb = KZ_WINDOW_CURRENT_WEB(action->kz);

	go_nth = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(menu), "go_nth"));

	kz_web_go_history_index(kzweb, go_nth);
}

static void
remove_item (GtkWidget *child, gpointer data)
{
	gtk_container_remove(GTK_CONTAINER(child->parent), child);
}

static void
cb_menu_activated (GtkMenuToolButton *b, KzNavigationAction *action)
{
	int i, pos, count;
	GtkWidget *menu;
	KzWeb *kzweb = KZ_WINDOW_CURRENT_WEB(action->kz);
	KzFavicon *kzfav = KZ_GET_FAVICON;

	menu = gtk_menu_tool_button_get_menu(b);
	if (menu)
	{
		gtk_container_foreach (GTK_CONTAINER(menu),
				       remove_item, NULL);
	}
	else
	{
		menu = gtk_menu_new();
	}

	kz_web_shistory_get_pos(kzweb, &pos, &count);

	for (i = 0; i < pos; i++)
	{
		gchar *title = NULL, *uri = NULL;
		GtkWidget *item;
		GtkWidget *favicon = NULL;

		kz_web_shistory_get_nth (kzweb, i, FALSE,
					   &uri, &title);
		item = gtk_image_menu_item_new_with_label(title);

		favicon	= kz_favicon_get_widget(kzfav, uri,
						KZ_ICON_SIZE_BOOKMARK_MENU);
		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(item),
					      favicon);

		gtk_menu_shell_prepend(GTK_MENU_SHELL(menu), item);
		g_object_set_data(G_OBJECT(item), "go_nth",
				  GINT_TO_POINTER(i));
		g_signal_connect(item, "activate",
                                 G_CALLBACK (cb_menu_item_activate), action);
		gtk_widget_show(item);
	
		if (title)
			g_free(title);
		if (uri)
			g_free(uri);
	}

	gtk_menu_tool_button_set_menu(b, menu);
}

