/*
 *  Copyright (C) 2006 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "rd-image.h"
#include "rendako.h"

#include <glib/gi18n.h>

typedef struct _RdImagePrivate RdImagePrivate;
struct _RdImagePrivate
{
	GtkWindow *popup;
	gint width, height;
	gint view_x, view_y;
#ifdef RD_THUMBNAIL_SCROLL
	gint scroll_diff;
#endif
	cairo_surface_t *cs;
	cairo_surface_t *orig_cs;
	gint cs_width, cs_height;
	gdouble ratio;
};

#define PADDING 0
#define RD_IMAGE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), RD_TYPE_IMAGE, RdImagePrivate))

static void     rd_image_class_init     (RdImageClass    *klass);
static void     rd_image_init           (RdImage         *image);
static void     rd_image_dispose        (GObject         *object);
static void     rd_image_finalize       (GObject         *object);

static void     rd_image_realize        (GtkWidget       *widget);
static void	rd_image_size_request   (GtkWidget       *widget,
					 GtkRequisition  *requisition);
static void	rd_image_size_allocate  (GtkWidget       *widget,
					 GtkAllocation   *allocation);
static gint	rd_image_expose         (GtkWidget       *widget,
					 GdkEventExpose  *event);
#ifdef RD_THUMBNAIL_SCROLL
static gboolean rd_image_scroll_event   (GtkWidget       *widget,
					 GdkEventScroll  *event);
#endif
static gboolean rd_image_button_release (GtkWidget      *widget,
					 GdkEventButton *event);

static void rd_image_create_scaled_cairo_surface (RdImage *image);

static GtkWidgetClass *parent_class = NULL;

G_DEFINE_TYPE (RdImage, rd_image, GTK_TYPE_WIDGET)

static void
rd_image_class_init (RdImageClass *klass)
{
	GObjectClass *gobject_class;
	GtkWidgetClass *widget_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *)   klass;
	widget_class  = (GtkWidgetClass *) klass;
  
	gobject_class->dispose  = rd_image_dispose;
	gobject_class->finalize = rd_image_finalize;

	widget_class->size_request         = rd_image_size_request;
	widget_class->size_allocate        = rd_image_size_allocate;
	widget_class->realize              = rd_image_realize;
	widget_class->expose_event         = rd_image_expose;
	widget_class->button_release_event = rd_image_button_release;
#ifdef RD_THUMBNAIL_SCROLL
	widget_class->scroll_event         = rd_image_scroll_event;
#endif

	g_type_class_add_private (gobject_class, sizeof(RdImagePrivate));
}

static void
rd_image_init (RdImage *image)
{
	RdImagePrivate *priv = RD_IMAGE_GET_PRIVATE (image);

	priv->width  = 0;
	priv->height = 0;
	priv->ratio  = 1.0;

	priv->view_x = 0;
	priv->view_y = 0;

#ifdef RD_THUMBNAIL_SCROLL
	priv->scroll_diff = 10;
#endif
	priv->orig_cs   = NULL;
	priv->cs        = NULL; 
	priv->cs_width  = 0;
	priv->cs_height = 0;
}

static void
rd_image_dispose (GObject *object)
{
	RdImagePrivate *priv = RD_IMAGE_GET_PRIVATE (object);
	RdImage *image = RD_IMAGE (object);

	rd_image_set_cairo_surface (image, NULL);

	if (priv->cs)
	{
		cairo_surface_destroy (priv->cs);
		priv->cs = NULL;
	}

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose (object);
}

static void
rd_image_finalize (GObject *object)
{
	if (G_OBJECT_CLASS (parent_class)->finalize)
		G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
rd_image_realize (GtkWidget *widget)
{
	GdkWindowAttr attributes;

	GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);


	attributes.window_type = GDK_WINDOW_CHILD;
	attributes.wclass = GDK_INPUT_OUTPUT;
	attributes.visual = gtk_widget_get_visual (widget);
	attributes.colormap = gtk_widget_get_colormap (widget);

	attributes.x = widget->allocation.x;
	attributes.y = widget->allocation.y;
	attributes.width = widget->allocation.width;
	attributes.height = widget->allocation.height;
	attributes.event_mask = GDK_EXPOSURE_MASK |
				GDK_BUTTON_PRESS_MASK |
				GDK_BUTTON_RELEASE_MASK |
				GDK_SCROLL_MASK |
				GDK_KEY_PRESS_MASK |
				GDK_POINTER_MOTION_MASK |
				GDK_POINTER_MOTION_HINT_MASK |
		                GDK_ENTER_NOTIFY_MASK |
		                GDK_LEAVE_NOTIFY_MASK;

	widget->window = gdk_window_new (gtk_widget_get_parent_window (widget),
					 &attributes,
					 GDK_WA_X | GDK_WA_Y |
					 GDK_WA_COLORMAP |
					 GDK_WA_VISUAL);
	gdk_window_set_user_data (widget->window, widget);
	widget->style = gtk_style_attach (widget->style, widget->window);

	gdk_window_set_background (widget->window, &widget->style->bg [GTK_STATE_NORMAL]);
}

static gint
rd_image_expose (GtkWidget *widget, GdkEventExpose *event)
{
	RdImagePrivate *priv = RD_IMAGE_GET_PRIVATE (widget);
	cairo_t *cr;
	gint width;

	if (!priv->cs)
		return TRUE;

	width = cairo_image_surface_get_width (priv->cs);
	
	if (width != priv->width)
		rd_image_create_scaled_cairo_surface (RD_IMAGE (widget));

	cr = gdk_cairo_create (GDK_DRAWABLE (widget->window));

	cairo_save (cr);
	gdk_cairo_rectangle (cr, &event->area);
	cairo_set_source_surface (cr, priv->cs, priv->view_x * priv->ratio + PADDING, -priv->view_y * priv->ratio + PADDING);
	cairo_fill (cr);
	cairo_restore (cr);

	cairo_destroy (cr);

	return TRUE;
}

static void
rd_image_size_request (GtkWidget *widget, GtkRequisition *requisition)
{
	RdImagePrivate *priv = RD_IMAGE_GET_PRIVATE (widget);

	requisition->width = priv->width + PADDING * 2;
	requisition->height = priv->height + PADDING * 2;
}

static void
rd_image_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
	rd_image_size_request (widget, &widget->requisition);

	if (GTK_WIDGET_CLASS (parent_class)->size_allocate)
		GTK_WIDGET_CLASS (parent_class)->size_allocate (widget, allocation);
}

static gboolean
cb_popup_window_button_release (GtkWidget *widget,
				GdkEventButton *event,
				gpointer data)
{
	if (event->button == 2)
	{
		gtk_widget_destroy (GTK_WIDGET (data));
		return TRUE;
	}
	return FALSE;
}

static void
popup_window (RdImage *image)
{
	GtkWidget *popup, *scroll, *full_image, *event_box;
	GdkPixmap *pixmap;
	cairo_t *cr;

	RdImagePrivate *priv = RD_IMAGE_GET_PRIVATE (image);

	popup = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_transient_for (GTK_WINDOW (popup), 
				      GTK_WINDOW (gtk_widget_get_toplevel (GTK_WIDGET (image))));
	gtk_window_set_destroy_with_parent (GTK_WINDOW (popup), TRUE);
	gtk_window_set_default_size (GTK_WINDOW(popup), RD_IMAGE_WIDTH, 600);

	scroll = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show (scroll);
	gtk_container_add (GTK_CONTAINER (popup), scroll);

	event_box = gtk_event_box_new ();
	gtk_widget_show (event_box);

	pixmap = gdk_pixmap_new (NULL, priv->cs_width, priv->cs_height, 24);
	cr = gdk_cairo_create (GDK_DRAWABLE (pixmap));
	cairo_set_source_surface (cr, priv->orig_cs, 0, 0);
	cairo_paint (cr);
	cairo_destroy (cr);
	full_image = gtk_image_new_from_pixmap (pixmap, NULL);
	gtk_widget_show (full_image);
	g_object_unref (pixmap);
	gtk_container_add (GTK_CONTAINER (event_box), full_image);

	gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroll), event_box);

	g_signal_connect (event_box,
			  "button_release_event",
			  G_CALLBACK (cb_popup_window_button_release), popup);

	gtk_widget_show (popup);
}

static gboolean
rd_image_button_release (GtkWidget *widget, GdkEventButton *event)
{
	if (event->button == 2)
	{
		popup_window (RD_IMAGE (widget));
		return TRUE;
	}

	return FALSE;
}

#ifdef RD_THUMBNAIL_SCROLL
static gboolean
rd_image_scroll_event (GtkWidget *widget, GdkEventScroll *event)
{
	RdImagePrivate *priv = RD_IMAGE_GET_PRIVATE (widget);

	switch (event->direction)
	{
	case GDK_SCROLL_UP:
		if (priv->view_y >= priv->scroll_diff)
			priv->view_y -= priv->scroll_diff;
		gtk_widget_queue_resize (widget);
		return TRUE;
		break;
	case GDK_SCROLL_DOWN:
		if (priv->view_y <= priv->cs_height - priv->height / priv->ratio - priv->scroll_diff)
			priv->view_y += priv->scroll_diff;
		gtk_widget_queue_resize (widget);
		return TRUE;
		break;
	default:
		break;
	}

	return FALSE;
}
#endif

GtkWidget *
rd_image_new (void)
{
	RdImage *image;

	image = g_object_new (RD_TYPE_IMAGE, 
			      NULL);

	return GTK_WIDGET (image);
}

static void
rd_image_create_scaled_cairo_surface (RdImage *image)
{
	cairo_t *cr;
	RdImagePrivate *priv = RD_IMAGE_GET_PRIVATE (image);

	priv->ratio = (gdouble) priv->width / priv->cs_width;

	if (priv->cs)
		cairo_surface_destroy (priv->cs);

	priv->cs = cairo_surface_create_similar (priv->orig_cs,
			CAIRO_CONTENT_COLOR_ALPHA,
			priv->width, priv->height);
	cr = cairo_create (priv->cs);

	cairo_save (cr);
	cairo_rectangle (cr, 0, 0, priv->width, priv->height);
	cairo_scale (cr, priv->ratio, priv->ratio);
	cairo_clip (cr);
	cairo_set_source_surface (cr, priv->orig_cs, 0, 0);
	cairo_set_operator (cr, CAIRO_OPERATOR_SOURCE);
	cairo_paint (cr);
	cairo_restore (cr);

	cairo_destroy (cr);
}

void
rd_image_set_cairo_surface (RdImage *image, cairo_surface_t *cs)
{
	RdImagePrivate *priv = RD_IMAGE_GET_PRIVATE (image);

	if (priv->orig_cs)
	{
		cairo_surface_destroy (priv->orig_cs);
	}

	priv->orig_cs = cs;

	if (priv->orig_cs)
	{
		priv->cs_width = cairo_image_surface_get_width (priv->orig_cs);
		priv->cs_height = cairo_image_surface_get_height (priv->orig_cs);

		cairo_surface_reference (priv->orig_cs);

		rd_image_create_scaled_cairo_surface (image);
	}
}

void
rd_image_set_size (RdImage *image, gint width, gint height)
{
	RdImagePrivate *priv = RD_IMAGE_GET_PRIVATE (image);

	if (priv->width == width && priv->height == height)
		return;

	priv->width = width;
	priv->height = height;
}

