package jp.kirikiri.tvp2.base;

import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;

public class ByteBufferInputStream extends InputStream {

	private ByteBuffer mData;

	public ByteBufferInputStream( ByteBuffer src ) {
		mData = src;
	}
	/**
	 *  この入力ストリームのメソッドの次の呼び出しによって、ブロックせずにこの入力ストリームから読み込むことができる
	 *   (またはスキップできる) 推定バイト数を返します。
	 */
	@Override
	public int available() {
		return mData.limit() - mData.position();
	}

	/**
	 * この入力ストリームを閉じて、そのストリームに関連するすべてのシステムリソースを解放します。
	 */
	@Override
	public void close() {
	}

	/**
	 *   この入力ストリームの現在位置にマークを設定します。
	 */
	@Override
	public void mark(int readlimit) {
		mData.mark();
	}
	/**
	 * 入力ストリームが mark と reset メソッドをサポートしているかどうかを判定します。
	 */
	@Override
	public boolean markSupported() { return true; }

	/**
	 * 入力ストリームからデータの次のバイトを読み込みます。
	 */
	@Override
	public int read() throws IOException {
		return mData.get();
	}
	/**
	 * 入力ストリームから数バイトを読み込み、それをバッファー配列 b に格納します。
	 */
	@Override
	public int read(byte[] b) {
		int len = b.length;
		if( b.length > available() ) {
			len = available();
		}
		mData.get( b, 0, len );
		return len;
	}
	/**
	 * 最大 len バイトまでのデータを、入力ストリームからバイト配列に読み込みます。
	 */
	@Override
	public int read(byte[] b, int off, int len ) {
		if( (len + off) > available() ) {
			len = available() - off;
		}
		if( len >  0 ) {
			mData.get( b, off, len );
		} else {
			return -1;
		}
		return len;
	}
	/**
	 * このストリームの位置を、入力ストリームで最後に mark メソッドが呼び出されたときのマーク位置に再設定します。
	 */
	@Override
	public void reset() { mData.reset(); }

	/**
	 *  この入力ストリームから n バイト分をスキップおよび破棄します。
	 */
	@Override
	public long skip(long n) {
		if( n > (mData.limit() - mData.position()) ) {
			n = mData.limit() - mData.position();
		}
		mData.position( (int) (mData.position() + n) );
		return n;
	}
}
