package com.tryandroid.media;

import java.io.File;
import java.text.SimpleDateFormat;
import java.util.Date;

import com.kurukurupapa.tryandroid.fw.util.ActivityUtil;
import com.kurukurupapa.tryandroid.fw.util.LogUtil;
import com.tryandroid.R;

import android.app.Activity;
import android.media.MediaRecorder;
import android.os.Bundle;
import android.os.Environment;
import android.view.SurfaceHolder;
import android.view.SurfaceView;
import android.view.View;
import android.widget.Toast;

/**
 * 動画の録画機能を持ったアクティビティ
 */
public class VideoRecActivity extends Activity {

	private SurfaceView surfaceView;
	private MediaRecorder recorder;
	private String outputPath;

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);

		// レイアウトを設定
		setContentView(R.layout.video_rec_activity);

		// プレビュー準備
		surfaceView = (SurfaceView) findViewById(R.id.surface_view);
		SurfaceHolder surfaceHolder = surfaceView.getHolder();
		surfaceHolder.setType(SurfaceHolder.SURFACE_TYPE_PUSH_BUFFERS);
	}

	/**
	 * 録画開始ボタン押下時処理
	 *
	 * @param view
	 */
	public void onStartButtonClick(View view) {
		// 出力先ファイル名作成
		outputPath = getOutputPath();

		// MediaRecorder準備
		if (!setupRecorder(outputPath)) {
			return;
		}

		// 録画開始
		recorder.start();
	}

	/**
	 * 録画終了ボタン押下時処理
	 *
	 * @param view
	 */
	public void onStopButtonClick(View view) {
		// 録画終了
		recorder.stop();

		// レコーダー開放
		recorder.release();
		recorder = null;

		// 完了メッセージ
		ActivityUtil.showDialog(this, getTitle(), "ファイルを保存しました。\r\n"
				+ outputPath);
	}

	/**
	 * 出力先ファイル名作成
	 *
	 * @return
	 */
	private String getOutputPath() {
		SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd-HHmmss");

		File storageDir = Environment.getExternalStorageDirectory();
		File videoDir = new File(storageDir, "video");
		if (!videoDir.exists()) {
			videoDir.mkdir();
		}

		return new File(videoDir, "VideoRec-"
				+ format.format(new Date(System.currentTimeMillis())) + ".mp4")
				.getAbsolutePath();
	}

	/**
	 * MediaRecorder準備<br>
	 *
	 * マニフェストファイルに次のパーミッションが必要<br>
	 * - android.permission.CAMERA<br>
	 * - android.permission.RECORD_AUDIO<br>
	 * - android.permission.WRITE_EXTERNAL_STORAGE<br>
	 *
	 * 保存形式によって、動画や音声の保存可否がある。詳細は次のページを参照。<br>
	 * - http://developer.android.com/guide/appendix/media-formats.html
	 *
	 * @return 成功した場合true
	 */
	private boolean setupRecorder(String path) {
		boolean result = false;

		// 初期化
		recorder = new MediaRecorder();
		recorder.setVideoSource(MediaRecorder.VideoSource.CAMERA);
		recorder.setAudioSource(MediaRecorder.AudioSource.MIC);

		// 出力ファイル設定
		// recorder.setOutputFormat(MediaRecorder.OutputFormat.THREE_GPP);
		recorder.setOutputFormat(MediaRecorder.OutputFormat.MPEG_4);
		recorder.setOutputFile(path);

		// 画像設定
		// VGA（640×480ピクセル）
		// QVGA（320×240ピクセル）
		recorder.setVideoFrameRate(15);
		recorder.setVideoSize(320, 240);
		recorder.setVideoEncoder(MediaRecorder.VideoEncoder.MPEG_4_SP);

		// 音声設定
		recorder.setAudioEncoder(MediaRecorder.AudioEncoder.AMR_NB);

		// プレビュー設定
		recorder.setPreviewDisplay(surfaceView.getHolder().getSurface());
		try {
			recorder.prepare();
			result = true;
		} catch (Exception e) {
			LogUtil.e("録画準備失敗", e);
			Toast.makeText(this, "録画準備に失敗しました。", Toast.LENGTH_LONG).show();
		}

		return result;
	}
}
