package com.kurukurupapa.tryandroid.fw;

import com.kurukurupapa.tryandroid.fw.model.FwPreference;
import com.kurukurupapa.tryandroid.fw.util.ActivityUtil;
import com.kurukurupapa.tryandroid.fw.util.LogUtil;
import com.kurukurupapa.tryandroid.fw.util.RUtil;

import android.os.Bundle;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;

/**
 * メイン画面の基本クラス
 *
 * <p>
 * メニュー用XML（/res/menu/option_menu.xml）が定義されていたら、読み込んでメニューを作成する。
 * メニュー選択時の処理は、子クラスにて、onOptionsItemSelectedをオーバーライドし実装する。<br />
 *
 * メニュー用XMLの例
 *
 * <pre>
 * &lt;?xml version="1.0" encoding="utf-8"?&gt;
 * &lt;menu xmlns:android="http://schemas.android.com/apk/res/android"&gt;
 *   &lt;item android:id="@+id/menu_help" android:title="ヘルプ"
 *     android:icon="@android:drawable/ic_menu_help" /&gt;
 * &lt;/menu&gt;
 * </pre>
 *
 * TODO 子クラスによるonOptionsItemSelectedメソッド定義の例
 *
 * <pre>
 * </pre>
 *
 * </p>
 */
public class BaseMainActivity extends BaseActivity {

	private FwPreference preferenceDto;

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		LogUtil.called();

		preferenceDto = new FwPreference(this);
	}

	@Override
	protected void onResume() {
		super.onResume();
		LogUtil.called();

		// インストール後 or バージョンアップ後にメッセージを表示する
		boolean saveFlag = false;
		preferenceDto.load();
		if (preferenceDto.isFirstInstall()) {
			// インストール直後
			onResumeInFirstInstall();
			saveFlag = true;
		} else if (preferenceDto.isVersionUp()) {
			// バージョンアップ直後
			onResumeInVersionUp();
			saveFlag = true;
		}

		// 設定ファイルにバージョンコードを保存する
		if (saveFlag) {
			preferenceDto.setVersionCode();
			preferenceDto.save();
		}
	}

	/**
	 * アプリインストール直後におけるonResume()イベント
	 */
	protected void onResumeInFirstInstall() {
		LogUtil.called();

		// ダイアログ表示
		String msg = getFirstInstallDialogMsg();
		if (msg != null) {
			ActivityUtil.showDialog(this, getTitle().toString(), msg);
		}
	}

	/**
	 * バージョンアップ直後におけるonResume()イベント
	 */
	protected void onResumeInVersionUp() {
		LogUtil.called();

		// ダイアログ表示
		String msg = getVersionUpDialogMsg();
		if (msg != null) {
			ActivityUtil.showDialog(this, getTitle().toString(), msg);
		}
	}

	/**
	 * インストール直後初回表示で表示するダイアログのメッセージ<br>
	 * 未設定の場合、ダイアログ表示なし。
	 *
	 * @return メッセージ
	 */
	protected String getFirstInstallDialogMsg() {
		return null;
	}

	/**
	 * バージョンアップ直後初回表示で表示するダイアログのメッセージ<br>
	 * 未設定の場合、ダイアログ表示なし。
	 *
	 * @return メッセージ
	 */
	protected String getVersionUpDialogMsg() {
		return null;
	}

	@Override
	public boolean onCreateOptionsMenu(Menu menu) {
		super.onCreateOptionsMenu(menu);
		LogUtil.called();

		// メニュー用XML（/res/menu/option_menu.xml）が定義されていたら、
		// 読み込んでメニューを作成する。
		Integer id = RUtil.getResourceIdOrNull(this, "menu", "option_menu");
		if (id != null) {
			MenuInflater inflater = getMenuInflater();
			inflater.inflate(id, menu);
		}
		return true;
	}

	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		boolean result = super.onOptionsItemSelected(item);
		LogUtil.called();
		return result;
	}
}
