/*
 * Copyright (C) 2004-2013 L2J DataPack
 * 
 * This file is part of L2J DataPack.
 * 
 * L2J DataPack is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * L2J DataPack is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package handlers.admincommandhandlers;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.l2jserver.Config;
import com.l2jserver.L2DatabaseFactory;
import com.l2jserver.gameserver.datatables.ClassListData;
import com.l2jserver.gameserver.datatables.SkillTable;
import com.l2jserver.gameserver.datatables.SkillTreesData;
import com.l2jserver.gameserver.handler.IAdminCommandHandler;
import com.l2jserver.gameserver.model.L2Clan;
import com.l2jserver.gameserver.model.L2Object;
import com.l2jserver.gameserver.model.L2SkillLearn;
import com.l2jserver.gameserver.model.actor.instance.L2PcInstance;
import com.l2jserver.gameserver.model.skills.L2Skill;
import com.l2jserver.gameserver.network.SystemMessageId;
import com.l2jserver.gameserver.network.serverpackets.NpcHtmlMessage;
import com.l2jserver.gameserver.network.serverpackets.PledgeSkillList;
import com.l2jserver.gameserver.network.serverpackets.SystemMessage;
import com.l2jserver.util.StringUtil;

/**
 * This class handles following admin commands:
 * <ul>
 * <li>show_skills</li>
 * <li>remove_skills</li>
 * <li>skill_list</li>
 * <li>skill_index</li>
 * <li>add_skill</li>
 * <li>remove_skill</li>
 * <li>get_skills</li>
 * <li>reset_skills</li>
 * <li>give_all_skills</li>
 * <li>give_all_skills_fs</li>
 * <li>admin_give_all_clan_skills</li>
 * <li>remove_all_skills</li>
 * <li>add_clan_skills</li>
 * <li>admin_setskill</li>
 * </ul>
 * @version 2012/02/26 Small fixes by Zoey76 05/03/2011
 */
public class AdminSkill implements IAdminCommandHandler
{
	private static Logger _log = Logger.getLogger(AdminSkill.class.getName());
	static final boolean L2J_JP = false;
	
	private static final String[] ADMIN_COMMANDS =
	{
		"admin_show_skills",
		"admin_remove_skills",
		"admin_skill_list",
		"admin_skill_index",
		"admin_add_skill",
		"admin_remove_skill",
		"admin_get_skills",
		"admin_reset_skills",
		"admin_give_all_skills",
		"admin_give_all_skills_fs",
		"admin_give_all_clan_skills",
		"admin_remove_all_skills",
		"admin_add_clan_skill",
		"admin_setskill"
	};
	
	private static L2Skill[] adminSkills;
	
	@Override
	public boolean useAdminCommand(String command, L2PcInstance activeChar)
	{
		if (command.equals("admin_show_skills"))
		{
			showMainPage(activeChar);
		}
		else if (command.startsWith("admin_remove_skills"))
		{
			try
			{
				String val = command.substring(20);
				removeSkillsPage(activeChar, Integer.parseInt(val));
			}
			catch (StringIndexOutOfBoundsException e)
			{
			}
		}
		else if (command.startsWith("admin_skill_list"))
		{
if (L2J_JP) {{
			// [L2J_JP EDIT - TSL]
        	try {
        		String val = command.substring(17);
    			showSkillListPage(activeChar, val);
        	}
			catch (StringIndexOutOfBoundsException e){}
}} else {{
			AdminHtml.showAdminHtml(activeChar, "skills.htm");
}}
		}
		else if (command.startsWith("admin_skill_index"))
		{
			try
			{
if (L2J_JP) {{
				// [L2J_JP EDIT - TSL]
				String val = "";
				if (command.length() > 17)
					val = command.substring(18);
				showSkillIndexPage(activeChar, val);
}} else {{
				if (command.length() > 17)
				{
					String val = command.substring(18);
					AdminHtml.showAdminHtml(activeChar, "skills/" + val + ".htm");
				}
				else
				{
					AdminHtml.showAdminHtml(activeChar, "skills.htm");	//<<== charskills.htm
				}
}}
			}
			catch (StringIndexOutOfBoundsException e)
			{
			}
		}
		else if (command.startsWith("admin_add_skill"))
		{
			try
			{
				String val = command.substring(15);
				adminAddSkill(activeChar, val);
			}
			catch (Exception e)
			{
				activeChar.sendMessage("Usage: //add_skill <skill_id> <level>");
			}
		}
		else if (command.startsWith("admin_remove_skill"))
		{
			try
			{
				String id = command.substring(19);
				int idval = Integer.parseInt(id);
				adminRemoveSkill(activeChar, idval);
			}
			catch (Exception e)
			{
				activeChar.sendMessage("Usage: //remove_skill <skill_id>");
			}
		}
		else if (command.equals("admin_get_skills"))
		{
			adminGetSkills(activeChar);
		}
		else if (command.equals("admin_reset_skills"))
		{
			adminResetSkills(activeChar);
		}
		else if (command.equals("admin_give_all_skills"))
		{
			adminGiveAllSkills(activeChar, false);
		}
		else if (command.equals("admin_give_all_skills_fs"))
		{
			adminGiveAllSkills(activeChar, true);
		}
		else if (command.equals("admin_give_all_clan_skills"))
		{
			adminGiveAllClanSkills(activeChar);
		}
		else if (command.equals("admin_remove_all_skills"))
		{
			final L2Object target = activeChar.getTarget();
			if ((target == null) || !target.isPlayer())
			{
				activeChar.sendPacket(SystemMessageId.INCORRECT_TARGET);
				return false;
			}
			final L2PcInstance player = target.getActingPlayer();
			for (L2Skill skill : player.getAllSkills())
			{
				player.removeSkill(skill);
			}
			activeChar.sendMessage("You have removed all skills from " + player.getName() + ".");
			player.sendMessage("Admin removed all skills from you.");
			player.sendSkillList();
			player.broadcastUserInfo();
		}
		else if (command.startsWith("admin_add_clan_skill"))
		{
			try
			{
				String[] val = command.split(" ");
				adminAddClanSkill(activeChar, Integer.parseInt(val[1]), Integer.parseInt(val[2]));
			}
			catch (Exception e)
			{
				activeChar.sendMessage("Usage: //add_clan_skill <skill_id> <level>");
			}
		}
		else if (command.startsWith("admin_setskill"))
		{
			String[] split = command.split(" ");
			int id = Integer.parseInt(split[1]);
			int lvl = Integer.parseInt(split[2]);
			L2Skill skill = SkillTable.getInstance().getInfo(id, lvl);
			activeChar.addSkill(skill);
			activeChar.sendSkillList();
			activeChar.sendMessage("You added yourself skill " + skill.getName() + "(" + id + ") level " + lvl);
		}
		return true;
	}
	
	/**
	 * This function will give all the skills that the target can learn at his/her level
	 * @param activeChar the active char
	 * @param includedByFs if {@code true} Forgotten Scroll skills will be delivered.
	 */
	private void adminGiveAllSkills(L2PcInstance activeChar, boolean includedByFs)
	{
		final L2Object target = activeChar.getTarget();
		if ((target == null) || !target.isPlayer())
		{
			activeChar.sendPacket(SystemMessageId.INCORRECT_TARGET);
			return;
		}
		final L2PcInstance player = target.getActingPlayer();
		// Notify player and admin
		activeChar.sendMessage("You gave " + player.giveAvailableSkills(includedByFs, true) + " skills to " + player.getName());
		player.sendSkillList();
	}
	
	/**
	 * This function will give all the skills that the target's clan can learn at it's level.<br>
	 * If the target is not the clan leader, a system message will be sent to the Game Master.
	 * @param activeChar the active char, probably a Game Master.
	 */
	private void adminGiveAllClanSkills(L2PcInstance activeChar)
	{
		final L2Object target = activeChar.getTarget();
		if ((target == null) || !target.isPlayer())
		{
			activeChar.sendPacket(SystemMessageId.INCORRECT_TARGET);
			return;
		}
		
		final L2PcInstance player = target.getActingPlayer();
		final L2Clan clan = player.getClan();
		if (clan == null)
		{
			activeChar.sendPacket(SystemMessageId.TARGET_MUST_BE_IN_CLAN);
			return;
		}
		
		if (!player.isClanLeader())
		{
			final SystemMessage sm = SystemMessage.getSystemMessage(SystemMessageId.TARGET_MUST_BE_IN_CLAN);
			sm.addPcName(player);
			activeChar.sendPacket(sm);
		}
		
		final List<L2SkillLearn> skills = SkillTreesData.getInstance().getAvailablePledgeSkills(clan);
		SkillTable st = SkillTable.getInstance();
		for (L2SkillLearn s : skills)
		{
			clan.addNewSkill(st.getInfo(s.getSkillId(), s.getSkillLevel()));
		}
		
		// Notify target and active char
		clan.broadcastToOnlineMembers(new PledgeSkillList(clan));
		for (L2PcInstance member : clan.getOnlineMembers(0))
		{
			member.sendSkillList();
		}
		
		activeChar.sendMessage("You gave " + skills.size() + " skills to " + player.getName() + "'s clan " + clan.getName() + ".");
		player.sendMessage("Your clan received " + skills.size() + " skills.");
	}
	
	/**
	 * TODO: Externalize HTML
	 * @param activeChar the active Game Master.
	 * @param page
	 */
	private void removeSkillsPage(L2PcInstance activeChar, int page)
	{
		final L2Object target = activeChar.getTarget();
		if ((target == null) || !target.isPlayer())
		{
			activeChar.sendPacket(SystemMessageId.TARGET_IS_INCORRECT);
			return;
		}
		
		final L2PcInstance player = target.getActingPlayer();
		final L2Skill[] skills = player.getAllSkills().toArray(new L2Skill[player.getAllSkills().size()]);
		
		int maxSkillsPerPage = 10;
		int maxPages = skills.length / maxSkillsPerPage;
		if (skills.length > (maxSkillsPerPage * maxPages))
		{
			maxPages++;
		}
		
		if (page > maxPages)
		{
			page = maxPages;
		}
		
		int skillsStart = maxSkillsPerPage * page;
		int skillsEnd = skills.length;
		if ((skillsEnd - skillsStart) > maxSkillsPerPage)
		{
			skillsEnd = skillsStart + maxSkillsPerPage;
		}
		
		final NpcHtmlMessage adminReply = new NpcHtmlMessage();
		final StringBuilder replyMSG = StringUtil.startAppend(500 + (maxPages * 50) + (((skillsEnd - skillsStart) + 1) * 50), "<html><body>" + "<table width=260><tr>" + "<td width=40><button value=\"Main\" action=\"bypass -h admin_admin\" width=40 height=15 back=\"L2UI_ct1.button_df\" fore=\"L2UI_ct1.button_df\"></td>" + "<td width=180><center>Character Selection Menu</center></td>" + "<td width=40><button value=\"Back\" action=\"bypass -h admin_show_skills\" width=40 height=15 back=\"L2UI_ct1.button_df\" fore=\"L2UI_ct1.button_df\"></td>" + "</tr></table>" + "<br><br>" + "<center>Editing <font color=\"LEVEL\">", player.getName(), "</font></center>" + "<br><table width=270><tr><td>Lv: ", String.valueOf(player.getLevel()), " ", ClassListData.getInstance().getClass(player.getClassId()).getClientCode(), "</td></tr></table>" + "<br><table width=270><tr><td>Note: Dont forget that modifying players skills can</td></tr>" + "<tr><td>ruin the game...</td></tr></table>" + "<br><center>Click on the skill you wish to remove:</center>" + "<br>" + "<center><table width=270><tr>");
		
		for (int x = 0; x < maxPages; x++)
		{
			int pagenr = x + 1;
			if (x == page)
			{
				replyMSG.append("<td>").append("Page ").append(pagenr).append("</td>");
			}
			else
			{
				StringUtil.append(replyMSG,
					"<td><a action=\"bypass -h admin_remove_skills ",
					String.valueOf(x),
					"\">Page ",
					String.valueOf(pagenr),
					"</a></td>"
				);
			}
		}
		
		replyMSG.append("</tr></table></center>" + "<br><table width=270>" + "<tr><td width=80>Name:</td><td width=60>Level:</td><td width=40>Id:</td></tr>");
		
		for (int i = skillsStart; i < skillsEnd; i++)
		{
			StringUtil.append(replyMSG, "<tr><td width=80><a action=\"bypass -h admin_remove_skill ", String.valueOf(skills[i].getId()), "\">", skills[i].getName(), "</a></td><td width=60>", String.valueOf(skills[i].getLevel()), "</td><td width=40>", String.valueOf(skills[i].getId()), "</td></tr>");
		}
		
		replyMSG.append("</table>" + "<br><center><table>" + "Remove skill by ID :" + "<tr><td>Id: </td>" + "<td><edit var=\"id_to_remove\" width=110></td></tr>" + "</table></center>" + "<center><button value=\"Remove skill\" action=\"bypass -h admin_remove_skill $id_to_remove\" width=110 height=15 back=\"L2UI_ct1.button_df\" fore=\"L2UI_ct1.button_df\"></center>" + "<br><center><button value=\"Back\" action=\"bypass -h admin_current_player\" width=40 height=15 back=\"L2UI_ct1.button_df\" fore=\"L2UI_ct1.button_df\"></center>" + "</body></html>");
		adminReply.setHtml(replyMSG.toString());
		activeChar.sendPacket(adminReply);
	}
	
	/**
	 * @param activeChar the active Game Master.
	 */
	private void showMainPage(L2PcInstance activeChar)
	{
		final L2Object target = activeChar.getTarget();
		if ((target == null) || !target.isPlayer())
		{
			activeChar.sendPacket(SystemMessageId.INCORRECT_TARGET);
			return;
		}
		final L2PcInstance player = target.getActingPlayer();
		final NpcHtmlMessage adminReply = new NpcHtmlMessage();
		adminReply.setFile(activeChar.getHtmlPrefix(), "data/html/admin/charskills.htm");
		adminReply.replace("%name%", player.getName());
		adminReply.replace("%level%", player.getLevel());
		adminReply.replace("%class%", ClassListData.getInstance().getClass(player.getClassId()).getClientCode());
		activeChar.sendPacket(adminReply);
	}
	
	/**
	 * @param activeChar the active Game Master.
	 */
	private void adminGetSkills(L2PcInstance activeChar)
	{
		final L2Object target = activeChar.getTarget();
		if ((target == null) || !target.isPlayer())
		{
			activeChar.sendPacket(SystemMessageId.INCORRECT_TARGET);
			return;
		}
		final L2PcInstance player = target.getActingPlayer();
		if (player.getName().equals(activeChar.getName()))
		{
			player.sendPacket(SystemMessageId.CANNOT_USE_ON_YOURSELF);
		}
		else
		{
			L2Skill[] skills = player.getAllSkills().toArray(new L2Skill[player.getAllSkills().size()]);
			adminSkills = activeChar.getAllSkills().toArray(new L2Skill[activeChar.getAllSkills().size()]);
			for (L2Skill skill : adminSkills)
			{
				activeChar.removeSkill(skill);
			}
			for (L2Skill skill : skills)
			{
				activeChar.addSkill(skill, true);
			}
			activeChar.sendMessage("You now have all the skills of " + player.getName() + ".");
			activeChar.sendSkillList();
		}
		showMainPage(activeChar);
	}
	
	/**
	 * @param activeChar the active Game Master.
	 */
	private void adminResetSkills(L2PcInstance activeChar)
	{
		final L2Object target = activeChar.getTarget();
		if ((target == null) || !target.isPlayer())
		{
			activeChar.sendPacket(SystemMessageId.INCORRECT_TARGET);
			return;
		}
		final L2PcInstance player = target.getActingPlayer();
		if (adminSkills == null)
		{
			activeChar.sendMessage("You must get the skills of someone in order to do this.");
		}
		else
		{
			L2Skill[] skills = player.getAllSkills().toArray(new L2Skill[player.getAllSkills().size()]);
			for (L2Skill skill : skills)
			{
				player.removeSkill(skill);
			}
			for (L2Skill skill : activeChar.getAllSkills())
			{
				player.addSkill(skill, true);
			}
			for (L2Skill skill : skills)
			{
				activeChar.removeSkill(skill);
			}
			for (L2Skill skill : adminSkills)
			{
				activeChar.addSkill(skill, true);
			}
			player.sendMessage("[GM]" + activeChar.getName() + " updated your skills.");
			activeChar.sendMessage("You now have all your skills back.");
			adminSkills = null;
			activeChar.sendSkillList();
			player.sendSkillList();
		}
		showMainPage(activeChar);
	}
	
	/**
	 * @param activeChar the active Game Master.
	 * @param val
	 */
	private void adminAddSkill(L2PcInstance activeChar, String val)
	{
		final L2Object target = activeChar.getTarget();
		if ((target == null) || !target.isPlayer())
		{
			activeChar.sendPacket(SystemMessageId.INCORRECT_TARGET);
			showMainPage(activeChar);
			return;
		}
		final L2PcInstance player = target.getActingPlayer();
		final StringTokenizer st = new StringTokenizer(val);
		if (st.countTokens() != 2)
		{
			showMainPage(activeChar);
		}
		else
		{
			L2Skill skill = null;
			try
			{
				String id = st.nextToken();
				String level = st.nextToken();
				int idval = Integer.parseInt(id);
				int levelval = Integer.parseInt(level);
				skill = SkillTable.getInstance().getInfo(idval, levelval);
			}
			catch (Exception e)
			{
				_log.log(Level.WARNING, "", e);
			}
			if (skill != null)
			{
				String name = skill.getName();
				// Player's info.
				player.sendMessage("Admin gave you the skill " + name + ".");
				player.addSkill(skill, true);
				player.sendSkillList();
				// Admin info.
				activeChar.sendMessage("You gave the skill " + name + " to " + player.getName() + ".");
				if (Config.DEBUG)
				{
					_log.fine("[GM]" + activeChar.getName() + " gave skill " + name + " to " + player.getName() + ".");
				}
				activeChar.sendSkillList();
			}
			else
			{
				activeChar.sendMessage("Error: there is no such skill.");
			}
			showMainPage(activeChar); // Back to start
		}
	}
	
	/**
	 * @param activeChar the active Game Master.
	 * @param idval
	 */
	private void adminRemoveSkill(L2PcInstance activeChar, int idval)
	{
		final L2Object target = activeChar.getTarget();
		if ((target == null) || !target.isPlayer())
		{
			activeChar.sendPacket(SystemMessageId.INCORRECT_TARGET);
			return;
		}
		final L2PcInstance player = target.getActingPlayer();
		L2Skill skill = SkillTable.getInstance().getInfo(idval, player.getSkillLevel(idval));
		if (skill != null)
		{
			String skillname = skill.getName();
			player.sendMessage("Admin removed the skill " + skillname + " from your skills list.");
			player.removeSkill(skill);
			// Admin information
			activeChar.sendMessage("You removed the skill " + skillname + " from " + player.getName() + ".");
			if (Config.DEBUG)
			{
				_log.fine("[GM]" + activeChar.getName() + " removed skill " + skillname + " from " + player.getName() + ".");
			}
			activeChar.sendSkillList();
		}
		else
		{
			activeChar.sendMessage("Error: there is no such skill.");
		}
		removeSkillsPage(activeChar, 0); // Back to previous page
	}
	
	/**
	 * @param activeChar the active Game Master.
	 * @param id
	 * @param level
	 */
	private void adminAddClanSkill(L2PcInstance activeChar, int id, int level)
	{
		final L2Object target = activeChar.getTarget();
		if ((target == null) || !target.isPlayer())
		{
			activeChar.sendPacket(SystemMessageId.INCORRECT_TARGET);
			showMainPage(activeChar);
			return;
		}
		final L2PcInstance player = target.getActingPlayer();
		if (!player.isClanLeader())
		{
			final SystemMessage sm = SystemMessage.getSystemMessage(SystemMessageId.S1_IS_NOT_A_CLAN_LEADER);
			sm.addString(player.getName());
			activeChar.sendPacket(sm);
			showMainPage(activeChar);
			return;
		}
		if ((id < 370) || (id > 391) || (level < 1) || (level > 3))
		{
			activeChar.sendMessage("Usage: //add_clan_skill <skill_id> <level>");
			showMainPage(activeChar);
			return;
		}
		
		final L2Skill skill = SkillTable.getInstance().getInfo(id, level);
		if (skill == null)
		{
			activeChar.sendMessage("Error: there is no such skill.");
			return;
		}
		
		String skillname = skill.getName();
		SystemMessage sm = SystemMessage.getSystemMessage(SystemMessageId.CLAN_SKILL_S1_ADDED);
		sm.addSkillName(skill);
		player.sendPacket(sm);
		final L2Clan clan = player.getClan();
		clan.broadcastToOnlineMembers(sm);
		clan.addNewSkill(skill);
		activeChar.sendMessage("You gave the Clan Skill: " + skillname + " to the clan " + clan.getName() + ".");
		
		clan.broadcastToOnlineMembers(new PledgeSkillList(clan));
		for (L2PcInstance member : clan.getOnlineMembers(0))
		{
			member.sendSkillList();
		}
		
		showMainPage(activeChar);
	}
	
    // [L2J_JP ADD - TSL]
	private void showSkillIndexPage(L2PcInstance activeChar, String val)
	{
if (L2J_JP) {{
		StringBuilder content = new StringBuilder("<html><body>");
		content.append("<table width=260><tr>");
		content.append("<td width=40><button value=\"Main\" action=\"bypass -h admin_admin\" width=40 height=15 back=\"L2UI_ct1.button_df_down\" fore=\"L2UI_ct1.button_df\"></td>");
		content.append("<td width=180><center>ClassList Menu</center></td>");
		content.append("<td width=40><button value=\"Back\" action=\"bypass -h admin_show_skills\" width=40 height=15 back=\"L2UI_ct1.button_df_down\" fore=\"L2UI_ct1.button_df\"></td>");
		content.append("</tr></table>");
		content.append("<br>");
		NpcHtmlMessage html = new NpcHtmlMessage();
		
		int parentId = -1;
		if (val != "")
			parentId = Integer.parseInt(val);
		int classCount = 0;
		
		content.append("NXIF<br>");
		
		// Select class
		//[JOJO]Note: Ignore log message "Multiply prepared statement!"
		final String SQL_QUERY = "SELECT class_list.id AS id, classname_ja.name AS class_name FROM class_list LEFT JOIN classname_ja ON (class_list.id = classname_ja.id) WHERE class_list.parent_id = ?  ORDER BY class_list.id";
		try (java.sql.Connection con = L2DatabaseFactory.getInstance().getConnection();
			PreparedStatement statement1 = con.prepareStatement(SQL_QUERY);
			PreparedStatement statement2 = con.prepareStatement(SQL_QUERY) )
		{
			statement1.setInt(1, parentId);
			ResultSet rset = statement1.executeQuery();
			while (rset.next())
			{
				int classId = rset.getInt("id");
				String className = rset.getString("class_name");
				content.append("<a action=\"bypass -h admin_skill_list "
				        + classId + "\"><font color=LEVEL>" + className
				        + "</font></a><br1>");
				
				// Select class2
				statement2.setInt(1, classId);
				ResultSet rset2 = statement2.executeQuery();
				while (rset2.next())
				{
					int tmpId = rset2.getInt("id");
					className = rset2.getString("class_name");
					if (tmpId >= 88 && tmpId <= 118 || tmpId >= 131 && tmpId <= 136) //3rd job
						content.append("@@<a action=\"bypass -h admin_skill_list "
						        + tmpId
						        + "\"><font color=LEVEL>"
						        + className
						        + "</font></a><br1>");
					else
						content.append("@@<a action=\"bypass -h admin_skill_index "
						        + classId
						        + "\">"
						        + className
						        + "</a><br1>");
				}
				statement2.clearParameters();
				
				content.append("<br>");
				classCount++;
			}
		}
		catch (SQLException e)
		{
			_log.log(Level.WARNING, "Could not Create ClassIndex Page: ", e);
		}
		
		if (classCount == 0)
			content.append("NX܂B");
		content.append("</body></html>");
		
		// If the content is greater than for a basic blank page,
		// then assume no external HTML file was assigned.
		if (content.length() > 26)
			html.setHtml(content.toString());
		
		activeChar.sendPacket(html);
}}
	}

	// [L2J_JP ADD - TSL]
	private void showSkillListPage(L2PcInstance activeChar, String val)
	{
if (L2J_JP) {{
		StringBuilder content = new StringBuilder("<html><body>");
		content.append("<table width=260><tr>");
		content.append("<td width=40><button value=\"Main\" action=\"bypass -h admin_admin\" width=40 height=15 back=\"L2UI_ct1.button_df_down\" fore=\"L2UI_ct1.button_df\"></td>");
		content.append("<td width=180><center>SkillList Menu</center></td>");
		content.append("<td width=40><button value=\"Back\" action=\"bypass -h admin_skill_index\" width=40 height=15 back=\"L2UI_ct1.button_df_down\" fore=\"L2UI_ct1.button_df\"></td>");
		content.append("</tr></table>");
		content.append("<br>");
		NpcHtmlMessage html = new NpcHtmlMessage();
		
		StringTokenizer st = new StringTokenizer(val, " ");
		int classId = 0;
		if (st.hasMoreTokens())
			classId = Integer.parseInt(st.nextToken());
		int pageNo = 1;
		if (st.hasMoreTokens())
			pageNo = Integer.parseInt(st.nextToken());
		int pageCount = 1;
		int startCount = 15 * (pageNo - 1);
		int endCount = 15 * pageNo;
		int skillCount = 0;
		
//		content.append("XLIF<br>");
		
		// Select Skills
		try (java.sql.Connection con = L2DatabaseFactory.getInstance().getConnection();
			PreparedStatement statement = con.prepareStatement("SELECT * FROM skill_trees WHERE class_id = ? ORDER BY skill_id, level"))
		{
			statement.setInt(1, classId);
			ResultSet rset = statement.executeQuery();
			while (rset.next())
			{
				int skillId = rset.getInt("skill_id");
				int skillLv = rset.getInt("level");
				String skillName = rset.getString("name");
				int minLv = rset.getInt("min_level");
				String colorA = "", colorB = "";
				L2Skill skill = SkillTable.getInstance().getInfo(skillId, skillLv);
				if (skill.isActive()) { colorA = "<font color=00ffff>"; colorB = "</font>"; }
				
				if (skillCount >= startCount && skillCount < endCount)
				{
					content.append("<a action=\"bypass -h admin_add_skill "
					        + skillId + " " + skillLv + "\">" + colorA + "Lv." + skillLv
					        + " " + skillName + " (Lv." + minLv + ")" + colorB
					        + "</a><br1>");
				}
				skillCount++;
				if (skillCount % 15 == 0)
					pageCount++;
			}
		}
		catch (SQLException e)
		{
			_log.log(Level.WARNING, "Could not Create SkillList Page: ", e);
		}
		
		if (skillCount > 0)
		{
			content.append("<table width=260><tr>");
			content.append("<td width=40>");
			if (pageNo - 1 > 0)
			{
				content.append("<button value=\"Page"
				        + (pageNo - 1)
				        + "\" action=\"bypass -h admin_skill_list "
				        + classId
				        + " "
				        + (pageNo - 1)
				        + "\" width=40 height=15 back=\"L2UI_ct1.button_df_down\" fore=\"L2UI_ct1.button_df\">");
			}
			content.append("</td>");
			content.append("<td width=180><center>");
			content.append("<button value=\"Home\" action=\"bypass -h admin_skill_list "
			        + classId
			        + " 1\" width=40 height=15 back=\"L2UI_ct1.button_df_down\" fore=\"L2UI_ct1.button_df\">");
			content.append("</center></td>");
			content.append("<td width=40>");
			if (pageNo + 1 <= pageCount)
			{
				content.append("<button value=\"Page"
				        + (pageNo + 1)
				        + "\" action=\"bypass -h admin_skill_list "
				        + classId
				        + " "
				        + (pageNo + 1)
				        + "\" width=40 height=15 back=\"L2UI_ct1.button_df_down\" fore=\"L2UI_ct1.button_df\">");
			}
			content.append("</td>");
			content.append("</tr></table>");
		}
		else
			content.append("XL܂B");
		
		content.append("<br>");
		content.append("</body></html>");
		
		
		// If the content is greater than for a basic blank page,
		// then assume no external HTML file was assigned.
		if (content.length() > 26)
			html.setHtml(content.toString());
		
		activeChar.sendPacket(html);
}}
	}
	
	@Override
	public String[] getAdminCommandList()
	{
		return ADMIN_COMMANDS;
	}
}
