/*
 * Copyright (C) 2005-2008 L2J_JP / 2008-2013 L2J-SFJP
 * 
 * This file is part of L2J Server.
 * 
 * L2J Server is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * L2J Server is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package instances.ZakenDaydream;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.List;

import com.l2jserver.gameserver.ai.CtrlIntention;
import com.l2jserver.gameserver.datatables.SkillTable;
import com.l2jserver.gameserver.enums.MountType;
import com.l2jserver.gameserver.instancemanager.InstanceManager;
import com.l2jserver.gameserver.model.L2CommandChannel;
import com.l2jserver.gameserver.model.L2Object;
import com.l2jserver.gameserver.model.L2Party;
import com.l2jserver.gameserver.model.L2World;
import com.l2jserver.gameserver.model.actor.L2Attackable;
import com.l2jserver.gameserver.model.actor.L2Character;
import com.l2jserver.gameserver.model.actor.L2Npc;
import com.l2jserver.gameserver.model.actor.instance.L2MonsterInstance;
import com.l2jserver.gameserver.model.actor.instance.L2PcInstance;
import com.l2jserver.gameserver.model.effects.AbnormalEffect;
import com.l2jserver.gameserver.model.instancezone.InstanceWorld;
import com.l2jserver.gameserver.model.quest.Quest;
import com.l2jserver.gameserver.model.skills.L2Skill;
import com.l2jserver.gameserver.model.skills.L2SkillType;
import com.l2jserver.gameserver.network.SystemMessageId;
import com.l2jserver.gameserver.network.serverpackets.ExShowScreenMessageNPCString;
import com.l2jserver.gameserver.network.serverpackets.L2GameServerPacket;
import com.l2jserver.gameserver.network.serverpackets.PlaySound;
import com.l2jserver.gameserver.network.serverpackets.SystemMessage;
import com.l2jserver.gameserver.util.Util;
import com.l2jserver.util.Rnd;

/**
 * Cavern of the Pirate Captain (daydream)
 * 
 * @author JOJO
 */
public class ZakenDaydream extends Quest
{
	private static final boolean debug = false;
	
	private static final String qn = "ZakenDaydream";
	private static final String INSTANCE_XML = "ZakenDaydream.xml";

	private static final boolean CONFIG_ALONE_PLAYER = true;
	private static final int ZAKEN_SKILL_RATE = 15;
	
	private static final int MAX_PLAYERS = 27;
	
	private static final int[] RESET_DAY_OF_WEEK = { Calendar.MONDAY, Calendar.WEDNESDAY, Calendar.FRIDAY };
	private static final int   RESET_HOUR_OF_DAY = 6;
	private static final int   RESET_MINUTE = 30;
	private static final int EXIT_TIME = 600000;
	
	private static final int MIN_LEVEL_LO = 55;
	private static final int MAX_LEVEL_LO = 65;
	private static final int INSTANCE_ID_LO = 133;
	private static final int ZAKEN_LO = 29176;
	
	private static final int MIN_LEVEL_HI = 78;
	private static final int MAX_LEVEL_HI = 85;
	private static final int INSTANCE_ID_HI = 135;
	private static final int ZAKEN_HI = 29181;
	
	private static final int ZAKENS_CANDLE = 32705;
	private static final int PATHFINDER_WORKER = 32713;
	private static final int[] RED_CANDLE_MOBS = { 29023, 29024, 29026, 29027, 21637 };	/*FIXME*/
	
	private static final int NOTSPAWN = 0, SPAWNED = 1, DEAD = 2;
	
	// item
	private static final int CANDLE_FUSE = 15280;
	private static final int CANDLE_BLUE = 15302;
	private static final int CANDLE_RED = 15281;
	
	// coord
	protected static final int[] FLOOR_Z = { -3488, -3216, -2944 };
	
	private static final int[] VOID = null;
	protected static final int[][][] CANDLE_COORD = {
		{     VOID     ,{54240,217155},     VOID     ,{56288,217155},     VOID     },
		{{53320,218080},    (null)    ,{55265,218170},    (null)    ,{57220,218080}},
		{     VOID     ,{54340,219105},    (null)    ,{56195,219105},     VOID     },
		{{53320,220125},    (null)    ,{55265,220040},    (null)    ,{57210,220125}},
		{     VOID     ,{54240,221050},     VOID     ,{56288,221050},     VOID     },
	};
	
	protected static final int[][][] ZAKEN_COORD = {
		{     VOID     ,    (null)    ,     VOID     ,    (null)    ,     VOID     },
		{    (null)    ,{54240,218100},    (null)    ,{56290,218080},    (null)    },
		{     VOID     ,    (null)    ,{55255,219100},    (null)    ,     VOID     },
		{    (null)    ,{54250,220130},    (null)    ,{56280,220115},    (null)    },
		{     VOID     ,    (null)    ,     VOID     ,    (null)    ,     VOID     },
	};
	
	protected static final int[][] ZAKEN_ROOM_TO_CANDLE_MATRIX = {
		{1,1},{3,1},{2,2},{3,3},{1,3}
	};
	
	private static final String MUSIC_BOSS_APPEARANCE = "bs11_a";
	private static final String MUSIC_BOSS_DEAD       = "bs11_d";
	
	class TheWorld extends InstanceWorld
	{
		final int ZAKEN_ID;
		L2MonsterInstance zaken;
		final int zakenFloor, zakenWE, zakenNS;
		L2Npc[][][] candleMatrix = {
				{new L2Npc[5], new L2Npc[5], new L2Npc[5], new L2Npc[5], new L2Npc[5]},
				{new L2Npc[5], new L2Npc[5], new L2Npc[5], new L2Npc[5], new L2Npc[5]},
				{new L2Npc[5], new L2Npc[5], new L2Npc[5], new L2Npc[5], new L2Npc[5]},
			};
		final L2Npc[] blueCandles;
		final long startTime;
		
		TheWorld(int instanceId, int templateId, int ZAKEN_ID)
		{
			super();
			
			this.setInstanceId(instanceId);
			this.setTemplateId(templateId);
			this.ZAKEN_ID = ZAKEN_ID;
			
			// spawn candles
			for (int floor = 0; floor < 3; floor++)
			{
				for (int ns = 0; ns < 5; ns++)
				{
					for (int we = 0; we < 5; we++)
					{
						int[]coord = CANDLE_COORD[ns][we];
						if (coord != null)
						{
							int x = coord[0];
							int y = coord[1];
							int z = FLOOR_Z[floor];
							int h = 0;	//TODO:
							candleMatrix[floor][ns][we] = addSpawn(ZAKENS_CANDLE, x, y, z, h, false, 0, false, instanceId);
						}
					}
				}
			}
			zakenFloor = Rnd.get(3);
			int[] coord = ZAKEN_ROOM_TO_CANDLE_MATRIX[Rnd.get(5)];
			zakenNS = coord[0];
			zakenWE = coord[1];
			
			blueCandles = new L2Npc[]{
				candleMatrix[zakenFloor][zakenNS-1][zakenWE  ],
				candleMatrix[zakenFloor][zakenNS  ][zakenWE-1],
				candleMatrix[zakenFloor][zakenNS+1][zakenWE  ],
				candleMatrix[zakenFloor][zakenNS  ][zakenWE+1],
			};
			if (debug) for (L2Npc m : blueCandles) m.setTitle("!");
			
			startTime = System.currentTimeMillis();
		}
		
		//spawn zaken
		void spawnBoss()
		{
			setStatus(SPAWNED);
			int[] coord = ZAKEN_COORD[zakenNS][zakenWE];
			int x = coord[0];
			int y = coord[1];
			int z = FLOOR_Z[zakenFloor];
			zaken = (L2MonsterInstance)addSpawn(ZAKEN_ID, x, y, z, 0, true, 0, false, getInstanceId());
			broadcastPacket(new PlaySound(1, MUSIC_BOSS_APPEARANCE, 1, zaken.getObjectId(), x, y, z));
			
			for (L2Skill skill : zaken.getAllSkills())
			{
				if (!skill.isPassive())
					zaken.disableSkill(skill, 0);
			}
		}
		
		void broadcastPacket(L2GameServerPacket packet)
		{
			for (int objectId : getAllowed())
			{
				L2PcInstance member = L2World.getInstance().getPlayer(objectId);
				if (member != null && member.isOnline())
					member.sendPacket(packet);
			}
		}
		
		void addItemRandom(int chance, int itemId, long count, L2Object reference, boolean announce)
		{
			if (chance == 0)
				return;
			for (int objectId : getAllowed())
			{
				L2PcInstance member = L2World.getInstance().getPlayer(objectId);
				if (member != null && member.isOnline())
				{
					if (Rnd.get(100) < chance)
					{
						member.addItem("Quest", itemId, 1, reference, !announce);
						if (announce)
							member.broadcastPacket(SystemMessage.getSystemMessage(SystemMessageId.ANNOUNCEMENT_C1_PICKED_UP_S2).addCharName(member).addItemName(itemId));
					}
				}
			}
		}
		
		void setInstanceTime(long instanceTime)
		{
			SystemMessage zoneRestrictedMessage = SystemMessage.getSystemMessage(SystemMessageId.INSTANT_ZONE_S1_RESTRICTED)
					.addInstanceName(getTemplateId());
			for (int objectId : getAllowed())
			{
				L2PcInstance member = L2World.getInstance().getPlayer(objectId);
				if (member != null && member.isOnline())
				{
					InstanceManager.getInstance().setInstanceTime(member.getObjectId(), getTemplateId(), instanceTime);
					member.sendPacket(zoneRestrictedMessage);
				}
			}
		}
		
	}
	
	private void burnFuse(L2Npc candle)
	{
		candle.setRHandId(CANDLE_FUSE);
	}
	
	private void burnBlue(L2Npc candle)
	{
		candle.startAbnormalEffect(AbnormalEffect.IMPRISIONING_1);
		candle.setRHandId(CANDLE_BLUE);
	}
	
	private void burnRed(L2Npc candle)
	{
		candle.startAbnormalEffect(AbnormalEffect.BLEEDING);
		candle.setRHandId(CANDLE_RED);
	}
	
	private boolean isBurning(L2Npc candle)
	{
		return candle.getRightHandItem() != 0;
	}
	
	private boolean isBurningBlue(L2Npc candle)
	{
		return candle.getRightHandItem() == CANDLE_BLUE;
	}
	
	private void enterInstance(L2PcInstance player, boolean isLo)
	{
		if (InstanceManager.getInstance().getPlayerWorld(player) != null)
		{
			player.sendPacket(SystemMessageId.ALREADY_ENTERED_ANOTHER_INSTANCE_CANT_ENTER);
			return;
		}
		
		L2Party party = player.getParty();
		if (party == null)
		{
if (!CONFIG_ALONE_PLAYER) {{
			player.sendPacket(SystemMessageId.NOT_IN_PARTY_CANT_ENTER);
			return;
}}
		}
		else if (party.getLeader() != player)
		{
			player.sendPacket(SystemMessageId.ONLY_PARTY_LEADER_CAN_ENTER);
			return;
		}
		
		List<L2PcInstance> partyMembers = getPartyMembers(player);
		
		final int MIN_LEVEL, MAX_LEVEL, INSTANCE_ID, ZAKEN_ID;
		if (isLo) { INSTANCE_ID = INSTANCE_ID_LO; MIN_LEVEL = MIN_LEVEL_LO; MAX_LEVEL = MAX_LEVEL_LO; ZAKEN_ID = ZAKEN_LO; }
		else      { INSTANCE_ID = INSTANCE_ID_HI; MIN_LEVEL = MIN_LEVEL_HI; MAX_LEVEL = MAX_LEVEL_HI; ZAKEN_ID = ZAKEN_HI; }
		
		if (partyMembers.size() > MAX_PLAYERS)
		{
			player.sendPacket(SystemMessageId.PARTY_EXCEEDED_THE_LIMIT_CANT_ENTER);
			return;
		}
		
		boolean partyCondition = true;
		for (L2PcInstance member : partyMembers)
		{
			int level;
			if ((level = member.getLevel()) < MIN_LEVEL || level > MAX_LEVEL)
			{
				player.sendPacket(SystemMessage.getSystemMessage(SystemMessageId.C1_LEVEL_REQUIREMENT_NOT_SUFFICIENT)
						.addPcName(member));
				partyCondition = false;
			}
			if (!Util.checkIfInRange(1000, player, member, true))
			{
				player.sendPacket(SystemMessage.getSystemMessage(SystemMessageId.C1_IS_IN_LOCATION_THAT_CANNOT_BE_ENTERED)
						.addPcName(member));
				partyCondition = false;
			}
			if (System.currentTimeMillis() < InstanceManager.getInstance().getInstanceTime(member.getObjectId(), INSTANCE_ID))
			{
				player.sendPacket(SystemMessage.getSystemMessage(SystemMessageId.C1_MAY_NOT_REENTER_YET)
						.addPcName(member));
				partyCondition = false;
			}
		}
		if (!partyCondition)
		{
			return;
		}
		
		int instanceId = InstanceManager.getInstance().createDynamicInstance(INSTANCE_XML);
		TheWorld world = new TheWorld(instanceId, INSTANCE_ID, ZAKEN_ID);
		InstanceManager.getInstance().addWorld(world);
		
		// teleport players
		//long instanceTime = System.currentTimeMillis() + 43200000;	//TODO
		for (L2PcInstance member : partyMembers)
		{
			//InstanceManager.getInstance().setInstanceTime(member.getObjectId(), INSTANCE_ID, instanceTime);	//TODO
			member.getAI().setIntention(CtrlIntention.AI_INTENTION_IDLE);
			member.setInstanceId(instanceId);
			member.teleToLocation(52560 + Rnd.get(-100, 100), 219100 + Rnd.get(-50, 50), -3235);
			world.addAllowed(member);
		}
	}
	
	private long calculateNextInstanceTime()
	{
		long now = System.currentTimeMillis();
		GregorianCalendar reset = new GregorianCalendar();
		reset.setTimeInMillis(now);
		reset.set(Calendar.MILLISECOND, 0);
		reset.set(Calendar.SECOND, 0);
		reset.set(Calendar.MINUTE, RESET_MINUTE);
		reset.set(Calendar.HOUR_OF_DAY, RESET_HOUR_OF_DAY);
		while (reset.getTimeInMillis() < now)
			reset.add(Calendar.DATE, 1);
		while (! Util.contains(RESET_DAY_OF_WEEK, reset.get(Calendar.DAY_OF_WEEK)))
			reset.add(Calendar.DATE, 1);
		return reset.getTimeInMillis();
	}
	
	@Override
	public String onAdvEvent(String event, L2Npc npc, L2PcInstance player)
	{
		if (event == "burn")
		{
			TheWorld world = (TheWorld) InstanceManager.getInstance().getWorld(npc.getInstanceId());
			if (Util.contains(world.blueCandles, npc))
			{
				burnBlue(npc);
				int count = world.blueCandles.length;
				for (L2Npc blueCandle : world.blueCandles)
					if (isBurningBlue(blueCandle))
						--count;
				if (count == 0)
				{
					startQuestTimer("spawnBoss", 3000, npc, player);
					startQuestTimer("spawnMobs", 5000, npc, player);
					world.broadcastPacket(new ExShowScreenMessageNPCString(1800867, 10000));
				}
			}
			else
			{
				burnRed(npc);
				if (Rnd.get(100) < 33)
					startQuestTimer("sayHint", 3000, npc, player);
				startQuestTimer("spawnMobs", 5000, npc, player);
			}
		}
		else if (event == "spawnMobs")
		{
			int x = npc.getX();
			int y = npc.getY();
			int z = npc.getZ();
			int instanceId = npc.getInstanceId();
			for (int npcId : RED_CANDLE_MOBS)
			{
				for (int n = Rnd.get(1, 3); --n >= 0; )
				{
					L2MonsterInstance mob = (L2MonsterInstance) addSpawn(npcId, x, y, z, 0, true, 0, false, instanceId);
					mob.setTarget(player);
					mob.addDamageHate(player, 1, 1);
				}
			}
		}
		else if (event == "spawnBoss")
		{
			TheWorld world = (TheWorld) InstanceManager.getInstance().getWorld(npc.getInstanceId());
			world.spawnBoss();
		}
		else if (event == "onKillZakenLO")
		{
			TheWorld world = (TheWorld) InstanceManager.getInstance().getWorld(npc.getInstanceId());
			/*FIXME*/
			world.addItemRandom(2, 21713, 1, npc, true);	//󂳂ꂽUP̃N[N [T26/H5]
			world.addItemRandom(2, 6659, 1, npc, true);		//UP CAO
			world.addItemRandom(50, 6569, 1, npc, false);	//jꂽ틭XN[FAO[h
			world.addItemRandom(50, 6570, 1, npc, false);	//jꂽhXN[FAO[h
			world.addItemRandom(40, 729, 1, npc, false);	//틭XN[FAO[h
			world.addItemRandom(40, 730, 1, npc, false);	//hXN[FAO[h
			
			world.broadcastPacket(SystemMessage.getSystemMessage(SystemMessageId.RAID_WAS_SUCCESSFUL));
			world.broadcastPacket(new PlaySound(1, MUSIC_BOSS_DEAD, 1, npc.getObjectId(), npc.getX(), npc.getY(), npc.getZ()));
			world.setInstanceTime(calculateNextInstanceTime());
			InstanceManager.getInstance().getInstance(world.getInstanceId()).setDuration(EXIT_TIME);
		}
		else if (event == "onKillZakenHI")
		{
			TheWorld world = (TheWorld) InstanceManager.getInstance().getWorld(npc.getInstanceId());
			/*FIXME*/
			int time, rate;
			time = (int)(System.currentTimeMillis() - world.startTime) / 60000;
			if (time <= 5)
				rate = 100;
			else if (time <= 10)
				rate = 75;
			else if (time <= 15)
				rate = 50;
			else if (time <= 20)
				rate = 25;
			else if (time <= 25)
				rate = 12;
			else
				rate = 6;
			world.addItemRandom(8 * rate / 100, 21712, 1, npc, true);	//sł̃UP CAO [T26/H5]
			world.addItemRandom(10 * rate / 100, 21713, 1, npc, true);	//󂳂ꂽUP̃N[N [T26/H5]
			world.addItemRandom(10 * rate / 100, 6659, 1, npc, true);	//UP CAO
			world.addItemRandom(50 * rate / 100, 15763, 1, npc, true);	//󂳂ꂽo[yX O
			world.addItemRandom(50 * rate / 100, 15764, 1, npc, true);	//󂳂ꂽo[yX CAO
			world.addItemRandom(50 * rate / 100, 15765, 1, npc, true);	//󂳂ꂽo[yX lbNX
			
			world.broadcastPacket(SystemMessage.getSystemMessage(SystemMessageId.RAID_WAS_SUCCESSFUL));
			world.broadcastPacket(new PlaySound(1, MUSIC_BOSS_DEAD, 1, npc.getObjectId(), npc.getX(), npc.getY(), npc.getZ()));
			world.setInstanceTime(calculateNextInstanceTime());
			InstanceManager.getInstance().getInstance(world.getInstanceId()).setDuration(EXIT_TIME);
		}
		else if (event == "sayHint")
		{
			TheWorld world = (TheWorld) InstanceManager.getInstance().getWorld(npc.getInstanceId());
			sayHint(world, npc);
		}
		else if (event.equals("LO"))
		{
				//32713-2.htm <a action="bypass -h Quest ZakenDaydream LO">̃UP|ɍs</a>
				enterInstance(player, true);
		}
		else if (event.equals("HI"))
		{
				//32713-3.htm <a action="bypass -h Quest ZakenDaydream HI">ʒ̃UP|ɍs</a>
				enterInstance(player, false);
		}
		return super.onAdvEvent(event, npc, player);
	}
	
	@Override
	public String onFirstTalk(L2Npc npc, L2PcInstance player)
	{
		switch (npc.getId())
		{
			case PATHFINDER_WORKER:
				return "data/html/default/32713.htm";
			case ZAKENS_CANDLE:
				synchronized (InstanceManager.getInstance().getWorld(npc.getInstanceId()))
				{
					if (!isBurning(npc))
					{
						burnFuse(npc);
						startQuestTimer("burn", 10000, npc, player);
					}
				}
				return null;
		}
		throw new RuntimeException();
	}
	
	@Override
	public String onSpawn(L2Npc npc)
	{
		switch (npc.getId())
		{
			case ZAKEN_LO:
			case ZAKEN_HI:
				((L2Attackable)npc).setOnKillDelay(100);	//Default 5000ms.
				break;
		}
		return super.onSpawn(npc);
	}
	
	@Override
	public String onKill(L2Npc npc, L2PcInstance killer, boolean isPet)
	{
		TheWorld world = (TheWorld) InstanceManager.getInstance().getWorld(npc.getInstanceId());
		switch (npc.getId())
		{
			case ZAKEN_LO:
				world.setStatus(DEAD);
				cancelAllQuestTimers(world.getInstanceId());
				startQuestTimer("onKillZakenLO", 777, npc, killer);
				break;
			case ZAKEN_HI:
				world.setStatus(DEAD);
				cancelAllQuestTimers(world.getInstanceId());
				startQuestTimer("onKillZakenHI", 777, npc, killer);
				break;
		}
		return super.onKill(npc, killer, isPet);
	}
	
	private void cancelAllQuestTimers(int instanceId)
	{
	//	cancelQuestTimers("xxx", instanceId);
	//	cancelQuestTimers("yyy", instanceId);
	//	cancelQuestTimers("zzz", instanceId);
	}
	
	private void sayHint(TheWorld world, L2Npc candle)
	{
		if (world.getStatus() != NOTSPAWN)
			return;
		for (int floor = 0; floor < 3; floor++)
		{
			for (int ns = 0; ns < 5; ns++)
			{
				for (int we = 0; we < 5; we++)
				{
					if (world.candleMatrix[floor][ns][we] == candle)
					{
						final int say;
						if (floor < world.zakenFloor)
							say = 1800868;
						else if (floor > world.zakenFloor)
							say = 1800870;
						else
							say = 1800869;
						world.broadcastPacket(new ExShowScreenMessageNPCString(say, 10000));
						return;
					}
				}
			}
		}
	}
	
	private List<L2PcInstance> getPartyMembers(L2PcInstance player)
	{
		L2Party party;
		L2CommandChannel commandChannel;
		if ((party = player.getParty()) == null)
		{
			List<L2PcInstance> m = new ArrayList<>(1);
			m.add(player);
			return m;
		}
		else if ((commandChannel = party.getCommandChannel()) == null)
		{
			return party.getMembers();
		}
		else 
		{
			List<L2PcInstance> m = new ArrayList<>(27);
			for (L2Party pp : commandChannel.getPartys())
				m.addAll(pp.getMembers());
			return m;
		}
	}
	
	/**********************************************************************************************/
	@Override
	public String onAttack(L2Npc npc, L2PcInstance attacker, int damage, boolean isPet)
	{
		//TRACE("onAttack("+STR(npc)+","+STR(attacker)+","+damage+","+isPet+")");
		final int npcId = npc.getId();
		if (npcId == ZAKEN_LO || npcId == ZAKEN_HI)
		{
			if (attacker.getMountType() == MountType.STRIDER)
			{
				// 4258 A` XgC_[ X[(XgC_[ɏ҂̑xቺ܂B)
				if (attacker.getFirstEffect(4258) == null)
				{
					npc.setTarget(attacker);
					npc.doCast(SkillTable.getInstance().getInfo(4258, 1));/*eXg*/
				}
			}
			if (Rnd.get(10) == 0)
				doRandamSkill(npc, attacker);
		}
		return super.onAttack(npc, attacker, damage, isPet);
	}
	
	@Override
	public String onSkillSee(L2Npc npc, L2PcInstance caster, L2Skill skill, L2Object[] targets, boolean isPet)
	{
		//TRACE("onSkillSee("+STR(npc)+","+STR(caster)+","+STR(skill)+","+STR(targets)+","+isPet+")");
		final int npcId = npc.getId();
		if (npcId == ZAKEN_LO || npcId == ZAKEN_HI)
		{
			if (Rnd.get(12) == 0)
				doRandamSkill(npc, caster);
		}
		return super.onSkillSee(npc, caster, skill, targets, isPet);
	}
	
	@Override
	public String onAggroRangeEnter(L2Npc npc, L2PcInstance player, boolean isPet)
	{
		//TRACE("onAggroRangeEnter("+STR(npc)+","+STR(player)+","+isPet+")");
		final int npcId = npc.getId();
		if (npcId == ZAKEN_LO || npcId == ZAKEN_HI)
		{
			if (Rnd.get(15) == 0)
				doRandamSkill(npc, player);
		}
		return super.onAggroRangeEnter(npc, player, isPet);
	}
	
	private void doRandamSkill(L2Npc npc, L2PcInstance player)
	{
		if (npc.isCastingNow())
		{
			//TRACE(STR(npc)+".isCastingNow()");
			return;
		}
		
		int skillId = 0;
		final int chance = Rnd.get(15 * ZAKEN_SKILL_RATE);
		if (chance < 1) skillId = 4216;	// G̕U
		else if (chance < 2) skillId = 4217;	// ͈͓G̕U
		else if (chance < 4) skillId = 6690;	// z[h
		else if (chance < 8) skillId = 6689;	// HP/MP̋z
		else if (chance < 15) {
			if (player == ((L2Attackable) npc).getMostHated() && Util.checkIfInRange(100, npc, player, true)) skillId = 6692;	// 񓁗͈͕̔KEZ
		}
		else if (Rnd.get(2) == 0) {
			if (player == ((L2Attackable) npc).getMostHated()) skillId = 6691;	// 񓁗̕KEZ
		}
		if (skillId != 0) {
			npc.setTarget(player);
			npc.doCast(SkillTable.getInstance().getInfo(skillId, 1));
		}
	}
	
	@Override
	public String onSpellFinished(L2Npc npc, L2PcInstance player, L2Skill skill)
	{
		//TRACE("onSpellFinished("+STR(npc)+","+STR(player)+","+STR(skill)+")");
		switch (skill.getId())
		{
			case 4222:	//uԈړ
			{
				if (skill.getSkillType() != L2SkillType.DUMMY) throw new RuntimeException();
				teleToRandom(npc, npc, 0);
				npc.getAI().setIntention(CtrlIntention.AI_INTENTION_ACTIVE);
				break;
			}
			case 4216:	///G̕U
			{
				if (skill.getSkillType() != L2SkillType.DUMMY) throw new RuntimeException();
				teleToRandom(npc, player, 300);
				((L2Attackable) npc).stopHating(player);
				
				L2Character nextTarget = ((L2Attackable) npc).getMostHated();
				if (nextTarget != null)
					npc.getAI().setIntention(CtrlIntention.AI_INTENTION_ATTACK, nextTarget);
				break;
			}
			case 4217:	//͈͓G̕U
			{
				if (skill.getSkillType() != L2SkillType.DUMMY) throw new RuntimeException();
				final int range = skill.getEffectRange();
			//	final int range = skill.getCastRange();
				for (L2PcInstance pc : npc.getKnownList().getKnownPlayersInRadius(range))
				{
					if (pc == null || pc.isDead() || !pc.isVisible())
						continue;
					teleToRandom(npc, pc, 300);
					((L2Attackable) npc).stopHating(pc);
				}
				L2Character nextTarget = ((L2Attackable) npc).getMostHated();
				if (nextTarget != null)
					npc.getAI().setIntention(CtrlIntention.AI_INTENTION_ATTACK, nextTarget);
				break;
			}
		}
		return super.onSpellFinished(npc, player, skill);
	}
	private void teleToRandom(L2Character o, L2Character target, int randomOffset)
	{
		int x, y, z;
		do
		{
			int[] nswe = ZAKEN_ROOM_TO_CANDLE_MATRIX[Rnd.get(5)];
			int ns = nswe[0];
			int we = nswe[1];
			int[] xy = ZAKEN_COORD[ns][we];
			x = xy[0];
			y = xy[1];
			z = FLOOR_Z[Rnd.get(3)];
		} while (Util.calculateDistance(o.getX(), o.getY(), x, y) < 650 && Math.abs(o.getZ() - z) < 270);
		target.teleToLocation(x, y, z, randomOffset);
	}
	/**********************************************************************************************/
	
	private ZakenDaydream(int questId, String name, String descr)
	{
		super(questId, name, descr);
		
		addStartNpc(PATHFINDER_WORKER);
		addFirstTalkId(PATHFINDER_WORKER);
		addFirstTalkId(ZAKENS_CANDLE);
		addTalkId(PATHFINDER_WORKER);
		addSpawnId(ZAKEN_LO);
		addSpawnId(ZAKEN_HI);
		addKillId(ZAKEN_LO);
		addKillId(ZAKEN_HI);
		
		addAggroRangeEnterId(ZAKEN_LO);
		addAggroRangeEnterId(ZAKEN_HI);
		addAttackId(ZAKEN_LO);
		addAttackId(ZAKEN_HI);
		addSkillSeeId(ZAKEN_LO);
		addSkillSeeId(ZAKEN_HI);
		addSpellFinishedId(ZAKEN_LO);
		addSpellFinishedId(ZAKEN_HI);
	}
	
	public static void main(String[] args)
	{
		if (debug) {
			if (SkillTable.getInstance().getInfo(6689, 1).getSkillType() == L2SkillType.NOTDONE) throw new RuntimeException();	// HP/MP̋z
			if (SkillTable.getInstance().getInfo(6690, 1).getSkillType() == L2SkillType.NOTDONE) throw new RuntimeException();	// z[h
			if (SkillTable.getInstance().getInfo(6691, 1).getSkillType() == L2SkillType.NOTDONE) throw new RuntimeException();	// 񓁗̕KEZ
			if (SkillTable.getInstance().getInfo(6692, 1).getSkillType() == L2SkillType.NOTDONE) throw new RuntimeException();	// 񓁗͈͕̔KEZ
		}
		
		new ZakenDaydream(-1, qn, "instances");
	}
}
