#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>

#include "mtytypes.h"
#include "translate.h"

#include "l2.h"
#include "triplib.h"

#define   MAX_TRIPS  7615	/* nbVlƂ̍őgbv */
#define   NKIND    (0x3fff+1)	/* nbVl̍ől */
#define   L2INDEX( bin )  ((bin & 0x3fff00000)>>20)	/* nbVl */
/*
NKIND * MAX_TRIPS <= 128M
fffff(20) *   118 = 123731850, 24: 37602 20: 45427 16:  6258 12:  5923 8:  4129
0ffff(16) *  1903 = 124713105, 24:363592 20:406040 16:226101 12:216980 8: 27771
07fff(15) *  3807 = 124743969, 24:634769 20:769452 16:430397 12:413364 8: 56438
03fff(14) *  7615 = 124756545, 24:911322 20:------ 16:------ 12:542854 8:147056
00fff(12) * 30460 = 124733700, 24:------ 20:------ 16:------ 12:750796 8:383992

0x3fff000000
001111 111111 111100 000000 000000 000000
4 - 6 
???456????
1234567???
???1234567
Œ 7 
*/

#define   MINLENPAD   7	/* ̍ŏ */
#define   MINLENCUT   6	/* O̍ŏ */

struct bintrip {
 uint64_t	btrip;
 uint64_t	bmask;
};

struct bintrip	trip[NKIND][MAX_TRIPS];
unsigned int	tripNum[NKIND];

uint64_t	bmask[10] = {
  0x000000000000003fULL,
  0x0000000000000fc0ULL,
  0x000000000003f000ULL,
  0x0000000000fc0000ULL,
  0x000000003f000000ULL,
  0x0000000fc0000000ULL,
  0x000003f000000000ULL,
  0x0000fc0000000000ULL,
  0x003f000000000000ULL,
  0x0fc0000000000000ULL
};

unsigned int	kind;
unsigned int	kindTable[128];

#define   HIT_FILE    "hit-l2.txt"
#define   LIST_FILE   "l2.txt"
#define   ERR_FILE    "error.txt"

FILE	*hitFile;
FILE	*logFile;
FILE	*errFile;

#define   LIST_LINE_LEN   200	/* ^[Qbgt@C̍ős */

enum types { ALL, CAP, LEAD, CAPLEAD };

static void	hit( uint64_t bin, unsigned char *key, uint64_t targ );
static void	bin2trip( uint64_t bin, unsigned char *trip );
static void	trip2bin( uint64_t *bin, unsigned char *trip );
static void	signalHandler( int sig );
static int	checkTrip( unsigned char *trip, unsigned int tripLen );
static void	addTrip( unsigned char *trip );
static void	mkvar( unsigned char *trip, enum types vtype );
static void	mkvarSub( unsigned char *org, enum types vtype, unsigned int nq );
static void	mkvarSub2( unsigned char *org, enum types vtype, unsigned int pad );

#ifdef DUMPTARGET
static void	dumpTargets();
#endif /* DUMPTARGET */

#ifdef HASHCHECK
static void	hashCheck();
#endif /* HASHCHECK */


/*
           1         2         3         4         5         6
 01234567890123456789012345678901234567890123456789012345678901234
 ./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz?
 . :  000000
 / :  000001
 0 :  000010
 9 :  001011
 A :  001100
 Z :  100101
 a :  100110
 z :  111111
 ? : 1000000
*/
static unsigned const char	bit2char[] = "./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";

unsigned int       tripCount;

static
void
bin2trip( bin, trip )
uint64_t	bin;
unsigned char	*trip;
{
  int	i;
  unsigned int	c;

  for ( i = 1; i < 11; i++ ) {
   c = ( bin >> (6 * (i - 1))) & 0x3F;
   trip[i - 1] = bit2char[c];
  }
  trip[10] = '\0';

#ifdef OLDDEBUG
  {
   uint64_t	tmpbin;

   printf( "bin2trip : trip = <%s>\n", trip );
   trip2bin( &tmpbin, trip );
   printf( "bin = 0x%x, ", bin>>32 );
   printf( "0x%x\n", bin & 0xffffffff );
   printf( "tmpbin = 0x%x, ", tmpbin>>32 );
   printf( "0x%x\n", tmpbin & 0xffffffff );
  }
#endif /* DEBUG */
}

static
void
trip2bin( bin, trip )
uint64_t	*bin;
unsigned char	*trip;
{
  int	i, j;

/*
 0123456789
 abc
 len = 3
 */
  *bin = 0;
  for ( i = strlen( trip ) - 1; i >= 0; i-- ) {
   if ( trip[i] == '?' ) {
    j = 0;
   } else {
    for ( j = 0; bit2char[j] != '\0'; j++ ) {
     if ( trip[i] == bit2char[j] ) break;
    }
   }
   *bin <<= 6;
   *bin |= j;

#ifdef OLDDEBUG
   printf( "bit2char[%d] = %c\n", j, bit2char[j] );
   printf( "*bin = 0x%x, ", (*bin)>>32 );
   printf( "0x%x\n", *bin & 0xffffffff );
#endif /* DEBUG */
  }
}

static
void
signalHandler( sig )
int	sig;
{
  time_t	t;
  char	timeStr[26];

  t = time( NULL ); strcpy( timeStr, ctime( &t ) ); timeStr[19] = '\0';
  switch ( sig ) {
#ifndef __MINGW32__
   case SIGHUP :
    (void)signal( SIGHUP, signalHandler );
    fflush( stdout );
    break;
#endif /* __MINGW32__ */
   case SIGINT :
   case SIGTERM :
    fclose( hitFile );
    exit( 0 );
  }
}

static
void
hit( bin, key, targ )
uint64_t	bin;
unsigned char	*key;
uint64_t	targ;
{
  unsigned char	res[30], tres[30], tmpKey[30];

  bin2trip( bin, res );
  bin2trip( targ, tres );
  memcpy( tmpKey, key, 30 );
  translate( tmpKey, 0, 1 );

  printf(           "%s : #%s (%s)\n", res, tmpKey, tres );
  fprintf( hitFile, "%s : #%s\n", res, tmpKey );
  fflush( stdout );
  fflush( hitFile );

  return;
}

static
int
checkTrip( trip, tripLen )
unsigned char	*trip;
unsigned int	tripLen;
{
  unsigned int	i;
  char	tripChars[] = "abcdefghijklmnopqrstuvwxyz"
                      "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
                      "0123456789./?";
  char	specialChars[] = ".26AEIMQUYcgkosw";

  if ( tripLen > 10 ) {
   return( -1 );
  }
  for ( i = 0; i < tripLen && i < 10; i++ ) {
   if ( strchr( tripChars, trip[i] ) == NULL ) {
    return( -1 );
   }
  }
  if ( tripLen == 10 ) {
#ifdef OLDDEBUG
   printf( "specialChars = %s, trip[9] = %c\n", specialChars, trip[9] );
#endif /* DEBUG */
   if ( strchr( specialChars, trip[9] ) == NULL ) {
    return( -1 );
   }
  }

  return( 0 );
}

#ifdef DUMPTARGET
static
void
dumpTargets()
{
  unsigned int	i, j;
  unsigned char	ret[20];

  for ( i = 0; i < NKIND; i++ ) {
   if ( tripNum[i] != 0 ) {
    printf( "tripNum[%d] = %d\n", i, tripNum[i] );
   }
  }
  for ( i = 0; i < NKIND; i++ ) {
   if ( tripNum[i] != 0 ) {
    for ( j = 0; j < tripNum[i]; j++ ) {
     bin2trip( trip[i][j].btrip[0], trip[i][j].btrip[1], ret );
     printf( "trip[%02d][%02d] = %s\n", i, j, ret+3 );
    }
   }
  }
}
#endif /* DUMPTARGET */

#ifdef HASHCHECK
static
void
hashCheck()
{
  unsigned int	i, max, maxIdx, zero;

  zero = 0;
  max = 0;
  for ( i = 0; i < NKIND; i++ ) {
   if ( tripNum[i] == 0 ) {
     zero++;
   }
   if ( tripNum[i] > max ) {
     max = tripNum[i];
     maxIdx = i;
   }
  }
  printf( "max = %4d@%6d, zero = %6d/%6d (%3.1f)\n",
          max, maxIdx, zero, NKIND, zero * 100.0 / NKIND );
}
#endif /* HASHCHECK */

static
void
addTrip( tripv )
unsigned char	*tripv;
{
  uint64_t	out;
  unsigned long	idx;
  uint64_t	mask0, mask1;
  unsigned int	tLen;
  unsigned int	i;

  tLen = strlen( tripv );
  if ( tLen < MINLENPAD || tLen > TRIP_LEN ) {
   return;
  }
  if ( checkTrip( tripv, tLen ) != 0 ) {
   return;
  }

 #ifdef OLDDEBUG
  printf( "addtrip : <%s>\n", tripv );
 #endif /* DEBUG */

 #ifdef DUMPPAT
  printf( "%s\n", tripv );
 #endif /* DUMPPAT */

  tripCount++;
  trip2bin( &out, tripv );
  idx = L2INDEX( out );
  if ( tripNum[idx] == MAX_TRIPS ) {
   fprintf( stderr, "tripv = <%s>, tripCount = %d\n", tripv, tripCount );
   fprintf( stderr, "trip[%lu] overflow.\n", idx );
   exit( 1 );
  }
  trip[idx][tripNum[idx]].btrip = out;
  mask0 = mask1 = 0;
  for ( i = 0; i < tLen && i < 10; i++ ) {
   if ( tripv[i] != '?' ) {
    mask0 |= bmask[i];
   }
  }
  trip[idx][tripNum[idx]].bmask = mask0;
 #ifdef OLDDEBUG
  strcpy( trip[idx][tripNum[idx]].trip, tripv );
 #endif /* DEBUG */
  tripNum[idx]++;

#ifdef OLDDEBUG
  if ( strcmp( startLog.s, "12345678" ) != 0 ) {
   printf( "idx = %d, tripNum[%d] = %d, tripv = <%s>, tripCount = %d\n",
           idx, idx, tripNum[idx], tripv, tripCount );
   printf( "4 : startLog.s = %s\n", startLog.s );
   exit( 1 );
  }
#endif /* DEBUG */
}

static
void
mkvarSub( org, vtype, nq )
unsigned char	*org;
enum types	vtype;
unsigned int	nq;
{
  unsigned int	i, j, k;
  unsigned char	*pat[3];
  unsigned char	work[30];
  unsigned char	*workPtr;

/*
@ : CAP     召Œ
& : CAPLEAD 召擪Œ
^ : LEAD    擪Œ
*/
  if ( vtype == LEAD || vtype == CAPLEAD ) {
   nq = 0;
  }

  for ( i = 0; i <= nq; i++ ) {
   workPtr = work;
   for ( j = 0; j < i; j++ ) {
    *workPtr = '?';
    workPtr++;
   }
   strcpy( workPtr, org );
   addTrip( work );
   if ( vtype != CAPLEAD && vtype != CAP ) {
    make3Pattern( work, pat );
    for ( k = 0; k < 3; k++ ) {
     if ( pat[k] != NULL ) {
      addTrip( pat[k] );
     }
    }
   }
  }
}

static
void
mkvarSub2( org, vtype, pad )
unsigned char	*org;
enum types	vtype;
unsigned int	pad;
{
  unsigned int	i, j, len;
  unsigned char	delm[3];
  unsigned char	work[30];
  unsigned char	*workPtr;

  delm[0] = '.';
  delm[1] = '/';
  delm[2] = '\0';

  len = strlen( org );
  strcpy( work, org );
  for ( i = 0; delm[i] != '\0'; i++ ) {
   workPtr = work + len;
   for ( j = 0; j < pad; j++ ) {
    *workPtr = delm[i];
    workPtr++;
   }
   *workPtr = '\0';
   mkvarSub( work, vtype, TRIP_LEN - len );
#ifdef OLDDEBUG
   printf( "mkvarSub2 : <%s>\n", work );
#endif /* DEBUG */
  }
}

static
void
mkvar( tmpTrip, vtype )
unsigned char	*tmpTrip;
enum types	vtype;
{
  unsigned int	tmpTripLen;

  tmpTripLen = strlen( tmpTrip );
  if ( tmpTripLen > TRIP_LEN ) {
   return;
  }
  if ( tmpTripLen >= MINLENPAD ) {
   mkvarSub( tmpTrip, vtype, TRIP_LEN - tmpTripLen );
   return;
  }
  if ( tmpTripLen >= MINLENCUT ) {
   mkvarSub2( tmpTrip, vtype, MINLENPAD - tmpTripLen );
   return;
  }
}

void
readList()
{
  unsigned int	i;
  unsigned int	lineNo;
  enum types	vtype;
  FILE	*tripFile;
  unsigned char	tmpTrip[LIST_LINE_LEN];
  unsigned char	*tmpTripPtr;

  for ( i = 0; i < NKIND; i++ ) {
   tripNum[i] = 0;
  }

  if ( (tripFile = fopen( LIST_FILE, "r" )) == NULL ) {
   fprintf( stderr, "Can't open %s.\n", LIST_FILE );
   exit( 1 );
  }

  tripCount = 0;
  lineNo = 0;
  while ( fgets( tmpTrip, LIST_LINE_LEN, tripFile ) != NULL ) {
   lineNo++;
   tmpTripPtr = tmpTrip;
   for ( i = 0; tmpTrip[i] != '\0'; i++ ) {
    if ( tmpTrip[i] == '\n' || tmpTrip[i] == '\r' ||
         tmpTrip[i] == ' ' || tmpTrip[i] == '\t' ) {
     tmpTrip[i] = '\0';
     break;
    }
   }
#ifdef OLDDEBUG
   printf( "readList : <%s>\n", tmpTrip );
#endif /* DEBUG */
   if ( tmpTrip[0] != '#' && tmpTrip[0] != '\0' ) {
    switch ( tmpTrip[0] ) {
     case '&' :
      vtype = CAPLEAD;
      tmpTripPtr++;
      break;
     case '@' :
      vtype = CAP;
      tmpTripPtr++;
      break;
     case '^' :
      vtype = LEAD;
      tmpTripPtr++;
      break;
     default :
      vtype = ALL;
      break;
    }
    mkvar( tmpTripPtr, vtype );
   }
  }
  fclose( tripFile );

  printf( "Pattern : %d\n", tripCount );

#ifdef DUMPPAT
  exit( 0 );
#endif /* DUMPPAT */

#ifdef DUMPTARGET
  dumpTargets();
  exit( 0 );
#endif /* DUMPTARGET */

#ifdef HASHCHECK
  hashCheck();
#endif /* HASHCHECK */

  fflush( stdout );
}

void
initComm()
{
  if ( (errFile = fopen( ERR_FILE, "a" )) == NULL ) {
   fprintf( stderr, "Can't open %s.\n", ERR_FILE );
   exit( 1 );
  }

  if ( (hitFile = fopen( HIT_FILE, "a" )) == NULL ) {
   fprintf( stderr, "Can't open %s.\n", HIT_FILE );
   exit( 1 );
  }

  (void)signal( SIGINT, signalHandler );
  (void)signal( SIGTERM, signalHandler );
#ifndef __MINGW32__
  (void)signal( SIGHUP, signalHandler );
#endif /* __MINGW32__ */
}

void
checkIt( bin, key )
uint64_t	bin;
unsigned char	*key;
{
  int	i;
  unsigned long	kind;

  kind = L2INDEX( bin );
  for ( i = 0; i < tripNum[kind]; i++ ) {
   if ( ((bin&trip[kind][i].bmask) == trip[kind][i].btrip) ) {
     hit( bin, key, trip[kind][i].btrip );
     return;
   }
  }
}
