#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

int
file_exists(char *file)
{
    struct stat st;
    int ret;

    ret = stat(file, &st);
    if (ret == -1) {
        switch (errno) {
        case ENOENT:
            return 0;
        case ENAMETOOLONG:
            warn("enametoolong: %d", strlen(file));
            return 0;
        default:
            error("failed to stat() for '%s' (%s)", file, strerror(errno));
        }
    }

    /* file exists */
    return 1;
}

int
file_mtime(char *file, time_t *t)
{
    struct stat st;
    int ret;

    ret = stat(file, &st);
    if (ret == -1) {
        if (errno == ENOENT)
            return -1;
        else
            error("failed to stat() for '%s' (%s)", file, strerror(errno));
    }

    *t = st.st_mtime;
    return 0;
}

int
copy_stream(FILE *rfp, FILE *wfp)
{
    int readsz;
    char buf[BUFSIZ];

    while ((readsz = fread(buf, 1, sizeof(buf), rfp)) != 0) {
        if (fwrite(buf, readsz, 1, wfp) == 0) {
            return -1;
        }
    }

    if (ferror(rfp)) {
        return -1;
    }

    return 0;
}

int
move_file_to_stream(char *file, FILE *wfp)
{
    FILE *rfp = NULL;

    if ((rfp = fopen(file, "r")) == NULL)
        goto err;

    copy_stream(rfp, wfp);

    if (rfp) fclose(rfp);

    unlink(file);

    return 0;
err:
    if (rfp) fclose(rfp);

    return -1;
}

int
xrename(char *from, char *to)
{
    FILE *rfp = NULL, *wfp = NULL;

    if ((rfp = fopen(from, "r")) == NULL)
        goto err;

    if ((wfp = fopen(to, "w")) == NULL)
        goto err;

    copy_stream(rfp, wfp);

    if (rfp) fclose(rfp);
    if (wfp) fclose(wfp);

    unlink(from);

    return 0;
err:
    if (rfp) fclose(rfp);
    if (wfp) fclose(wfp);

    return -1;
}
