/***********************************************************
	ar.c -- main file
***********************************************************/

static char *version = "0.01";

static char *usage =
    "ar -- compression archiver -- written by Haruhiko Okumura\n"
    "  PC-VAN:SCIENCE        CompuServe:74050,1022\n"
    "  NIFTY-Serve:PAF01022  INTERNET:74050.1022@compuserve.com\n"
    "Usage: ar command archive [file ...]\n"
    "Commands:\n"
    "   a: Add files to archive (replace if present)\n"
    "   x: Extract files from archive\n"
    "   r: Replace files in archive\n"
    "   d: Delete files from archive\n"
    "   p: Print files on standard output\n"
    "   l: List contents of archive\n"
    "If no files are named, all files in archive are processed,\n"
    "   except for commands 'a' and 'd'.\n"
    "You may copy, distribute, and rewrite this program freely.\n";

/***********************************************************

Structure of archive block (low order byte first):
-----preheader
 1	basic header size
		= 25 + strlen(filename) (= 0 if end of archive)
 1	basic header algebraic sum (mod 256)
-----basic header
 5	method ("-lh0-" = stored, "-lh5-" = compressed)
 4	compressed size (including extended headers)
 4	original size
 4	not used
 1	0x20
 1	0x01
 1	filename length (x)
 x	filename
 2	original file's CRC
 1	0x20
 2	first extended header size (0 if none)
-----first extended header, etc.
-----compressed file

***********************************************************/

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <unistd.h>
#include "ar.h"

struct lha_method methods[] = {
    /* note: dicbit == 0 means no compress */
    /*       (1U << pbit) > np(dicbit+1) */

    /* id, dicbit, pbit, maxmatch */
    {"-lh0-", 0,  0,  0},  /* 0: no compress */
    {"-lh1-", 12, 0, 60},  /* 1: 2^12 =  4KB dynamic huffman (LHarc) */
    {"-lh2-", 13, 0,256},  /* 2: 2^13 =  8KB dynamic huffman */
    {"-lh3-", 13, 0,256},  /* 3: 2^13 =  8KB static huffman */
    {"-lh4-", 12, 4,256},  /* 4: 2^12 =  4KB static huffman (pos and len)*/
    {"-lh5-", 13, 4,256},  /* 5: 2^13 =  8KB static huffman (pos and len)*/
    {"-lh6-", 15, 5,256},  /* 6: 2^15 = 32KB static huffman (pos and len)*/
    {"-lh7-", 16, 5,256},  /* 7: 2^16 = 64KB static huffman (pos and len)*/
    {"-lzs-", 11, 0, 17},  /* 8: 2^11 =  2KB (LArc) */
    {"-lz5-", 12, 0, 17},  /* 9: 2^12 =  4KB (LArc) */
    {"-lz4-", 0,  0,  0},  /*10: no compress (LArc) */
    {"-lhd-", 0,  0,  0},  /*11: directory */
};

struct lha_opts opts;

struct lha_method *
which_method(char *id)
{
    int i;

    for (i = 0; i < sizeof(methods)/sizeof(methods[0]); i++) {
        if (memcmp(id, methods[i].id, sizeof(methods[0].id)) == 0) {
            return &methods[i];
        }
    }
    return NULL;
}

static void
print_usage()
{
    printf("%s", usage);
    exit(0);
}

static void
print_version()
{
    printf("version %s\n", version);
    exit(0);
}

unsigned int
ratio(off_t a, off_t b)
{                               /* [(1000a + [b/2]) / b] */
    int i;

    for (i = 0; i < 3; i++)
        if (a <= ULONG_MAX / 10) /* XXX: should use max of off_t instead of ULONG_MAX */
            a *= 10;
        else
            b /= 10;
    if ((a + (b >> 1)) < a) {
        a >>= 1;
        b >>= 1;
    }
    if (b == 0)
        return 0;
    return (unsigned int) ((a + (b >> 1)) / b);
}

void
skip(FILE *fp, struct lzh_header *h)
{
    int i;
    if (opts.archive_to_stdio)
        for (i = 0; i < h->compsize; i++)
            fgetc(fp);
    else
        fseek(fp, h->compsize, SEEK_CUR);
}

static void
copy(FILE *arcfile, FILE *outfile, struct lzh_header *h)
{
    char buf[BUFSIZ];
    off_t remainder = h->compsize;

    write_header(outfile, h);
    while (remainder != 0) {
        size_t n = (size_t) MIN(remainder, sizeof(buf));

        if (fread(buf, 1, n, arcfile) != n)
            error("Can't read");
        if (fwrite(buf, 1, n, outfile) != n)
            error("Can't write");
        remainder -= n;
    }
}

int
get_line(char *s, int n)
{
    int i, c;

    i = 0;
    while ((c = getchar()) != EOF && c != '\n')
        if (i < n)
            s[i++] = (char) c;
    s[i] = '\0';
    return i;
}

static int
match(char *s1, char *s2)
{
    for (;;) {
        while (*s2 == '*' || *s2 == '?') {
            if (*s2++ == '*')
                while (*s1 && *s1 != *s2)
                    s1++;
            else if (*s1 == 0)
                return 0;
            else
                s1++;
        }
        if (*s1 != *s2)
            return 0;
        if (*s1 == 0)
            return 1;
        s1++;
        s2++;
    }
}

static int
search(int argc, char *argv[], char *filename)
{
    int i;

    if (argc == 0)
        return -1;
    for (i = 0; i < argc; i++)
        if (argv[i] && match(filename, argv[i]))
            return i+1;
    return 0;
}

#include "getopt_long.h"

void
parse_args(int argc, char **argv)
{
    int c;

    for (;;) {
        /* int this_option_optind = optind ? optind : 1; */
        int option_index = 0;

        enum {
            LHA_OPT_HELP = 128,
            LHA_OPT_VERSION,
        };

        static struct option long_options[] = {
            /* name, has_arg, *flag, val */
            /* has_arg:
               no_argument (0)
               required_argument (1)
               optional_argument (2)
               flag:
               NULL: getopt_long() return val
               non-NULL: getopt_long() return 0, and *flag set val.
            */
            {"help", no_argument, NULL, LHA_OPT_HELP},
            {"version", no_argument, NULL, LHA_OPT_VERSION},
            {0, 0, 0, 0}
        };

        c = getopt_long(argc, argv, "012fgo[567]q[012]vw:z",
                        long_options, &option_index);

        if (c == -1) break;     /* end of parsing options */

        switch (c) {
        case '?':
            print_usage();
            break;
        case 0:
            /* set value by long option */
            break;
        case '0': case '1': case '2':
            /* header level */
            opts.header_level = c - '0';
            break;
        case 'f':
            opts.force_extract = 1;
            break;
        case 'g':
            opts.generic = 1;
            opts.header_level = 0;
            break;
        case 'o':
            /* compress method */
            {
                int idx = 1;    /* -o means -lh1- method */

                if (optarg)
                    idx = *optarg - '0'; /* -lh[567]- method */

                opts.method   = &methods[idx];
            }
            break;
        case 'q':
            /* quiet mode */
            opts.quiet = 2;     /* -q is equivalent to -q2 */
            if (optarg)
                opts.quiet = *optarg - '0';
            break;
        case 'v':
            /* verbose mode */
            opts.verbose = 1;
            break;

        case 'w':
            /* extract directory */
            if (!optarg)
                error("extract directory does not specified for `-w'");
            if (*optarg == '=')
                optarg++;

            opts.outdir = optarg;
            break;
        case 'z':               /* no compress */
            opts.nocompress = 1;
            break;
        case LHA_OPT_HELP:
            print_usage();
            break;
        case LHA_OPT_VERSION:
            print_version();
            break;
        default:
            break;
        }
    }
}

FILE *
open_tempfile()
{
    FILE *outfile;

    outfile = tmpfile();
    if (outfile == NULL)
        error("Can't open temporary file");

    return outfile;
}

static void
op_add(int cmd, char *archive_file, int argc, char **argv)
{
    int i, nfiles, found;
    struct lzh_header h;
    struct lzh_ostream w, *wp;
    FILE *arcfile, *outfile;

    wp = &w;

    nfiles = 0;

    if (*argv == 0)
        error("archived files are not specified.");

    wp->fp = outfile = open_tempfile();
    wp->buf = 0;

    arcfile = fopen(archive_file, "rb");
    if (arcfile == NULL)
        error("Can't open archive '%s'", archive_file);

    while (read_header(arcfile, &h)) {

        found = search(argc, argv, h.filename);
        if (found <= 0) {
            copy(arcfile, outfile, &h);
            continue;
        }

        /* found */
        argv[found-1] = 0;

        if (cmd == 'u') {
            time_t mtime;

            if (file_mtime(h.filename, &mtime) == -1 || h.mtime > mtime) {
                /* no update */
                copy(arcfile, outfile, &h);
                continue;
            }
        }

        /* replace */
        add(wp, 1, h.filename, h.namelen);
        skip(arcfile, &h);
        nfiles++;
    }

    /* add non replacement files */
    for (i = 0; i < argc; i++) {
        if (argv[i]) {
            nfiles++;
            add(wp, 0, argv[i], strlen(argv[i]));
        }
    }

    if (opts.quiet < 2)
        printf("  %d files\n", nfiles);

    if (nfiles > 0) {
        if (fputc(0, outfile) == EOF)      /* end of archive */
            error("Can't write");

        fclose(arcfile);
        unlink(archive_file);

        rewind(outfile);
        if (copy_stream_to_file(outfile, archive_file) == -1)
            error("fail to copy_stream_to_file(): temp -> %s",archive_file);
    }

    fclose(outfile);
}

static void
op_create_to_stdout(int cmd, char *archive_file, int argc, char **argv)
{
    int i;
    struct lzh_ostream w, *wp;
    FILE *outfile;

    wp = &w;

    wp->buf = 0;
    if (*argv == 0)
        error("archived files are not specified.");

    for (i = 0; i < argc; i++) {
        wp->fp = outfile = open_tempfile();

        add(wp, 0, argv[i], strlen(argv[i]));

        rewind(outfile);
        if (copy_stream(outfile, stdout) == -1)
            error("fail to copy_stream(): temp -> %s","stdout");
        fclose(outfile);
    }

    if (fputc(0, stdout) == EOF)      /* end of archive */
        error("Can't write");
}

static void
op_create(int cmd, char *archive_file, int argc, char **argv)
{
    int i;
    struct lzh_ostream w, *wp;
    FILE *outfile;

    wp = &w;

    wp->fp = outfile = open_tempfile();
    wp->buf = 0;
    if (*argv == 0)
        error("archived files are not specified.");

    for (i = 0; i < argc; i++) {
        add(wp, 0, argv[i], strlen(argv[i]));
    }

    if (fputc(0, outfile) == EOF)      /* end of archive */
        error("Can't write");

    rewind(outfile);
    if (copy_stream_to_file(outfile, archive_file) == -1)
        error("fail to copy_stream_to_file(): temp -> %s",archive_file);
    fclose(outfile);
}

static void
op_delete(int cmd, char *archive_file, int argc, char **argv)
{
    int nfiles, found;
    struct lzh_header h;
    int arc_nfiles;
    FILE *arcfile, *outfile;

    nfiles = 0;

    if (argc == 0) {
        message("No files given in argument, do nothing.");
        return;
    }
    outfile = open_tempfile();

    arcfile = fopen(archive_file, "rb");
    if (arcfile == NULL)
        error("Can't open archive '%s'", archive_file);

    arc_nfiles = 0;

    while (read_header(arcfile, &h)) {

        found = search(argc, argv, h.filename);
        if (found > 0) {
            nfiles++;
            message("'%s' deleted", h.filename);
            skip(arcfile, &h);
        }
        else {
            arc_nfiles++;
            copy(arcfile, outfile, &h);
        }
    }

    if (opts.quiet < 2)
        printf("  %d files\n", nfiles);

    if (nfiles > 0) {
        if (fputc(0, outfile) == EOF)      /* end of archive */
            error("Can't write");

        fclose(arcfile);
        unlink(archive_file);

        if (arc_nfiles > 0) {
            rewind(outfile);
            if (copy_stream_to_file(outfile, archive_file) == -1)
                error("fail to copy_stream_to_file(): temp -> %s",archive_file);
        }
        else {
            message("The archive file \"%s\" was removed because it would be empty.", archive_file);
        }
    }

    fclose(outfile);
}

static void
op_extract(int cmd, char *archive_file, int argc, char **argv)
{
    int nfiles, found;
    struct lzh_header h;
    struct lzh_istream r, *rp;
    FILE *arcfile;

    rp = &r;

    nfiles = 0;

    if (opts.archive_to_stdio)
        arcfile = stdin;
    else
        arcfile = fopen(archive_file, "rb");

    if (arcfile == NULL)
        error("Can't open archive '%s'", archive_file);

    while (read_header(arcfile, &h)) {

        found = search(argc, argv, h.filename);
        if (found == 0) {
            skip(arcfile, &h);
            break;
        }

        /* found or no arg */

        opts.method = which_method(h.method);
        if (opts.method == NULL) {
            fprintf(stderr, "Unknown method: %.5s\n", h.method);
            skip(arcfile, &h);
            break;
        }

        rp->fp = arcfile;
        rp->compsize = h.compsize;

        if (cmd == 'x')
            extract_to_file(rp, &h);
        else
            extract_to_stdout(rp, &h);
        nfiles++;

        if (!opts.has_wildcard && nfiles == argc)
            break;
    }

    if (cmd != 'p') {
        if (opts.quiet < 2)
            printf("  %d files\n", nfiles);
    }
}

static void
op_list(int cmd, char *archive_file, int argc, char **argv)
{
    int nfiles, found;
    struct lzh_header h;
    FILE *arcfile;

    nfiles = 0;

    if (opts.archive_to_stdio)
        arcfile = stdin;
    else
        arcfile = fopen(archive_file, "rb");

    if (arcfile == NULL)
        error("Can't open archive '%s'", archive_file);

    while (read_header(arcfile, &h)) {

        found = search(argc, argv, h.filename);
        if (found == 0) {
            skip(arcfile, &h);
            break;
        }

        /* found or no arg */
        if (nfiles == 0)
            list_start();
        list(&h);

        nfiles++;
#if 0
        if (nfiles == argc)
            break;
#endif
        skip(arcfile, &h);
    }

    if (opts.quiet < 2)
        printf("  %d files\n", nfiles);
}

int
main(int argc, char *argv[])
{
    int cmd;
    char *archive_file;

    INITIALIZE_OPTS(opts);

    if (argv[1] == 0)
        print_usage();

    /*take a command character */
    {
        char *arg1;

        arg1 = argv[1];
        if (arg1[0] == '-')
            arg1++;
        if (arg1[0] == 0)
            print_usage();

        cmd = *arg1;
        if (arg1[1] == 0) {
            /* -<cmd> -<opts> ... */
            argv++;
            argc--;
        }
        else {
            /* -<cmd><opts> => -<opts> */
            *arg1 = '-';
        }
    }

    parse_args(argc, argv);
    argv += optind;
    argc -= optind;

    archive_file = argv[0];

    if (strcmp(archive_file, "-") == 0)
        opts.archive_to_stdio = 1;
    if (opts.archive_to_stdio)
        opts.quiet = 2;

    argv++;
    argc--;

    make_crctable();

    {
        int i;
        for (i = 0; i < argc; i++) {
            if (strcspn(argv[i], "*?") > 0) {
                opts.has_wildcard = 1;
            }
        }
    }

    switch (cmd) {
    case 'a':
    case 'u':
        if (opts.archive_to_stdio)
            op_create_to_stdout(cmd, archive_file, argc, argv);
        else if (!file_exists(archive_file))
            op_create(cmd, archive_file, argc, argv);
        else
            op_add(cmd, archive_file, argc, argv);
        break;

    case 'c':
        if (opts.archive_to_stdio)
            op_create_to_stdout(cmd, archive_file, argc, argv);
        else
            op_create(cmd, archive_file, argc, argv);
        break;

    case 'd':
        if (opts.archive_to_stdio)
            error("should be specified the archive file");
        op_delete(cmd, archive_file, argc, argv);
        break;

    case 'x':
    case 'p':
        op_extract(cmd, archive_file, argc, argv);
        break;

    case 'l':
    case 'v':
        op_list(cmd, archive_file, argc, argv);
        break;

    default:
        print_usage();
        break;
    }
    return EXIT_SUCCESS;
}
