/*
  Copyright (c) 2002 Koji Arai

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation files
  (the "Software"), to deal in the Software without restriction,
  including without limitation the rights to use, copy, modify, merge,
  publish, distribute, sublicense, and/or sell copies of the Software,
  and to permit persons to whom the Software is furnished to do so,
  subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
  BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
  ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
*/

#include <string.h>
#define HAVE_BASENAME 0
#define HAVE_DIRNAME 1

int
path_addsep(char *path, size_t size)
{
    int len = strlen(path);

    if (len > 0 && path[len-1] != '/' && len + 1 < size) {
        path[len++] = '/';
        path[len] = 0;
    }

    return len;
}

#if !HAVE_BASENAME
char *
xbasename(char *path)
{
    char *p1, *p2;
    int len;

    len = strlen(path);
    if (len == 0)
        return path;
    if (len == 1 && path[0] == '/')
        return path;
    if (path[len-1] == '/')
        path[len-1] = '\0';

    for (p1 = p2 = path; *p1 != '\0'; p1++) {
        if (*p1 == '/') {
            p2 = p1 + 1;
        }
    }

    return p2;
}
#endif /* !HAVE_BASENAME */

#if !HAVE_DIRNAME
char *
dirname(char *path)
{
    char *p1, *p2;
    int len;
    static char current[] = ".";

    len = strlen(path);
    if (len == 0)
        return current;
    if (len == 1 && path[0] == '/')
        return path;
    if (path[len-1] == '/')
        path[len-1] = '\0';

    for (p1 = p2 = path; *p1 != '\0'; p1++) {
        if (*p1 == '/') {
            p2 = p1;
        }
    }
    if (*p2 == '/' && p2 == path)
        p2++;
    *p2 = '\0';

    if (path[0] == '\0')
        return current;

    return path;
}
#endif /* !HAVE_DIRNAME */

int
makepath(char *dest, int dest_size,
         char *dir, char *file, char *ext)
{
    int len;
    char *p;
    int size;

    p = dest;
    size = dest_size;

    /* copy directory part */
    if (dir) {
        len = string_copy(p, dir, size);
        size -= len;

        if (len > 0 && size > 0) {
            if (p[len-1] != '/') {
                p[len++] = '/';
                size--;
            }
        }
        p += len;
    }

    /* copy filename part */
    if (file) {
        len = string_copy(p, file, size);
        size -= len;
        p += len;
    }

    /* copy suffix part */
    if (ext) {
        if (ext[0] != '.' && size > 0) {
            *(p++) = '.';
            size--;
        }

        len = string_copy(p, ext, size);
        size -= len;
        p += len;
    }

    return p - dest;   /* result string length */
}

#ifdef DEBUG

#include <stdio.h>
#define basename glibc2_basename
#include <string.h>             /* strdup() */
#undef basename

#include <assert.h>

split_path(char *path)
{
    printf("\"%s\"\t= \"%s\" \"%s\"\n",
           path,
           dirname(strdup(path)),
           basename(strdup(path)));
}

main()
{
    char work[256];
    int len;

    split_path("/foo/bar");
    split_path("/bar");
    split_path("bar");
    split_path("bar/");         /* should print "." and "bar" */
    split_path("/");            /* should print "/" and "/" */
    split_path("");             /* should print "." and "" */


    len = makepath(work, sizeof work, "foo", "bar", "baz");
    assert(strcmp(work, "foo/bar.baz") == 0 && len == 11);
    printf("\"%s\"\n", work);

    len = makepath(work, sizeof work, "foo/", "bar", ".baz");
    assert(strcmp(work, "foo/bar.baz") == 0 && len == 11);
    printf("\"%s\"\n", work);

    len = makepath(work, strlen("foo/bar.baz")+1, "foo", "bar", "baz");
    assert(strcmp(work, "foo/bar.baz") == 0 && len == 11);
    printf("\"%s\"\n", work);

    len = makepath(work, strlen("foo/bar.baz")+1, "foo", "bar.baz", NULL);
    assert(strcmp(work, "foo/bar.baz") == 0 && len == 11);
    printf("\"%s\"\n", work);

    memset(work, '*', sizeof(work)-1);
    work[sizeof(work)-1] = 0;
    len = makepath(work+10, sizeof(work)-10, "foo", "bar", "baz");
    assert(strcmp(work+10, "foo/bar.baz") == 0 && len == 11);
    printf("\"%s\"\n", work);
    memset(work+10, '*', strlen("foo/bar.baz")+1);
    {
        int i;
        for (i = 0; i < sizeof(work)-1; i++)
            assert(work[i] == '*');
    }
    printf("\"%s\"\n", work);

    len = makepath(work, strlen("foo/bar.baz"), "foo", "bar", "baz");
    assert(strcmp(work, "foo/bar.ba") == 0 && len == 10);
    printf("\"%s\"\n", work);

    return 0;
}
#endif /* DEBUG */
