/**
 * @file task_bugshell.c
 * @author Shinichiro Nakamura
 * @brief Bug Shellタスクの実装。
 */

/*
 * ===============================================================
 *  Bug Shell Task
 * ===============================================================
 * Copyright (c) 2010-2011 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include <kernel.h>
#include <target_syssvc.h>
#include <syssvc/serial.h>
#include "task_bugshell.h"
#include "bugshell_usrcmd.h"
#include "bugshell_config.h"

/**
 * @brief 1行取得する。
 *
 * @param buf バッファ。
 * @param siz サイズ。
 */
void get_line(char *buf, int siz)
{
    int i = 0;
    char_t c;

    do {
        /*
         * 1文字読み込む。
         */
        serial_rea_dat(BUGSHELL_PORTID, &c, 1);

        /*
         * エコーバックする。
         */
        serial_wri_dat(BUGSHELL_PORTID, &c, 1);

        /*
         * 改行文字か、最大長に達したら終端に差し替える。
         */
        if ((c == '\r') || (c == '\n') || (siz <= (i + 1))) {
            c = '\0';
        }

        /*
         * 文字をバッファへ格納する。
         */
        buf[i++] = c;
    } while (c != '\0');
}

/**
 * @brief Bug Shell Taskの本体。
 * @param exinf オプション。
 */
void task_bugshell(intptr_t exinf)
{
    /*
     * シリアルポートをオープンする。
     */
    serial_opn_por(BUGSHELL_PORTID);

    /*
     * 制御コードが端末からシステムに入力された時、
     * そのままエコーバックすると都合が悪い。
     * そこでエコーバックのオプションは外す。
     */
    serial_ctl_por(BUGSHELL_PORTID,
            (IOCTL_CRLF | IOCTL_FCSND | IOCTL_FCRCV));

    char buf[32];

    while (1) {
        /*
         * プロンプトを表示する。
         */
        serial_wri_dat(BUGSHELL_PORTID, (char_t *)">", 1);

        /*
         * 1行取得する。
         */
        get_line(buf, sizeof(buf));

        /*
         * 改行を送信する。
         */
        serial_wri_dat(BUGSHELL_PORTID, (char_t *)"\r\n", 2);

        /*
         * 公正を保つためにコマンド層の実装はNT-Shellのものを用いる。
         * 要するに機能的には同じで、入力インターフェースのみ異なる
         * という状態を作る。
         */
        bugshell_usrcmd_execute(buf);
    }
}

