/**
 * @file task_colorled.c
 * @author Shinichiro Nakamura
 * @brief カラーLEDタスクの実装。
 */

/*
 * ===============================================================
 *  Color LED Task
 * ===============================================================
 * Copyright (c) 2010-2011 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include <kernel.h>
#include <target_syssvc.h>
#include <syssvc/serial.h>
#include "task_colorled.h"
#include "kernel_cfg.h"

#define COLOR_LED_R_BIT     (1 << 3)
#define COLOR_LED_G_BIT     (1 << 4)
#define COLOR_LED_B_BIT     (1 << 5)
#define COLOR_LED_FIODIR    (LPC_GPIO2->FIODIR)
#define COLOR_LED_FIOPIN    (LPC_GPIO2->FIOPIN)
#define COLOR_LED_FIOCLR    (LPC_GPIO2->FIOCLR)
#define COLOR_LED_FIOSET    (LPC_GPIO2->FIOSET)

#define USE_LED_R() do { COLOR_LED_FIODIR |= COLOR_LED_R_BIT; } while(0)
#define USE_LED_G() do { COLOR_LED_FIODIR |= COLOR_LED_G_BIT; } while(0)
#define USE_LED_B() do { COLOR_LED_FIODIR |= COLOR_LED_B_BIT; } while(0)

#define TOGGLE_R()  do { COLOR_LED_FIOPIN ^= COLOR_LED_R_BIT; } while(0)
#define TOGGLE_G()  do { COLOR_LED_FIOPIN ^= COLOR_LED_G_BIT; } while(0)
#define TOGGLE_B()  do { COLOR_LED_FIOPIN ^= COLOR_LED_B_BIT; } while(0)

#define LED_ON_R()  do { COLOR_LED_FIOCLR = COLOR_LED_R_BIT; } while(0)
#define LED_ON_G()  do { COLOR_LED_FIOCLR = COLOR_LED_G_BIT; } while(0)
#define LED_ON_B()  do { COLOR_LED_FIOCLR = COLOR_LED_B_BIT; } while(0)

#define LED_OFF_R()  do { COLOR_LED_FIOSET = COLOR_LED_R_BIT; } while(0)
#define LED_OFF_G()  do { COLOR_LED_FIOSET = COLOR_LED_G_BIT; } while(0)
#define LED_OFF_B()  do { COLOR_LED_FIOSET = COLOR_LED_B_BIT; } while(0)

/**
 * @brief ポートを初期化する。
 */
static void init_port(void)
{
    /*
     * 予め出力をオフにする。
     */
    LED_OFF_R();
    LED_OFF_G();
    LED_OFF_B();

    /*
     * ポートを出力に設定する。
     */
    USE_LED_R();
    USE_LED_G();
    USE_LED_B();
}

/**
 * @brief スプラッシュを実行する。
 */
static void splash(void)
{
    /*
     * R, G, Bと順番に点灯させて起動を演出する。
     */
    LED_ON_R(); tslp_tsk(500); LED_OFF_R();
    LED_ON_G(); tslp_tsk(500); LED_OFF_G();
    LED_ON_B(); tslp_tsk(500); LED_OFF_B();
}

/**
 * @brief カラーLEDタスクの本体。
 * @param exinf オプション。
 */
void task_colorled(intptr_t exinf)
{
    /*
     * ポートを初期化する。
     */
    init_port();

    /*
     * スプラッシュを実行する。
     */
    splash();

    /*
     * リクエスト待ちループ。
     */
    while (1) {
        uint16_t msg;
        if (rcv_dtq(DTQ_COLORLED, (intptr_t)&msg) == E_OK) {
            if (msg & COLOR_R) { LED_ON_R(); } else { LED_OFF_R(); }
            if (msg & COLOR_G) { LED_ON_G(); } else { LED_OFF_G(); }
            if (msg & COLOR_B) { LED_ON_B(); } else { LED_OFF_B(); }
        }
    }
}

/**
 * @brief カラーLEDタスクに色の変更をリクエストする。
 *
 * @param color 色。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
int tskapi_colorled_set_color(int color)
{
    snd_dtq(DTQ_COLORLED, (intptr_t)color);
    return 0;
}

