/**
 * @file task_led.c
 * @author Shinichiro Nakamura
 * @brief LEDタスクの実装。
 */

/*
 * ===============================================================
 *  LPCXpresso Clock
 * ===============================================================
 * Copyright (c) 2010-2011 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include <kernel.h>
#include <t_syslog.h>
#include <t_stdlib.h>
#include "kernel_cfg.h"
#include "lpc17xx_gpio.h"

#define MS_MIN  (100)   /**< 最小点灯間隔。 */
#define MS_MAX  (1000)  /**< 最大点灯間隔。 */

/**
 * LPCXpresso 1768/1769では、GPIO0のbit22がLEDに接続されている
 */
#define LEDMASK	(1<<22)

/**
 * @brief LED task.
 */
void task_led(intptr_t exinf)
{
    uint16_t msg;

    syslog(LOG_NOTICE, "LED task initialized.");
    int ms = MS_MIN + ((MS_MAX - MS_MIN) / 2);

    /* GPIO0のLED制御ピンを出力にする */
    GPIO_SetDir(
            0,          // GPIOポート番号
            LEDMASK,    // 操作するビットマスク
            1           // 設定する値
            );

    while(1)
    {
        /*
         * データキューをポーリングする。
         * データがあれば点灯間隔の変更要求と見なして処理する。
         */
        if (prcv_dtq(DTQ_LED, (intptr_t *)&msg) == E_OK ) {
            ms = msg;
            if (ms < MS_MIN) {
                ms = MS_MIN;
            }
            if (MS_MAX < ms) {
                ms = MS_MAX;
            }
        }

        /*
         * LED制御ピンをLにする
         */
        GPIO_ClearValue(
                0,          // GPIOポート番号
                LEDMASK     // 操作するビットマスク
                );
        tslp_tsk(ms);

        /*
         * LED制御ピンをHにする
         */
        GPIO_SetValue(
                0,          // GPIOポート番号
                LEDMASK     // 操作するビットマスク
                );
        tslp_tsk(ms);
    }

    syslog(LOG_NOTICE, "LED task terminated.");
    ext_ker();
    assert(0);
}

