/**
 * @file task_ntl.c
 * @author Shinichiro Nakamura
 * @brief NTLタスクの実装。
 */

/*
 * ===============================================================
 *  LPCXpresso Clock
 * ===============================================================
 * Copyright (c) 2010-2011 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include <kernel.h>
#include <t_syslog.h>
#include <t_stdlib.h>
#include <itron.h>
#include "kernel_cfg.h"
#include "task_ntl.h"
#include "ntl.h"

/**
 * @brief NTLタスク。
 */
void task_ntl(intptr_t exinf)
{
    /*
     * 開始処理。
     */
    syslog(LOG_NOTICE, "NTL task initialized.");
    ntl_open();

    /*
     * タスクループ。
     */
    ntl_msg_t *p;
    while(1) {
        /*
         * メールボックスへの着信を待つ。
         */
        if (rcv_mbx(MBX_NTL, (T_MSG **)&p) == E_OK) {
            ntl_evtsrc_t evtsrc = ((ntl_msg_t *)p)->evtsrc;
            ntl_evtdesc_t evtdesc = ((ntl_msg_t *)p)->evtdesc;
            ntl_log(evtsrc, evtdesc);
            rel_mpf(MPF_NTL, (VP)p);
        }
    }

    /*
     * 終了処理。
     */
    ntl_close();
    syslog(LOG_NOTICE, "NTL task terminated.");
    ext_ker();
    assert(0);
}

void tskapi_ntl_log(ntl_evtsrc_t evtsrc, ntl_evtdesc_t evtdesc)
{
    VP vp;

    /*
     * メモリプールからバッファを取得する。
     */
    get_mpf(MPF_NTL, &vp);

    /*
     * パラメータを設定する。
     */
    ((ntl_msg_t *)vp)->evtsrc = evtsrc;
    ((ntl_msg_t *)vp)->evtdesc = evtdesc;

    /*
     * メールボックスに送信する。
     */
    snd_mbx(MBX_NTL, vp);
}

