/**
 * @file task_ntshell.c
 * @author Shinichiro Nakamura
 * @brief NT-Shellタスクの実装。
 */

/*
 * ===============================================================
 *  Natural Tiny Shell Task
 * ===============================================================
 * Copyright (c) 2010-2011 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include <kernel.h>
#include <target_syssvc.h>
#include <syssvc/serial.h>
#include "ntshell.h"
#include "ntshell_usrcmd.h"
#include "ntshell_config.h"

static int func_ntshell_read(char *buf, int cnt);
static int func_ntshell_write(const char *buf, int cnt);
static int func_ntshell_callback(const char *text, void *extobj);

/**
 * @brief NT-Shellライブラリ用のリード関数。
 * @details この関数は指定した読み込みバイト数に達するまでブロックする。
 *
 * @param buf バッファ。
 * @param cnt 読み込みバイト数。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
static int func_ntshell_read(char *buf, int cnt)
{
    return serial_rea_dat(SHELL_PORTID, (char_t *)buf, cnt);
}

/**
 * @brief NT-Shellライブラリ用のライト関数。
 * @details この関数は指定した書き込みバイト数に達するまでブロックする。
 *
 * @param buf バッファ。
 * @param cnt 書き込みバイト数。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
static int func_ntshell_write(const char *buf, int cnt)
{
    return serial_wri_dat(SHELL_PORTID, (const char_t *)buf, cnt);
}

/**
 * @brief NT-Shellライブラリのコールバック関数。
 * @details
 *
 * @param text ユーザから入力された文字列。
 * @param extobj 拡張オブジェクト。
 *
 * @retval 0 成功。
 * @retval !0 失敗。
 */
static int func_ntshell_callback(const char *text, void *extobj)
{
    /*
     * ユーザコマンドを実行する。
     */
    return ntshell_usrcmd_execute(text);
}

/**
 * @brief NT-Shell Taskの本体。
 * @param exinf オプション。
 */
void task_ntshell(intptr_t exinf)
{
    ntshell_t ntshell;

    /*
     * シリアルポートをオープンする。
     */
    serial_opn_por(SHELL_PORTID);

    /*
     * 制御コードが端末からシステムに入力された時、
     * そのままエコーバックすると都合が悪い。
     * そこでエコーバックのオプションは外す。
     */
    serial_ctl_por(SHELL_PORTID, (IOCTL_CRLF | IOCTL_FCSND | IOCTL_FCRCV));

    /*
     * NT-Shellライブラリを実行する。
     * この関数は、処理を返さない。
     */
    ntshell_execute(
            &ntshell,
            func_ntshell_read,
            func_ntshell_write,
            func_ntshell_callback,
            NULL);

    /*
     * ここに到達することはない。
     */
    ext_tsk();
}

