# coding: UTF-8

# = HTML 用のメール解析器
#
# 最初の著者:: トゥイー
# リポジトリ情報:: $Id: m2w_mail_parser_html_text_type.rb 784 2012-07-31 12:17:17Z toy_dev $
# 著作権:: Copyright (C) Ownway.info, 2011. All rights reserved.
# ライセンス:: CPL(Common Public Licence)
#
# 本クラスは、以下を前提とする。
#
# == 本文とは？
#
# 以下の全ての条件を満たすものを指す。
#
# * body タグがある場合は、その body タグ内、かつ。
# * 単一 table タグの単一 tr, td タグ要素からなる場合は、その td タグ内、かつ。
# * 外郭の div タグに入れ子が無いもの。
#
# == 行とは？
#
# 行の区切りなるのは div, p, br のタグであり、以下のようなものを指す。
#
# * (<div>)*?.*?</div>
# * <p>.*?</p>
# * .*?<br>
# * .*?<br />
# * .*?<p>
# * .*?<p />
#
# div, p, br 各種タグの属性の有無は問わない。また、大文字・小文字の違いも問わない。
#
# 古い HTML では慣習として br, p タグを閉じずに単一で改行として利用する場合があるのを考慮し、行を以上の通り定義する。
#
# == 空行とは？
#
# 空行は、以下の全ての条件を満たすものを指す。
#
# * (<div>空文字列*)*</div>
# * <p>空文字列*</p>
# * 空文字列*<br>
# * 空文字列*<br />
# * 空文字列*<p>
# * 空文字列*<p />
class Mail2WeblogMailParserHtmlTextType

	HTML_SPACE_REGEX = '(?:\s|&nbsp;)'
	HTML_SPACES_REGEX = '(?:\s|&nbsp;)*?'

	# HTML メールを受け取り、その本文の行を配列として返します。
	def split(content)
		M2W_LOGGER.debug("Start  #{self.class}#split ... content = #{content}")

		content = content.gsub(/\r|\n/, '')

		# body がある場合は中身を取り出す
		@bgcolor = nil
		if %r!<body([^>]*?)>(.+)</body>!i =~ content then
			attrs = $1
			content = $2
			if attrs =~ /bgcolor\s*=\s*['"](.+?)['"]/ then
				@bgcolor = $1
			end
		end

		# 単行・単列のテーブルから中身を取り出す
		if %r!^#{HTML_SPACE_REGEX}*<table[^>]*?>.*?<tr[^>]*?><td[^>]*?>(.+?)</td>(.*?)</tr>(.*?)</table>#{HTML_SPACE_REGEX}*! =~ content then
			temp1 = $1
			temp2 = $2
			temp3 = $3
			if temp2 !~ /<(td|tr)/ && temp3 !~ /<(td|tr)/ then
				content = temp1
			end
		end

		# 外郭の div タグの入れ子を取り出す
		content = __balance_div_tags(__delete_suffix_tags(__delete_prefix_tags(content)))

		temp = []
		while true
			# </div>, </p>, <br>, <br />, <p /> で行を区切る
			(left, sep, right) = content.partition(%r!</(div|p)>|<(br|p)[^>]*?/?>!i)

			if sep.length == 0 then
				temp.push(__clean_html_line(left))
				break
			else
				line = __clean_html_line(left)
				line << sep

				# <p>...</p> を考慮する
				if sep =~ /<p/ then
					(left4p, sep4p, right4p) = right.partition(%r!</p>!i)
					if sep4p.length != 0 then
						line << __clean_html_line(left4p)
						line << sep4p
						right = right4p
					end
				end

				temp.push(line)
				content = right
			end
		end

		result = []
		temp.each do |line|
			if line.length > 0 then
				result.push(line)
			end
		end

		M2W_LOGGER.debug("Finish #{self.class}#split ... result = #{result.to_s}")
		return result
	end

	# 空行であることの判断をします。空行の場合は true を、それ以外の場合は false を返します。
	#
	# このメソッドの入力は、前提として「行」の要件を既に満たしていることが前提です。
	def is_space_line(line)
		if %r!^(<div[^>]*?>#{HTML_SPACES_REGEX})*</div>$!i =~ line then
			return true
		elsif %r!^<p[^>]*?>#{HTML_SPACES_REGEX}</p>$!i =~ line then
			return true
		elsif %r!^#{HTML_SPACES_REGEX}<(br|p)[^>]*/?>$!i =~ line then
			return true
		end

		return false
	end

	def parse_header(line, subject_separator)
		if %r!^<(div|p)[^>]*?>([0-9a-zA-Z_]+?)#{subject_separator}(.*?)</(div|p)>$!i =~ line then
			return [true, $2, $3]
		elsif %r!^([0-9a-zA-Z_]+?)#{subject_separator}(.*?)<(br|p)[^>]*/?>!i =~ line then
			return [true, $1, $2]
		elsif %r!^<(div|p)[^>]*?>([0-9a-zA-Z_]+?)#{subject_separator}(.*?)<(br|p)[^>]*/?>!i =~ line then
			return [true, $2, $3]
		else
			return [false, nil, nil]
		end
	end

	def parse_format_plugin_header(key)
		if /^#{M2W_FORMAT_PLUGIN_CONF_HEADER_PREFIX}([0-9a-zA-Z_]+)$/ =~ key then
			return [true, $1]
		else
			return [false, nil]
		end
	end

	def parse_subject_separation(line, subject_separator)
		if %r!^<(div|p)[^>]*?>([0-9a-zA-Z_]+?)(#{subject_separator})\3{3}</\1>$!i =~ line then
			return [true, $2]
		elsif %r!^([0-9a-zA-Z_]+?)(#{subject_separator})\2{3}<(br|p)[^>]*/?>!i =~ line then
			return [true, $1]
		else
			return [false, nil]
		end
	end

	def arrange_content(content)
		M2W_LOGGER.debug("Start  #{self.class}#arrange_content ... content = #{content}")

		result = __delete_suffix_tags(__delete_prefix_tags(content))

		buffers = result.split(/\n/)

		while buffers.length > 0 && is_space_line(buffers[0])
			buffers.shift
		end

		while buffers.length > 0 && is_space_line(buffers[buffers.length - 1])
			buffers.pop
		end

		if @bgcolor then
			buffers.unshift("<div style='background-color: #{@bgcolor}'>")
			buffers.push("</div>")
		end

		result = ""
		buffers.each do |line|
			result << line
		end

		result = __balance_div_tags(result).chomp

		M2W_LOGGER.debug("Finish #{self.class}#arrange_content ... result = #{result}")
		return result
	end

	# 前後のスペースを排除します。
	def __clean_html_line(line)
		if /^#{HTML_SPACES_REGEX}([^\s]*)#{HTML_SPACES_REGEX}$/ =~ line then
			line = $1
		end

		# タグから始まらず、一番最後に <div> があった場合は削除する
		# ※ 非常にレアなケースで、メール先頭がタグから始まらず、最初の改行に <div> タグが使われる場合の特殊処理（2012/05/11 時点 Yahoo! メールで発生する問題）
		if /^([^<].*?)<div>$/ =~ line then
			line = $1
		end

		return line
	end

	def __delete_prefix_tags(content)
		while %r!^\s*?<div[^>]*?>\s*?<div[^>]*?>(.+?)$!i =~ content
			content = "<div>#{$1}"
		end

		return content
	end

	def __delete_suffix_tags(content)
		while (pos = content.rindex(%r!</div>(\s|&nbsp;)*?</div>(\s|&nbsp;)*?!mi)) != nil
			if content.slice(pos, content.length - pos) =~ %r!</div>(\s|&nbsp;)*?</div>(.+)!mi then
				right = $2
				right.gsub!(/(\r|\n)/, "")
				if right !~ %r!^(\s|&nbsp;)*$!i then
					break
				end
			end
			content = content.slice(0, pos) + "</div>"
		end

		return content
	end

	def __count_pattern_matched(content, pattern)
		result = 0

		pos = 0
		while (pos = content.index(pattern, pos)) != nil
			pos = pos + 1
			result = result + 1
		end

		return result
	end

	def __balance_div_tags(content)
		M2W_LOGGER.debug("Start  __balance_div_tags")

		start_div_count = __count_pattern_matched(content, %r!<div!i)
		end_div_count = __count_pattern_matched(content, %r!</div!i)

		begin
			if start_div_count > end_div_count then
				return content + ("</div>" * (start_div_count - end_div_count))
			elsif start_div_count < end_div_count then
				return ("<div>" * (end_div_count - start_div_count)) + content
			else
				return content
			end
		ensure
			M2W_LOGGER.debug("Finish __balance_div_tags ... start_div_count = #{start_div_count}, end_div_count = #{end_div_count}")
		end
	end

end
