# coding: UTF-8

require 'mailutils/mime'
require 'net/http'
require 'lib/exception/m2w_error'
require 'lib/exception/m2w_warning'

module WeblogPostCommon

	def post_medias(client, username, password, blogid, media_posters, errors, warnings)
		@logger.debug("Start  #{self.class}#post_medias") if @logger && @logger.debug?

		mime = Mime.new

		media_posters.each do |media_poster|
			__post_media(media_poster, mime, client, username, password, blogid, media_posters, errors, warnings)
		end

		@logger.debug("Finish #{self.class}#post_medias") if @logger && @logger.debug?
	end

	def post_media(content, media_name, client, blogid, errors, warnings)
		raise WeblogPluginError.new(
			"未実装エラー",
			"実装されていません。")
	end

	def get_header_for_post(header)
		result = header.clone
		result.delete_if do |key, value|
			key == M2W_FORMAT_PLUGIN_CONF_HEADER_NAME || key =~ /^fmt_[0-9a-zA-Z]+$/
		end

		return result
	end

	def request_trackback(client, header, blogid, postid, description)
		@logger.debug("Start  #{self.class}#request_trackback ... header = #{header.keys}, blogid = #{blogid}, postid = #{postid}") if @logger && @logger.debug?

		if header['tb'] != nil then
			begin
				tb_urls = header['tb'].split(/[\s,]/)

				url = get_url4trackback(client, postid)
				blog_name = @parent.get_blog_name(client, blogid)

				title = header['title'] != nil ? header['title'] : ""
				excerpt = header['excerpt']
				if excerpt == nil then
					excerpt = ''
					# Ruby 1.9.0 より下のバージョンではマルチバイト文字列を綺麗に切れないため本文の要約に対応しない
					if RUBY_VERSION >= '1.9.0' then
						excerpt = description.length > 23 ? "#{description.slice(0, 23)} ..." : description
					end
				end

				tb_urls.each do |tb_url|
					@logger.debug("trackback to #{tb_url} ... title = #{title}, excerpt = #{excerpt}, url = #{url}, blog_name = #{blog_name}") if @logger && @logger.debug?
					response = Net::HTTP.post_form(
						URI.parse(tb_url),
						{
							'charset' => M2W_SYSTEM_INNER_ENCODING,
							'title' => title,
							'excerpt' => excerpt,
							'url' => url,
							'blog_name' => blog_name
						}
					)
					case response
					when Net::HTTPSuccess
						@logger.debug("trackback response = #{response.body}") if @logger && @logger.debug?
						if %r!<error>\s*([0-9]+)?\s*</error>! =~ response.body then
							error = $1
							if error != '0' then
								message = ''
								if %r!<message>(.+?)</message>! =~ response.body then
									message = $1
								end
								warnings.push(WeblogPluginWarning.new(
									"トラックバック送信エラー(送信先 = #{tb_url}, メッセージ = #{message})",
									"トラックバックの送信に失敗しました(送信先 = #{tb_url}, メッセージ = #{message})。"))
							end
						end
					else
						warnings.push(WeblogPluginWarning.new(
							"トラックバック送信エラー(送信先 = #{tb_url})",
							"トラックバックの送信に失敗しました(送信先 = #{tb_url}"))
					end
				end
			rescue => e
				@logger.error("トラックバックの転送に失敗しました。") if @logger
				@logger.error(e) if @logger
				warnings.push(WeblogPluginWarning.new(
					"トラックバック転送エラー(メッセージ = #{e.message})",
					"トラックバックの転送に失敗しました(メッセージ = #{e.message})。"))
			end

			@logger.debug("Finish #{self.class}#request_trackback") if @logger && @logger.debug?
		end
	end

	def get_url4trackback(client, postid)
		get_plugin = @parent.get_get_plugin()
		response = get_plugin.get(Mail2WeblogRequest.new(:header => {'postid' => postid}))
		if response[:link] != nil then
			return response[:link]
		else
			raise WeblogPluginWarning.new(
				"リンク取得エラー",
				"リンクの取得に失敗しました。"
			)
		end
	end

	def get_blog_name(client, blogid)
	end

	def __post_media(media_poster, mime, client, username, password, blogid, media_posters, errors, warnings)
		media_url = nil
		begin
			media_url = post_media(media_poster.content, media_poster.media_name, client, blogid, errors, warnings)
			@logger.debug("post media ... media_url ... #{media_url}") if @logger && @logger.debug?
			if media_url == nil then
				errors.push(WeblogPluginError.new(
					"ファイル転送エラー(転送ファイル名 = #{media_poster.media_name})",
					"ファイル転送に失敗しました(転送ファイル名 = #{media_poster.media_name})。"))
			end
		rescue => e
			@logger.error("ファイルの転送に失敗しました。") if @logger
			@logger.error(e) if @logger
			errors.push(WeblogPluginError.new(
					"ファイル転送エラー(転送ファイル名 = #{media_poster.media_name})",
					"ファイル転送に失敗しました(転送ファイル名 = #{media_poster.media_name})。"))
		end

		# サムネイル画像を投稿する
		thumbnail_url = nil
		if M2W_SYSTEM_RMAGICK_SUPPORTED && media_url != nil && %r!^image/! =~ mime.mime(media_poster.media_name) then
			begin
				thumbnail = __create_thumbnail(media_poster.content, media_poster.thumbnail_setting)
			rescue => e
				@logger.error("サムネイルの作成に失敗しました。") if @logger
				@logger.error(e) if @logger
				errors.push(WeblogPluginError.new(
					"サムネイル作成エラー(ファイル名 = #{media_poster.thumbnail_name})",
					"サムネイルの作成に失敗しました(ファイル名 = #{media_poster.thumbnail_name})。"))
			end

			if thumbnail != nil then
				@logger.debug("create thumbnail ... thumbnail_name = #{media_poster.thumbnail_name}") if @logger && @logger.debug?

				begin
					thumbnail.format = "JPEG"
					thumbnail_url = post_media(thumbnail.to_blob, media_poster.thumbnail_name, client, blogid, errors, warnings)
					@logger.debug("post thumbnail ... thumbnail_url ... #{thumbnail_url}") if @logger && @logger.debug?
					if thumbnail_url == nil then
						errors.push(WeblogPluginError.new(
							"サムネイル転送エラー(ファイル名 = #{media_poster.thumbnail_name})",
							"サムネイルの転送に失敗しました(ファイル名 = #{media_poster.thumbnail_name})。"))
					end
				rescue => e
					@logger.error("ファイルの転送に失敗しました。") if @logger
					@logger.error(e) if @logger
					errors.push(WeblogPluginError.new(
						"サムネイル転送エラー(ファイル名 = #{media_poster.thumbnail_name})",
						"サムネイルの転送に失敗しました(ファイル名 = #{media_poster.thumbnail_name})。"))
				end
			end
		end

		media_poster.finish(media_url, thumbnail_url)
	end

	def __create_thumbnail(content, thumbnail_setting)
		@logger.debug("Start  #{self.class}#__create_thumbnail ... thumbnail_setting = #{thumbnail_setting.to_s}") if @logger && @logger.debug?

		begin
			if thumbnail_setting && (thumbnail_setting['scale'] || thumbnail_setting['width'] || thumbnail_setting['height']) then
				image = Magick::Image.from_blob(content).shift
				if ____create_thumbnail?(image, thumbnail_setting) then
					return ____create_thumbnail(image, thumbnail_setting)
				end
			end

			@logger.debug("no create thumbnail ... setting = #{thumbnail_setting.to_s}") if @logger && @logger.debug?
			return nil
		ensure
			@logger.debug("Finish #{self.class}#__create_thumbnail") if @logger && @logger.debug?
		end
	end

	def ____create_thumbnail?(image, thumbnail_setting)
		if thumbnail_setting['condition_width'] || thumbnail_setting['condition_height'] then
			if thumbnail_setting['condition_width'] && thumbnail_setting['condition_height'] then
				return image.columns > thumbnail_setting['condition_width'].to_i && image.rows > thumbnail_setting['condition_height'].to_i
			elsif thumbnail_setting['condition_width'] then
				return image.columns > thumbnail_setting['condition_width'].to_i
			elsif thumbnail_setting['condition_height'] then
				return image.rows > thumbnail_setting['condition_height'].to_i
			end
		end

		return true
	end

	def ____create_thumbnail(image, thumbnail_setting)
		if thumbnail_setting['scale'] then
			@logger.debug("create thumbnail ... scale = #{thumbnail_setting['scale']}") if @logger && @logger.debug?
			return image.scale(0.01 * thumbnail_setting['scale'].to_i)
		elsif thumbnail_setting['width'] || thumbnail_setting['height'] then
			if thumbnail_setting['width'] && thumbnail_setting['height'] then
				@logger.debug("create thumbnail ... width = #{thumbnail_setting['width']}, height = #{thumbnail_setting['height']}") if @logger && @logger.debug?
				return image.scale(thumbnail_setting['width'].to_i, thumbnail_setting['height'].to_i)
			elsif thumbnail_setting['width'] then
				width = thumbnail_setting['width'].to_i
				height = thumbnail_setting['keep'] ? image.rows * width / image.columns : image.rows
				@logger.debug("create thumbnail ... width = #{width}, height = #{height}") if @logger && @logger.debug?
				return image.scale(width, height)
			elsif thumbnail_setting['height'] then
				height = thumbnail_setting['height'].to_i
				width = thumbnail_setting['keep'] ? image.columns * height / image.rows : image.columns
				@logger.debug("create thumbnail ... width = #{width}, height = #{height}") if @logger && @logger.debug?
				return image.scale(width, height)
			end
		end

		return nil
	end

end
