/*
 * Copyright 2006 Maskat Project.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maskat.framework;

import java.util.Iterator;
import java.util.List;

import org.maskat.ide.property.descriptor.DeleteChildCommand;

/**
 * An instance of this class represents definition node.<br>
 * <code>IBasicDef</code>s form a tree.<br>
 * The children should have orders, so have indices.<br>
 * The children are not required to be of the same type.
 * 
 * @author shengshen
 * 
 */
public interface IBasicDef {

	/**
	 * BasicDefs should have its parent reference. <br>
	 * For example: Child could see its parent so that it could delete itself
	 * from its parent.<br>
	 * Note: the returned object of getParentDef may not hold this "child"
	 * object as a child!!! For example, the child is removed from the parent,
	 * but the parentDef property is not set to null.
	 * 
	 * @see DeleteChildCommand
	 * 
	 */
	public IBasicDef getParentDef();

	// protected void setParentDef(IBasicDef parentDef);

	/**
	 * Get the <b>unremovable</b> iterator of all the children of the specified
	 * type(class). Children should be returned by the Iterator in the proper
	 * order.
	 * 
	 * @param clazz
	 * @return
	 */
	public Iterator getTypedChildren(Class clazz);

	/**
	 * Get the <b>unremovable</b> iterator of all the children objects.<br>
	 * Children should be returned by the Iterator in the proper order.
	 * 
	 * @return
	 */
	public Iterator getAllChildren();

	/**
	 * Get a unmodifiable list containing the children in the proper order.
	 * Generally the client could choose between getAllChildren and
	 * getUnmodifiableChildren for the appropriate one.
	 * 
	 * @return
	 */
	public List getUnmodifiableChildren();

	/**
	 * Add a child to this object, the newly-added child will have the last
	 * order.
	 * 
	 * @param comp
	 */
	public void addChild(Object child);

	/**
	 * Get the index of the specified child.
	 * 
	 * @param child
	 * @return
	 */
	public int getChildIdx(Object child);

	/**
	 * should only be used by <code>DelBasicDefCommand</code> or
	 * <code>DelGridHeaderCommand</code>, to restore the original index of
	 * child element
	 * 
	 * @param child
	 * @param idx
	 */
	public void addChildToIdx(Object child, int idx);

	/**
	 * Add one DescTag child with the content of <code>desc</code>.
	 * 
	 * @param desc
	 */
	public void addDesc(String desc);

	/**
	 * Remove the specified child.
	 * 
	 * @param obj
	 */
	public void removeChild(Object obj);

	/**
	 * In the ordered childrens of the specified type, returns the one that have
	 * index idx, or null if not found.
	 * 
	 * @param clazz
	 * @param idx
	 * @return returns null if cannot find a element with type clazz and have
	 *         index idx
	 */
	public Object getChildByTypeIdx(Class clazz, int idx);

	/**
	 * Remove the child which have the index of <code>idx</code> in the list
	 * of all the children of type <code>clazz</code>
	 * 
	 * @param clazz
	 * @param idx
	 * @throws IllegalArgumentException
	 * @return
	 */
	public boolean removeChildByTypeIdx(Class clazz, int idx);

	/**
	 * Remove all the children definition nodes.
	 */
	public void removeAllChildren();

	public void accept(IBasicDefVisitor visitor);
}